/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/model/updatedocumentmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/tk/utils/deferredutils'
], function (DrawingFrame, Config, DOM, Utils, DeferredUtils) {

    'use strict';

    // mix-in class UpdateDocumentMixin ======================================

    /**
     * A mix-in class for the 'updateDocumentFormatting' function.
     *
     * @constructor
     */
    function UpdateDocumentMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        // public methods -----------------------------------------------------

        /**
         * Load performance: Execute post process activities after loading the document from the
         * local storage.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formatting has been
         *  updated successfully, or rejected when an error has occurred.
         */
        this.updateDocumentFormattingStorage = function () {

            var // the page node
                editdiv = self.getNode(),
                // the page styles object
                pageStyles = self.getPageStyles(),
                // the page attributes object
                pageAttributes = pageStyles.getElementAttributes(editdiv),
                // the page layout object
                pageLayout = self.getPageLayout();

            // initialize default page formatting
            pageStyles.updateElementFormatting(editdiv);

            // handling page breaks correctly
            self.setPageAttributes(pageAttributes);
            self.setPageMaxHeight(Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1));
            self.setPagePaddingLeft(Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 1));
            self.setPagePaddingTop(Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 1));
            self.setPagePaddingBottom(Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 1));
            self.setPageWidth(Utils.convertHmmToLength(pageAttributes.page.width, 'px', 1));

            // receiving list of all change track authors of the document
            app.addAuthors(self.getDocumentAttribute('changeTrackAuthors'));

            // all canvas elements need to be rendered
            if (editdiv.find(DrawingFrame.CANVAS_NODE_SELECTOR).length > 0) {
                self.trigger('drawingHeight:update', self.getNode().find(DrawingFrame.CANVAS_NODE_SELECTOR));
            }

            if (pageLayout.hasContentHeaderFooterPlaceHolder()) {
                pageLayout.updateStartHeaderFooterStyle();
                pageLayout.headerFooterCollectionUpdate();
            }

            pageLayout.callInitialPageBreaks({ triggerEvent: true }); // necessary always because of fields update. see #42606

            return $.when();
        };

        /**
         * Updates the formatting of all elements, after the import operations
         * of the document have been applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formatting has been
         *  updated successfully, or rejected when an error has occurred.
         */
        this.updateDocumentFormatting = function () {

            var // the result deferred object
                def = DeferredUtils.createDeferred(app, 'UpdateDocumentMixin: updateDocumentFormatting'),
                // the page node
                editdiv = self.getNode(),
                // the content node of the page
                pageContentNode = DOM.getPageContentNode(editdiv),
                // the comment layer node of the page
                commentLayerNode = DOM.getCommentLayerNode(editdiv),
                // all top-level paragraphs, all tables (also embedded), and all drawings
                formattingNodes = pageContentNode.find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // all paragraphs, tables and drawings inside comments
                commentContentNodes = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR).children(DOM.COMMENTNODE_SELECTOR).find(DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // number of progress ticks for tables (1 tick per paragraph/drawing)
                TABLE_PROGRESS_TICKS = 20,
                // total number of progress ticks
                totalProgressTicks = formattingNodes.length + (TABLE_PROGRESS_TICKS - 1) * formattingNodes.filter(DOM.TABLE_NODE_SELECTOR).length,
                // current progress for formatting process
                currentProgressTicks = 0,
                // update progress bar at most 250 times
                progressBlockSize = totalProgressTicks / 250,
                // index of progress block from last progress update
                lastProgressBlock = -1,
                // the page layout object
                pageLayout = self.getPageLayout(),
                // marker for getting nodes for displaying immediately on document loading
                splitPoint,
                // header and footer container nodes needs to be updated also, if there is content inside them
                headerFooterFormattingNodes = pageLayout.getHeaderFooterPlaceHolder().find(DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // fast display first page -> initialPageBreaks is called 2x, abort first promise when second is triggered
                initialPbPromise,
                //
                splitListNodes,
                //
                indexSplitPoint,
                // the page styles object
                pageStyles = self.getPageStyles(),
                // the page attributes object
                pageAttributes = pageStyles.getElementAttributes(editdiv);

            self.setPageAttributes(pageAttributes);
            self.setPageMaxHeight(Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1));
            self.setPagePaddingLeft(Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 1));
            self.setPagePaddingTop(Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 1));
            self.setPagePaddingBottom(Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 1));
            self.setPageWidth(Utils.convertHmmToLength(pageAttributes.page.width, 'px', 1));

            // #27818 - On loading document with table, table style is missing for proper formatting
            if (app.isODF()) { self.insertMissingTableStyles(); }

            // add content of comment nodes
            formattingNodes = formattingNodes.add(commentContentNodes);

            // first 90% for formatting (last 10% for updating lists)
            function updateFormatProgress(progressTicks) {
                var progressBlock = Math.floor(progressTicks / progressBlockSize);
                if (lastProgressBlock < progressBlock) {
                    def.notify(0.9 * progressTicks / totalProgressTicks);
                    lastProgressBlock = progressBlock;
                }
            }

            // updates the formatting of the passed element, returns a promise
            function updateElementFormatting(element) {

                var // the Promise for asynchronous element formatting
                    promise = null,
                    // whether the element is a direct child of the page content node
                    topLevel = !element.parentNode;

                // convert element to jQuery object
                element = $(element);

                // insert the detached element into the page content node
                // (performance optimization for large documents, see below)
                if (topLevel) { pageContentNode.append(element); }

                // determine the type of the passed element
                if (DOM.isParagraphNode(element)) {
                    // validate DOM contents AND update formatting of the paragraph
                    self.validateParagraphNode(element);
                    self.getParagraphStyles().updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);

                } else if (DrawingFrame.isDrawingFrame(element)) {

                    // TODO: validate DOM of paragraphs embedded in text frames
                    self.getDrawingStyles().updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);
                    if (Utils.SMALL_DEVICE && $(element).hasClass('float')) {
                        //verticalOffsetNode.remove();
                        $(element).parent().find('.float.offset').remove();

                        $(element).removeClass('float left right').addClass('inline').css('margin', '0 1mm');
                        $(element).css({ width: 'auto' });
                    }

                    // also updating all paragraphs inside text frames (not searching twice for the drawings)
                    self.implParagraphChangedSync($(element).find(DOM.PARAGRAPH_NODE_SELECTOR));

                    // trigger update of parent paragraph
                    self.trigger('paragraphUpdate:after', $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR));

                } else if (DOM.isTableNode(element) && !DOM.isExceededSizeTableNode(element)) {

                    // Bug 28409: Validate DOM contents of embedded paragraphs without
                    // formatting (done by the following table formatting). Finds and
                    // processes all cell paragraphs contained in a top-level table
                    // (also from embedded tables).
                    if (topLevel) {
                        element.find(DOM.CELLCONTENT_NODE_SELECTOR + ' > ' + DOM.PARAGRAPH_NODE_SELECTOR).each(function () {
                            self.validateParagraphNode(this);
                        });
                    }
                    // update table formatting asynchronous to prevent browser alerts
                    promise = self.getTableStyles().updateElementFormatting(element, { async: true })
                        .progress(function (localProgress) {
                            updateFormatProgress(currentProgressTicks + TABLE_PROGRESS_TICKS * localProgress);
                        })
                        .always(function () {
                            currentProgressTicks += TABLE_PROGRESS_TICKS;
                        });
                }

                return promise;
            }

            // updates the formatting of all lists, forwards the progress, returns a promise
            function updateListFormatting() {
                return self.updateLists({ async: true }).progress(function (progress) {
                    def.notify(0.9 + 0.1 * progress);  // last 10% for updating lists
                });
            }

            // dumps profiling information to browser console
            function dumpProfilingInformation() {
                var tableCount,
                    cellCount,
                    paragraphCount,
                    spanCount,
                    drawingCount;

                function dumpUpdateFormattingCount(styleSheets, elementCount, elementName) {
                    var updateCount = styleSheets.DBG_COUNT || 0,
                        perElementCount = (elementCount === 0) ? 0 : Utils.round(updateCount / elementCount, 0.01);
                    if ((elementCount > 0) || (updateCount > 0)) {
                        Utils.info('Editor.updateDocumentFormatting(): ' + elementCount + ' ' + elementName + ' updated ' + updateCount + ' times (' + perElementCount + ' per element)');
                    }
                }

                function dumpParagraphCount(methodName, debugCount) {
                    var updateCount = debugCount || 0,
                        perElementCount = (paragraphCount === 0) ? 0 : Utils.round(updateCount / paragraphCount, 0.01);
                    Utils.info('Editor.' + methodName + '(): called ' + updateCount + ' times (' + perElementCount + ' per paragraph)');
                }

                if (!Config.DEBUG) { return $.noop; }

                tableCount = editdiv.find(DOM.TABLE_NODE_SELECTOR).length;
                cellCount = editdiv.find('td').length;
                paragraphCount = editdiv.find(DOM.PARAGRAPH_NODE_SELECTOR).length;
                spanCount = editdiv.find('span').filter(function () { return DOM.isPortionSpan(this) || DOM.isTextComponentNode(this.parentNode); }).length;
                drawingCount = editdiv.find(DrawingFrame.NODE_SELECTOR).length;

                dumpUpdateFormattingCount(self.getCharacterStyles(), spanCount, 'text spans');
                dumpUpdateFormattingCount(self.getParagraphStyles(), paragraphCount, 'paragraphs');
                dumpUpdateFormattingCount(self.getTableCellStyles(), cellCount, 'table cells');
                dumpUpdateFormattingCount(self.getTableStyles(), tableCount, 'tables');
                dumpUpdateFormattingCount(self.getDrawingStyles(), drawingCount, 'drawing objects');
                dumpParagraphCount('validateParagraphNode', self.getValidateParagraphNode().DBG_COUNT);
            }

            // resetting an existing selection, after all operations are applied
            self.getSelection().resetSelection();
            self.setLastOperationEnd(null);

            // receiving list of all change track authors of the document
            app.addAuthors(self.getDocumentAttribute('changeTrackAuthors'));

            //US 74030826 - fetch first part of elements for fast displaying of first page
            splitPoint = pageContentNode.children('.splitpoint');
            if (splitPoint.length > 0) {
                indexSplitPoint = formattingNodes.index(splitPoint);
                splitListNodes = formattingNodes.slice(0, indexSplitPoint);
                formattingNodes = formattingNodes.slice(indexSplitPoint);
            }

            // detach all child nodes of the page content node, this improves
            // performance in very large documents notably
            formattingNodes.filter(function () { return this.parentNode === pageContentNode[0]; }).detach();

            // first, update the root page node
            pageStyles.updateElementFormatting(editdiv);

            // assign style to header footer placeholder node
            pageLayout.updateStartHeaderFooterStyle();

            if (splitPoint.length > 0) {
                splitListNodes = splitListNodes.add(headerFooterFormattingNodes);
                self.firstPartOfDocumentFormatting(splitListNodes);
                initialPbPromise = pageLayout.callInitialPageBreaks();
                app.leaveBusyDuringImport();
            } else {
                // add header and footer to formatting nodes
                formattingNodes = formattingNodes.add(headerFooterFormattingNodes);
            }

            // abort first promise before calling function second time (avoid duplicate calls if first is not finished)
            if (initialPbPromise) { initialPbPromise.abort(); }

            this.iterateArraySliced(formattingNodes, updateElementFormatting, { delay: 'immediate', infoString: 'Text: updateElementFormatting', app: app })
                // .then(updateDrawingGroups)
                .then(updateListFormatting)
                .then(function () { return pageLayout.callInitialPageBreaks({ triggerEvent: true }); })
                .then(function () { return app.getModel().startParagraphMarginColorAndBorderFormating(); }) // must be after page-breaks
                .done(function () { dumpProfilingInformation(); def.resolve(); })
                .fail(function () { def.reject(); });

            return def.promise();
        };

        /**
         * Synchronous formatting of first part of document (1-2 pages) for fast display.
         *
         * @param {jQuery} [splitListNodes]
         *  If exists, collected nodes for synchronous formatting.
         */
        this.firstPartOfDocumentFormatting = function (splitListNodes) {

            var // the nodes to be formatted
                formattingNodes,
                // the page node
                editdiv = self.getNode();

            if (splitListNodes) {
                formattingNodes = splitListNodes;
            } else {
                // first, update the root page node
                self.getPageStyles().updateElementFormatting(editdiv);
                // assign style to header footer placeholder node
                self.getPageLayout().updateStartHeaderFooterStyle();

                formattingNodes = DOM.getPageContentNode(editdiv).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR);
            }

            _.each(formattingNodes, function (element) {

                var // whether the element is a direct child of the page content node
                    topLevel = !element.parentNode;

                element = $(element);
                // determine the type of the passed element
                if (DOM.isParagraphNode(element)) {
                    // validate DOM contents AND update formatting of the paragraph
                    self.validateParagraphNode(element);
                    self.getParagraphStyles().updateElementFormatting(element);
                } else if (DrawingFrame.isDrawingFrame(element)) {
                    // TODO: validate DOM of paragraphs embedded in text frames
                    self.getDrawingStyles().updateElementFormatting(element);
                } else if (DOM.isTableNode(element) && !DOM.isExceededSizeTableNode(element)) {
                    // Bug 28409: Validate DOM contents of embedded paragraphs without
                    // formatting (done by the following table formatting). Finds and
                    // processes all cell paragraphs contained in a top-level table
                    // (also from embedded tables).
                    if (topLevel) {
                        element.find(DOM.CELLCONTENT_NODE_SELECTOR + ' > ' + DOM.PARAGRAPH_NODE_SELECTOR).each(function () {
                            self.validateParagraphNode(this);
                        });
                    }
                    // update table formatting asynchronous to prevent browser alerts
                    self.getTableStyles().updateElementFormatting(element);
                }
            });
        };

        /**
         * Handler for loading empty document with fast load. It applies html string, then actions,
         * formats document synchronosly, calculates page breaks, and at the end, leaves busy mode.
         *
         * @param {String} markup
         *  The HTML mark-up to be shown as initial document contents.
         *
         * @param {Array} actions
         *  The operation actions to be applied to finalize the fast import.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the actions have been applied.
         */
        this.fastEmptyLoadHandler = function (markup, actions) {
            // markup is in form of object, parse and get data from mainDocument
            markup = JSON.parse(markup);
            self.setFullModelNode(markup.mainDocument);

            return self.applyActions(actions, { external: true, useStorageData: false }).done(function () {
                self.firstPartOfDocumentFormatting();
                self.getPageLayout().callInitialPageBreaks({ triggerEvent: true });
            });
        };

        /**
         * Set the default paper format according to UI language if it is a new document
         */
        this.setDefaultPaperFormat = function () {
            var attrs = self.getStyleCollection('page').getElementAttributes(self.getNode());

            if (attrs.page) {
                var page = _.copy(attrs.page, true),
                    width = null,
                    height = null;

                var loaclePaperSize = Utils.getPaperSizeByLocale(Config.LOCALE);

                if (page.width !== loaclePaperSize.width || page.height !== loaclePaperSize.Height) {
                    width = loaclePaperSize.width;
                    height = loaclePaperSize.height;
                }

                if (width !== null) {
                    app.stopOperationDistribution(function () {
                        self.setPaperFormat({ width: width, height: height, marginLeft: page.marginLeft, marginRight: page.marginRight }, page);
                        self.getUndoManager().clearUndoActions();
                    });
                }
            }

        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class UpdateDocumentMixin

    // exports ================================================================

    return UpdateDocumentMixin;

});
