/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/format/paragraphstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/render/canvas'
], function (Utils, AttributeUtils, Color, Border, LineHeight, StyleCollection, DOM, Canvas) {

    'use strict';

    var // definitions for paragraph attributes
        DEFINITIONS = {

            nextStyleId: {
                def: '',
                scope: 'style'
            },

            alignment: {
                def: 'left',
                format: function (element, value) {
                    element.css('text-align', value);
                },
                preview: true // 'true' calls the own 'format' method
            },

            fillColor: {
                def: Color.AUTO, // auto for paragraph fill resolves to 'transparent'
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            /**
             * Line height relative to font settings. The CSS attribute
             * 'line-height' must be set separately at every descendant text
             * span because a relative line height (e.g. 200%) would not be
             * derived from the paragraph relatively to the spans, but
             * absolutely according to the paragraph's font size. Example: The
             * paragraph has a font size of 12pt and a line-height of 200%,
             * resulting in 24pt. This value will be derived absolutely to a
             * span with a font size of 6pt, resulting in a relative line
             * height of 24pt/6pt = 400% instead of the expected 200%.
             */
            lineHeight: { def: LineHeight.SINGLE },

            listLevel: { def: -1 },

            listStyleId: { def: '' },

            listLabelHidden: { def: false },

            listStartValue: { def: -1 },

            outlineLevel: { def: 9 },

            tabStops: {
                def: [],
                merge: function (tabStops1, tabStops2) {
                    // Merge tabStops2 into array tabstop1
                    // Support to clear tab stops defined in tabStops1
                    var clearedTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value === 'clear';
                        }),
                        additionalTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value !== 'clear';
                        }),
                        newTabstops = tabStops1 ? (additionalTabstops ? tabStops1.concat(additionalTabstops) : tabStops1) : (additionalTabstops ? additionalTabstops : []);

                    // Make entries in newTabstops unique
                    newTabstops = _.unique(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });

                    // Filter out cleared tabStops
                    if (clearedTabstops.length > 0) {
                        newTabstops = _.filter(newTabstops, function (tabstop) {
                            // return only tab stops which are not part of the clearedTabstops array
                            return (_.find(clearedTabstops, function (cleared) {
                                return cleared.pos === tabstop.pos;
                            }) === undefined);
                        });
                    }

                    // Sort tabStops by position
                    return _.sortBy(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });
                }
            },

            borderLeft: { def: Border.NONE },

            borderRight: { def: Border.NONE },

            borderTop: { def: Border.NONE },

            borderBottom: { def: Border.NONE },

            borderInside: { def: Border.NONE },

            indentFirstLine: { def: 0 },

            indentLeft: { def: 0 },

            indentRight: { def: 0 },

            marginTop: { def: 0 },

            marginBottom: { def: 0 },

            contextualSpacing: { def: false },

            pageBreakBefore: {
                def: false,
                format: function (element, value) {
                    if (value && !DOM.isNodeInsideTextFrame(element)) {
                        if (element.closest('tr').length > 0) {
                            element.parents('table').last().addClass('manual-page-break');
                        }
                        element.addClass('manual-page-break');
                    }
                }
            },

            pageBreakAfter: {
                def: false,
                format: function (element, value) {
                    if (value && !DOM.isNodeInsideTextFrame(element)) {
                        if (element.closest('tr').length > 0) {
                            element.parents('table').last().addClass('manual-pb-after');
                        }
                        element.addClass('manual-pb-after');
                    }
                }
            }
        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            cell: function (paragraph) { return DOM.isCellContentNode(paragraph.parent()) ? paragraph.closest('td') : null; }
        },

        // maps fill character attribute values to fill characters
        TAB_FILL_CHARS = { dot: '.', hyphen: '-', underscore: '_' },

        NBSP = '\xa0',

        canvas = new Canvas();

    // private static functions ===============================================

    /**
     *  Checking if a paragraph has a outer border.
     *
     * @param {Object} paraAttr
     *  The paragraphs attributes.
     *
     * @returns {Boolean}
     *  Whether the paragraph has a border.
     */
    function hasBorder(paraAttr) {

        // note: borderInside must not be checked here
        return (Border.isVisibleBorder(paraAttr.borderTop) || Border.isVisibleBorder(paraAttr.borderBottom) || Border.isVisibleBorder(paraAttr.borderLeft) || Border.isVisibleBorder(paraAttr.borderRight));
    }

    /**
     *  Checking if two paragraphs have a merged border.
     *
     * @param {Object} attributes1
     *  The paragraphs attributes.
     *
     *  @param {Object} attributes2
     *  The paragraphs attributes.
     *
     * @returns {Boolean}
     *  Whether the two paragraph have a merged border.
     */
    function isMergeBorders(attributes1, attributes2) {

        // note: borderInside must not be compared here
        return  Utils.hasEqualProperties(attributes1, attributes2,
                ['borderLeft', 'borderRight', 'borderTop', 'borderBottom', 'listStyleId', 'listLevel', 'indentLeft', 'indentRight', 'indentFirstLine']);
    }

    /**
     * Utility check if attributes have properties pageBreakBefore or pageBreakAfter
     * set to true.
     *
     * @param {Object} attributes
     *
     * @returns {Boolean}
     *
     */
    function isSetPageBreakAttribute(attributes) {
        return attributes.pageBreakBefore === true || attributes.pageBreakAfter === true;
    }

    /**
     * Fills the passed text span with a sufficient number of the specified
     * fill character.
     *
     * @param {jQuery} spanNode
     *  The span node to be filled. The current CSS formatting of this node is
     *  used to calculate the number of fill characters needed.
     *
     * @param {String} fillChar
     *  The fill character.
     *
     * @param {Number} width
     *  The target width of the tabulator, in 1/100 of millimeters.
     */
    function insertTabFillChar(spanNode, fillChar, width) {

        var // multiplier for a better average calculation
            multiplier = 15,
            // multiplier fill characters, used to calculate average character width
            checkString = Utils.repeatString(fillChar, multiplier),
            // average character width, in 1/100 mm
            charWidth = Utils.convertLengthToHmm(spanNode.contents().remove().end().text(checkString).width(), 'px') / multiplier,
            // number of characters needed to fill the specified width
            charCount = Math.floor(width / charWidth),
            // the fill character, repeated by the calculated number
            fillString = (charCount > 0) ? Utils.repeatString(fillChar, charCount) : NBSP,
            // a shortened string, if element is too wide
            shortFillString = null;

        // insert the fill string into the element
        spanNode.contents().remove().end().text(fillString);
        if (!fillString) { DOM.ensureExistingTextNode(spanNode); }

        // shorten fill string by one character, if element is too wide (e.g. due to rounding errors)
        if ((fillString.length > 1) && (Utils.convertLengthToHmm(spanNode.width(), 'px') >= width)) {
            shortFillString = fillString.slice(0, -1);
            spanNode.contents().remove().end().text(shortFillString);
            if (!shortFillString) { DOM.ensureExistingTextNode(spanNode); }
        }

        // workaround for iOS & Safari mobile - needs a text node directly within
        // the tab div to enable an external keyboard to jump through a tab chains
        // with empty spans
        if (_.device('ios') && _.device('safari')) {
            if (!spanNode[0].nextSibling) {
                spanNode.after(document.createTextNode(NBSP));
            }
        }
    }

    // class ParagraphStyles ==================================================

    /**
     * Contains the style sheets for paragraph formatting attributes. The CSS
     * formatting will be read from and written to paragraph <p> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function ParagraphStyles(docModel) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'paragraph', {
            families: 'character changes',
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateParagraphFormatting,
            previewHandler: setParagraphPreviewFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @attention
         *  Currently, only left&right aligned tabs are supported!
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateTabStops(paragraph, mergedAttributes) {

            var // default tab stop width from document settings
                defaultTabStop = docModel.getDocumentAttribute('defaultTabStop'),
                // paragraph tab stop definitions
                paraTabStops = mergedAttributes.paragraph.tabStops,
                // effective left margin of the paragraph
                marginLeft = Utils.convertCssLengthToHmm(paragraph.css('margin-left'), 'px'),
                // end position of previous tabulator
                prevTabEndPos = null,
                // first node of paragraph
                firstNode = null,
                // top position of first node
                topFirst = 0,
                // height of first node
                heightFirst = 0,
                // leftmost position used a base to calculate tab raster
                zeroLeft = 0,
                // first line  indent
                indentFirstLine = mergedAttributes.paragraph.indentFirstLine,
                // intersect function
                intersect = function (xs1, xe1, xs2, xe2) {
                    return ((xs1 <= xs2) && (xe1 >= xs2)) || ((xs2 <= xs1) && (xe2 >= xs1));
                },
                // active paragraph tab stops
                activeParaTabStops = null,
                // if paraTabStops has left and right oriented tabs, mixedTabs is set by the right-tab. if it is set it overwrites behaviro of text-aligment and para-margin
                mixedTab = getMixedTab(paraTabStops);

            if (docModel.getApp().isInQuit()) {
                return;
            }

            // zoom factor in floating point notation
            var  zoomFactor = docModel.getApp().getView().getZoomFactor() / 100;

            if (paragraph.length) {
                // Calculate top position and height of first paragraph child. Our paragraph
                // have always at least one child element.
                firstNode = $(paragraph.get(0).firstChild);
                topFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.position().top, 'px') : 0;
                heightFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.height(), 'px') : 0;

                //reset all tab sizes, for a much cleaner new calculating
                //(for example changing text-alignment from justify to left)
                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {
                    $(this).css('width', 0);
                });

                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {

                    var // the current tab node of the paragraph
                        tabNode = $(this),
                        // the span within the tab node containing the fill characters
                        tabSpan = $(this.firstChild),
                        // the position of the tab node
                        pos = tabNode.position(),
                        // the left bound position of the tab node in 1/100th mm
                        leftHMM = 0,
                        // the top bound position of the tab node in 1/100th mm
                        topHMM = Utils.convertLengthToHmm(pos.top, 'px'),
                        // the top bound position of the first child node of the paragraph in 1/100th mm
                        topSpanHMM = 0,
                        // the height of the first child node of the paragraph in 1/100th mm
                        heightSpanHMM = 0,
                        // the calculated width of the tab stop
                        width = 0,
                        // the fill character of the tab stop
                        fillChar = null,
                        // the tab stop values
                        tabStop,
                        // ignore the tab position and continue with the next one
                        ignore = false,
                        // tab stop type
                        tabStopType = 'left',
                        // indent first line tab stop position
                        indentFirstLineTabStopValue = 0,
                        // first line tab stop
                        indentFirstLineTabStop = null,
                        // insert index for virtual first line indent tab stop
                        indentFirstLineTabStopIndex = 0;

                    var calcLeftTab = function () {
                        return Math.max(0, tabStop.pos - (leftHMM % tabStop.pos));
                    };
                    var calcRightTab = function () {
                        return calcRightAlignTabstop(tabStop, tabNode, leftHMM);
                    };
                    var calcMiddleTab = function () {
                        return 0.5 * (calcLeftTab() + calcRightTab());
                    };

                    if (mergedAttributes.paragraph.alignment === 'center') {
                        // for a centered paragraph tab stops are calculated where the first character is treated as leftmost (0) position
                        zeroLeft = firstNode.length ? -(marginLeft + Utils.convertLengthToHmm(firstNode.position().left, 'px')) : -marginLeft;
                    } else {
                        zeroLeft = marginLeft;
                    }

                    // calculate left bound position of the tab node in 1/100th mm, including zoom factor for browsers using transf.scale
                    leftHMM = zeroLeft + Utils.convertLengthToHmm(pos.left / zoomFactor, 'px');

                    if (prevTabEndPos) {
                        // Check to see if we are in a chain of tabs. Force to use the previous tab end position
                        // as start position. Some browsers provide imprecise positions therefore we use the correct
                        // previous position from the last calculation.

                        // Check previous node and that text is empty - this is a precondition for the chaining
                        // code.
                        if (tabNode.prev().length && tabNode.prev().text().length === 0) {
                            var checkPrevPrevNode = tabNode.prev().length ? tabNode.prev().prev() : null;
                            if (checkPrevPrevNode.length && DOM.isTabNode(checkPrevPrevNode) && prevTabEndPos.top === topHMM) {
                                leftHMM = prevTabEndPos.center || prevTabEndPos.right;
                            }
                        }
                    }

                    // Special case for first line indent. For the first line the negative indent
                    // must be used as a tab stop position using the absolute value. We need to set
                    // at least one character for the tabSpan to have a valid width otherwise we
                    // won't get a correct height which is needed for the check.
                    if (indentFirstLine < 0) {
                        indentFirstLineTabStopValue = Math.abs(indentFirstLine);
                        if (marginLeft > 0) {
                            // In case of a margin left, we need to add the value to the indentFirstLineTabStopValue! Bug 45030
                            indentFirstLineTabStopValue += marginLeft;
                        }
                        // Fix for 29265 and 30847: Removing empty text node in span with '.contents().remove().end()'.
                        // Otherwise there are two text nodes in span after '.text('abc')' in IE.
                        tabSpan.contents().remove().end().text(NBSP);
                        heightSpanHMM = Utils.convertLengthToHmm(tabSpan.height(), 'px');
                        topSpanHMM = Utils.convertLengthToHmm(tabSpan.position().top, 'px');
                        // checkout if the first line indent is active for this special tab stop
                        if (intersect(topFirst, topFirst + heightFirst, topSpanHMM, topSpanHMM + heightSpanHMM) &&
                            (leftHMM + 10) < indentFirstLineTabStopValue) {
                            width = Math.max(0, indentFirstLineTabStopValue - leftHMM);
                        }

                        if (width > 1) {
                            // find index for the first line indent position within the paragraph tab stop array
                            indentFirstLineTabStopIndex = Utils.findLastIndex(paraTabStops, function (tab) {
                                return (indentFirstLineTabStopValue > tab.pos);
                            }) + 1;
                            // create a copy of the paragraph tab stops and add the first indent tab stop at
                            // the correct position to the active paragraph tab stop array
                            indentFirstLineTabStop = { value: 'left', pos: indentFirstLineTabStopValue, fillChar: NBSP, processed: false };
                            activeParaTabStops = paraTabStops.slice();
                            activeParaTabStops.splice(indentFirstLineTabStopIndex, 0, indentFirstLineTabStop);
                            // reset width - the tab stop position must be calculated using all tab stop positions
                            width = 0;
                        } else {
                            // reset text within span
                            tabSpan.contents().remove();
                            tabSpan[0].appendChild(document.createTextNode(''));
                            // use only paragraph tab stops
                            activeParaTabStops = paraTabStops;
                        }
                    } else {
                        // reset text within span
                        tabSpan.contents().remove();
                        tabSpan[0].appendChild(document.createTextNode(''));
                        // use only paragraph tab stops
                        activeParaTabStops = paraTabStops;
                    }

                    // Paragraph tab stops. Only paragraph tab stop can have a fill character and
                    // define a new alignment
                    if (width <= 1) {
                        tabStop = _(activeParaTabStops).find(function (tab) {
                            return ((leftHMM + 10) < tab.pos) && !tab.processed;
                        });
                        if (tabStop) {

                            tabStopType = tabStop.value || 'left';

                            // calculate tab stop size based on tab stop properties
                            if (tabStopType === 'left') {
                                // left bound tab stop
                                width = calcLeftTab();
                            } else if (tabStopType === 'right') {
                                // right bound tab stop
                                width = calcRightTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);

                                if (!ignore && _.indexOf(activeParaTabStops, tabStop) === activeParaTabStops.length - 1) {
                                    if (!_.browser.IE) {
                                        paragraph.css('width', 'calc(100% + 5mm)');
                                    }
                                }

                            } else if (tabStopType === 'center') {
                                // right bound tab stop combinded with left bound tab stop
                                width = calcMiddleTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);
                            }

                            // insert fill character
                            if (width > 1) {
                                fillChar = _.isString(tabStop.fillChar) ? TAB_FILL_CHARS[tabStop.fillChar] : NBSP;
                                if (fillChar) {
                                    insertTabFillChar(tabSpan, fillChar, width);
                                }
                                // Set processed flag to prevent using the same tab again due to rounding errors
                                // resulting by the browser calculation.
                                tabStop.processed = true;
                            }
                        }
                    }

                    if (!ignore) {
                        // only process default tab stop if tab stop is not set to ignore
                        if (width <= 1) {
                            // tab size calculation based on default tab stop
                            width = Math.max(0, defaultTabStop - (leftHMM % defaultTabStop));
                            width = (width <= 10) ? defaultTabStop : width; // no 0 tab size allowed, check for <= 10 to prevent rounding errors
                            // reset possible fill character
                            insertTabFillChar(tabSpan, NBSP, width);
                        }
                        prevTabEndPos = { right: leftHMM + width, top: topHMM };
                    }
                    // always set a width to the tab div (even an ignored tab div needs to set the size to zero)
                    tabNode.css('width', (width / 100) + 'mm');
                });

                calculateMarginForMixedTab(docModel, paragraph, mergedAttributes, mixedTab);

                // reset processed flag for tab stops again
                paraTabStops.forEach(function (tab) {
                    tab.processed = false;
                });
            }
        }

        /**
         * right tab can lay right above from the border, this cannot be triggered by MS-WORD interface,
         * but it is a typical behavior for MS-"table of content"
         * so we calculate if right tab is outside of the margin, and if true,
         * we reduce the margin-right of the paragraph
         */
        function calculateMarginForMixedTab(docModel, paragraph, paraAttributes, mixedTab) {
            paragraph.css('margin-right', null);

            if (!mixedTab) {
                return;
            }

            var pageAttrs = docModel.getDefaultAttributes('page');
            var maxParaWidth = pageAttrs.width - pageAttrs.marginLeft - pageAttrs.marginRight;
            var indentRight = paraAttributes.paragraph.indentRight;
            var currentParaWidth = mixedTab.pos + indentRight;

            if (currentParaWidth > maxParaWidth) {
                var newIndentRight = indentRight - (currentParaWidth - maxParaWidth);
                paragraph.css('margin-right', (newIndentRight / 100) + 'mm');
            }
        }

        function getMixedTab(paraTabStops) {
            var leftTab = _.find(paraTabStops, function (tab) {
                return !tab.value || tab.value === 'left';
            });
            var rightTab = _.find(paraTabStops, function (tab) {
                return tab.value === 'center' || tab.value === 'right';
            });
            if (leftTab && rightTab) {
                return rightTab;
            }
        }

        /**
         * Calculates the width of a right aligned tab element
         * depending on the position and the remaining size of
         * the following nodes.
         *
         * @param {Object} tabStop
         *  The right aligned tab stop.
         *
         * @param {jQuery} tabNode
         *  The right aligned tab node as jQuery.
         *
         * @param {Number} tabNodePosHMM
         *  The current position of the tab node in 1/100th mm.
         *
         * @returns {Number}
         *  The width of the right aligned tab stop. May be zero if the
         *  tab stop is not active.
         */
        function calcRightAlignTabstop(tabStop, tabNode, tabNodePosHMM) {
            var nextNodes = tabNode.nextAll(),
                node,
                rightPos = tabNodePosHMM,
                width = 0, i = 0, len = nextNodes.length;

            // Loop through all following nodes and sum up their width. Exceeding
            // the tab stop position or a new tab stops the iteration.
            for (i = 0; i < len && rightPos < tabStop.pos; i++) {
                node = $(nextNodes.get(i));
                if (DOM.isTabNode(node)) {
                    break;
                }
                rightPos += Utils.convertLengthToHmm(node.width(), 'px');
            }

            if (rightPos < tabStop.pos) {
                // Active right aligned tab stop. Subtract at least 1/10mm
                // rounding error regarding position values from the browser.
                width = tabStop.pos - rightPos - 10;
                width = Math.max(0, width);
            }

            return width;
        }

        /**
         * Will be called for every paragraph whose character attributes have
         * been changed.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateParagraphFormatting(paragraph, mergedAttributes) {

            var // the paragraph attributes of the passed attribute map
                paragraphAttributes = mergedAttributes.paragraph,
                // the character styles/formatter
                characterStyles = docModel.getStyleCollection('character'),

                leftMargin = 0,
                rightMargin = 0,

                leftMarginTmp = 0,
                rightMarginTmp = 0,

                prevParagraph = paragraph.prev(),
                prevAttributes = (prevParagraph.length > 0) ? self.getElementAttributes(prevParagraph) : { paragraph: {} },

                nextParagraph = paragraph.next(),
                nextAttributes = (nextParagraph.length > 0) ? self.getElementAttributes(nextParagraph) : { paragraph: {} },
                collectedAttrs =  {},
                prevCollectedAttrs =  {},
                textIndent = 0,
                explicitParaAttributes = null,
                leftPadding;

            /**
             * Returning a padding related to a given 'border' and 'addSpace' value.
             * The result is returned in 'hmm' or in 'px'.
             *
             * @param {Object} border
             *  A border attribute object.
             *
             * @param {Number} addSpace
             *  A border attribute object.
             *
             * @param {Boolean} getValueInPx
             *  Wether the value should be returned in hmm or px.
             *
             * @returns {Number|String}
             *  When 'getValueInPx' is false, it returns a Number in hmm,
             *  when true it returns a String with the value in px.
             */
            function getBorderPadding(border, addSpace, getValueInPx) {
                var space = addSpace + ((border && border.space) ? border.space : 0);
                return getValueInPx ? Utils.convertHmmToCssLength(space, 'px', 1) : space;
            }

            /**
             * Check if the given paragraph has a color.
             *
             * @param {Object} paraAttr
             *  The paragraphs attributes.
             *
             * @returns {Boolean}
             *  Returns true if paragraph has a color.
             */
            function hasColor(paraAttr) {
                if (paraAttr.fillColor && paraAttr.fillColor.type !== 'auto') {
                    return true;
                } else {
                    return false;
                }
            }

            /**
             * Formatting the border and paragraph color for the current paragraph.
             * This function is just used wrap the related code in this function.
             *
             */
            function formatParagraphMarginAndBorder() {

                var // flags to indicate if it's allowed to draw a border for the current
                    // paragraph, the value is based on the state from the prev/next paragraph
                    // (e.g. current paragraph is merged with the next paragraph),
                    // important: this is not related to the actual border attributes, but only to the context
                    drawBorderTop = false,
                    drawBorderBottom = false,
                    drawBorderMarginUp = false,
                    drawBorderMarginDown = false,
                    drawInnerBorderBottom = false,

                    // flags to indicate if a the paragraph color should be drawn
                    // in the margin
                    drawColorMarginUp = false,
                    drawColorMarginDown = false,

                    // objects containing all needed informations for the setting borders
                    borderBottomObj = {},
                    borderTopObj = {},
                    borderRightObj = {},
                    borderLeftObj = {},

                    // values set for css padding
                    paddingTopPx,
                    paddingRightPx,
                    paddingBottomPx,
                    paddingLeftPx,

                    // values set for the css border-image
                    resultingCssBorderImageWidth,

                    // resulting margin height
                    resultingMarginHeight;

                /**
                 * Creates a canvas rectangle for the border-image. It has a 20px border
                 * at each side (matching the border attributes color) and a fill color inside.
                 *
                 * @param {Object} borderTopObj
                 *  Object containing needed information to set the top border.
                 *
                 * @param {Object} borderBottomObj
                 *  Object containing needed information to set the bottom border.
                 *
                 * @param {Object} borderLeftObj
                 *  Object containing needed information to set the left border.
                 *
                 * @param {Object} borderRightObj
                 *  Object containing needed information to set the right border.
                 *
                 * @returns {Boolean}
                 *  Returns the canvas as a base64 encoded image.
                 */
                function createBase64Canvas(borderTopObj, borderBottomObj, borderLeftObj, borderRightObj) {

                    // initialize the canvas with the passed bitmap size (implicitly clears the canvas)
                    canvas.initialize({ width: 60, height: 60 });

                    canvas.render(function (context) {

                    // Important for IE11: The canvas border painted 20px wide to prevent browser zoom/css scale errors
                    // in IE11 when scaling. When the border width is higher than the 'sliced border' (see 'borderImageSlice'),
                    // it stretches the 'sliced border', the results is a blurred border with 'stretch' or small blurred lines
                    // in the 'fill' color with 'repeat'. With 20px there is enough headroom to prevent this.

                        //left
                        context.setLineStyle({ style: borderLeftObj.color, width: 40 });
                        context.drawLine(0, 0, 0, 60);

                        // right
                        context.setLineStyle({ style: borderRightObj.color, width: 40 });
                        context.drawLine(60, 0, 60, 60);

                        //bottom
                        context.setLineStyle({ style: borderBottomObj.color, width: 40 });
                        context.drawLine(0, 60, 60, 60);

                        // top
                        context.setLineStyle({ style: borderTopObj.color, width: 40 });
                        context.drawLine(0, 0, 60, 0);

                        // fill color
                        context.setFillStyle(self.getCssColor(paragraphAttributes.fillColor, 'fill'));
                        context.drawRect(20, 20, 20, 20, 'fill');

                    });

                    return canvas.getDataURL();
                }

                /**
                * Calculates the resulting, visible margin top and down for the current paragraph. It's needed
                * for drawing borders and paragraph color.
                *
                * Note: When a paragraph has a top margin and the previous has a bottom margin,
                * both margins collapse.
                *
                * @param {Boolean} drawMarginUp
                *  Whether the margin should be drawn upwards
                *  (value is based on the state from the prev/next paragraph ).
                *
                * @param {Boolean} drawMarginDown
                *  Whether the margin should be drawn downwards
                *  (value is based on the state from the prev/next paragraph ).
                *
                * @param {Number} prevMarginBottom
                *  Related to the current paragraph, the previous paragraphs margin bottom.
                *
                * @param {Number} marginBottom
                *  The margin bottom from a paragraph.
                *
                * @param {Number} marginTop
                *  The margin bottom from a paragraph.
                *
                * @returns {Object}
                *  Returns an object with the resulting margin up and downwards.
                */
                function calcResultingMarginHeight(drawMarginUp, drawMarginDown, prevMarginBottom, marginTop, marginBottom) {

                    // convert hmm to px to prevent unpredictable rounding by the browser later
                    var prevMarginBottomPx = Utils.convertHmmToLength(prevMarginBottom, 'px', 1);
                    var marginTopPx = Utils.convertHmmToLength(marginTop, 'px', 1);
                    var marginBottomPx = Utils.convertHmmToLength(marginBottom, 'px', 1);

                    // calculates the actual values for the resulting, visible margin up and down
                    var marginSpaceUp = drawMarginUp ? ((prevMarginBottomPx > marginTopPx) ? 0 : (Math.abs(prevMarginBottomPx - marginTopPx))) : 0;
                    var marginSpaceDown = drawMarginDown ? marginBottomPx : 0;

                    return { up: marginSpaceUp, down: marginSpaceDown };
                }

                /**
                * Calculates values for a single given border (e.g. left, right, top, or bottom).
                *  The resulting border is based on:
                *  1) paragraph attributes from the border
                *  2) the flag for this border, if it's allowed to draw it
                *
                * @param {Boolean} drawBorder
                *  If the the border is allowed to be drawn or not.
                *  (value is based on the state from the prev/next paragraph, not on the current paragraphg attributes).
                *
                * @param {Object} borderAttr
                *  The attributes for a single border (e.g. left, right, top, or bottom).
                *
                *  @returns {Object}
                *  Returns an object with various values needed to draw the border later.
                */
                function resultingBorder(drawBorder, borderAttr) {

                    var // border attributes
                        borderAttributes,
                        // the css borderStyle for the border, ready to use for setting the border style via css
                        cssBorderStyle = 'none',
                        // the css BorderImageWidth for the border, a value of 1 makes the border visible, 0 not visible
                        cssBorderImageWidth = 0,
                        // the color from the border
                        borderColor = self.getCssColor(borderAttr.color, 'line'),
                        // if the border is visible based on its border attributes
                        borderIsVisible = false;

                    // when the border should be visible and is visible
                    if (drawBorder && Border.isVisibleBorder(borderAttr)) {

                        borderAttributes = self.getCssBorderAttributes(borderAttr);
                        cssBorderStyle = borderAttributes.width + 'px ' + borderAttributes.style + ' ' + borderAttributes.color;
                        cssBorderImageWidth = (borderAttributes.width > 0) ? 1 : 0;
                        borderIsVisible = true;
                    }

                    return { cssBorderStyle: cssBorderStyle, cssBorderImageWidth: cssBorderImageWidth, color: borderColor, borderIsVisible: borderIsVisible };
                }

                // CHECK: which borders needs to be drawn upwards
                // -> (if the paragraph has borders and merged borders with the previous paragraph)
                if ((hasBorder(paragraphAttributes) && isMergeBorders(paragraphAttributes, prevAttributes.paragraph)) && !isSetPageBreakAttribute(paragraphAttributes)) {
                    drawBorderTop = false;
                    drawBorderMarginUp = true;
                } else {
                    drawBorderTop = true;
                    drawBorderMarginUp = false;
                }

                // CHECK: which borders needs to be drawn downwards
                // -> (if the paragraph has borders and merged borders with the next paragraph)
                if ((hasBorder(paragraphAttributes) && isMergeBorders(paragraphAttributes, nextAttributes.paragraph)) && !isSetPageBreakAttribute(nextAttributes.paragraph)) {
                    drawBorderBottom = false;
                    drawBorderMarginDown = true;
                    if (Border.isVisibleBorder(paragraphAttributes.borderInside)) {
                        drawInnerBorderBottom = true;
                    }
                } else {
                    drawBorderBottom = true;
                    drawBorderMarginDown = false;
                }

                // CHECK: if the paragraph margin needs to be colored downwards and if a inner border must be drawn
                if (((!hasBorder(nextAttributes.paragraph) && !hasBorder(paragraphAttributes))) && isMergeBorders(paragraphAttributes, nextAttributes.paragraph) && hasColor(paragraphAttributes) && hasColor(nextAttributes.paragraph)) {
                    drawColorMarginDown = true;
                    if (Border.isVisibleBorder(paragraphAttributes.borderInside)) {
                        drawInnerBorderBottom = true;
                        drawBorderBottom = false;
                    }
                }

                // CHECK: if the paragraph margin needs to be colored upwards
                if (((!hasBorder(prevAttributes.paragraph) && !hasBorder(paragraphAttributes))) && isMergeBorders(paragraphAttributes, prevAttributes.paragraph) && hasColor(paragraphAttributes) && hasColor(prevAttributes.paragraph)) {
                    drawColorMarginUp = true;
                }

                // create resulting border attributes based on merged border flags and border attributes
                // when a flag due to merged borders is false, it overrides the border based on attributes
                borderBottomObj = resultingBorder(drawBorderBottom, paragraphAttributes.borderBottom);
                borderTopObj = resultingBorder(drawBorderTop, paragraphAttributes.borderTop);
                borderRightObj = resultingBorder(true, paragraphAttributes.borderRight);
                borderLeftObj = resultingBorder(true, paragraphAttributes.borderLeft);

                // padding
                paddingTopPx = getBorderPadding(paragraphAttributes.borderTop, 0, true);
                paddingRightPx = getBorderPadding(paragraphAttributes.borderRight, 0, true);
                paddingBottomPx = getBorderPadding(paragraphAttributes.borderBottom, 0, true);
                paddingLeftPx = getBorderPadding(paragraphAttributes.borderLeft, 0, true);

                // special case for bottom/inner border
                if (drawInnerBorderBottom && !drawBorderBottom) {
                    borderBottomObj = resultingBorder(drawInnerBorderBottom, paragraphAttributes.borderInside);
                    paddingBottomPx = getBorderPadding(paragraphAttributes.borderInside, 0, true);
                }

                // creating the resulting css values for the border-image
                resultingCssBorderImageWidth = borderTopObj.cssBorderImageWidth + ' ' + borderRightObj.cssBorderImageWidth + ' ' + borderBottomObj.cssBorderImageWidth + ' ' + borderLeftObj.cssBorderImageWidth + ' ';

                // setting css
                _.extend(collectedAttrs, {
                    // in some browsers (e.g. Chrome) a border style must be set to display border-images
                    borderTop: borderTopObj.cssBorderStyle,
                    borderRight: borderRightObj.cssBorderStyle,
                    borderLeft: borderLeftObj.cssBorderStyle,
                    borderBottom: borderBottomObj.cssBorderStyle,

                    // When a border should not be visible, the borderImageWidth must be 0 (border-style: 'none' is ignored in most browsers for border images).
                    // Important: Don't use px, just use numbers (numbers represents multiples of the corresponding border-width ). This scales much better with browser zoom.
                    borderImageWidth: resultingCssBorderImageWidth,

                    paddingTop: paddingTopPx,
                    paddingRight: paddingRightPx,
                    paddingBottom: paddingBottomPx,
                    paddingLeft: paddingLeftPx // note: maybe overwritten later again in list formatting in 'updateParagraphFormatting'

                });

                // add a class to the paragraph when it has a color or a border
                if (hasColor(paragraphAttributes) || hasBorder(paragraphAttributes)) {
                    paragraph.addClass('borderOrColor');
                } else {
                    paragraph.removeClass('borderOrColor');
                }

                // check if the paragraph needs to be formatted with margin color or border
                if (borderTopObj.borderIsVisible || borderBottomObj.borderIsVisible || borderRightObj.borderIsVisible || borderLeftObj.borderIsVisible || drawBorderMarginUp || drawColorMarginUp || drawColorMarginDown || drawBorderMarginDown) {

                    // get the resulting, visible margin for this paragraph
                    resultingMarginHeight = calcResultingMarginHeight(drawBorderMarginUp || drawColorMarginUp, drawBorderMarginDown || drawColorMarginDown, prevAttributes.paragraph.marginBottom, paragraphAttributes.marginTop, paragraphAttributes.marginBottom);

                    // setting css
                    _.extend(collectedAttrs, {
                        borderImageSource: 'url("' + createBase64Canvas(borderTopObj, borderBottomObj, borderLeftObj, borderRightObj) + '")',
                        // the used part of the canvas to draw the border,
                        // since the border in the source is 20px, it should be sliced with 20px, but the IE11 needs a special treated here.
                        borderImageSlice: (_.browser.IE === 11) ? '21 fill' : '20 fill',
                        // moving the border-image over the css-box-model, to 'paint' in the margin
                        borderImageOutset: resultingMarginHeight.up + 'px 0px ' + resultingMarginHeight.down + 'px 0px',
                        // Note: iOS and IE11: 'stretch must be use in these browsers to prevent display errors with 'repeat'.
                        // In other browsers 'repeat' can also be used (when there are display errors in a browser try 'repeat' here if it looks better)
                        borderImageRepeat: 'stretch',            //(Utils.IOS || _.browser.IE === 11) ? 'stretch' : 'repeat',
                        // Needed to fix browser painting problems with chrome/safari, they just need something in the margin top/bottom to repaint this area
                        boxShadow:  (_.browser.Chrome || _.browser.Safari) ? '0px ' + resultingMarginHeight.down + 'px transparent ,  0px ' + (resultingMarginHeight.up * -1) + 'px transparent' : ''

                    });

                // ... when border or paragraph color, set default values to override possible old css styles
                } else {

                    _.extend(collectedAttrs, {
                        // clear a possible border image
                        borderImage: '',
                        // clear the boxShadow
                        boxShadow: ''
                    });
                }
            }

            // Always update character formatting of all child nodes which may
            // depend on paragraph settings, e.g. automatic text color which
            // depends on the paragraph fill color. Also visit all helper nodes
            // containing text spans, e.g. numbering labels.
            Utils.iterateDescendantNodes(paragraph, function (node) {

                // visiting the span inside a hard break node
                // -> this is necessary for change tracking attributes
                if (DOM.isHardBreakNode(node)) {
                    characterStyles.updateElementFormatting(node.firstChild);
                }
                if (DOM.isComplexFieldNode(node)) {
                    characterStyles.updateElementFormatting(node);
                }

                DOM.iterateTextSpans(node, function (span) {
                    characterStyles.updateElementFormatting(span, { baseAttributes: mergedAttributes });
                });
            }, undefined, { children: true });

            // update border padding
            leftPadding = paragraphAttributes.borderLeft && paragraphAttributes.borderLeft.space ? paragraphAttributes.borderLeft.space : 0;
            leftMarginTmp = getBorderPadding(paragraphAttributes.borderLeft, 0, false) * -1;
            rightMarginTmp = getBorderPadding(paragraphAttributes.borderRight, 0, false) * -1;

            // Fix for BUG #36298:
            // only add the border-margins (left/right) if the paragraph isn't in a textframe
            // or if they were positive. Otherwise the left/right border will be hidden.
            if (!DOM.isNodeInsideTextFrame(paragraph) || (leftMarginTmp > 0 && rightMarginTmp > 0)) {
                leftMargin += leftMarginTmp;
                rightMargin += rightMarginTmp;
            }

            var topMargin = paragraphAttributes.marginTop;
            var bottomMargin = paragraphAttributes.marginBottom;

            // inside comments the paragraph bottom margin must be 0 (even if this was not applied via operation)
            // and the paragraph must be left aligned
            if (!docModel.getApp().isODF() && !docModel.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(paragraph)) {
                bottomMargin = 0;
                collectedAttrs.textAlign = 'left';
            }

            // calculate paragraph border and the paragraph color
            formatParagraphMarginAndBorder();

            //calculate list indents
            var listLabel = $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR);
            var listStyleId = paragraphAttributes.listStyleId;
            var listObject = null;
            if (listStyleId.length) {
                var listLevel = paragraphAttributes.listLevel,
                    lists = docModel.getListCollection();
                if (listLevel < 0) {
                    // is a numbering level assigned to the current paragraph style?
                    listLevel = lists.findIlvl(listStyleId, mergedAttributes.styleId);
                }
                if (listLevel !== -1 && listLevel < 10) {
                    var listItemCounter = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                    listObject = lists.formatNumber(paragraphAttributes.listStyleId, listLevel, listItemCounter, 1);

                    if (listObject.indent > 0) {
                        leftPadding += listObject.indent - listObject.firstLine;
                        leftMargin += listObject.firstLine;
                    }
                    if (listLabel.length) {
                        var listSpan = listLabel.children('span');
                        if (listObject.fontSize) {
                            listSpan.css('font-size', listObject.fontSize + 'pt');
                        }
                        if (listObject.color) {
                            listSpan.css('color', self.getCssTextColor(listObject.color, [paragraphAttributes.fillColor, listObject.fillColor]));
                        }
                    }
                } else {
                    //fix for Bug 37594 some list-helper dont disappear
                    listLabel.detach();
                }
            } else {
                //fix for Bug 37594 some list-helper dont disappear
                listLabel.detach();
            }

            // paragraph margin attributes - also applying to paragraphs in a list, if they are defined as explicit attribute
            // -> handle both cases correctly: 40792 and 41118
            if (listStyleId === '') {
                leftMargin += paragraphAttributes.indentLeft ? paragraphAttributes.indentLeft : 0;
                rightMargin += paragraphAttributes.indentRight ? paragraphAttributes.indentRight : 0;
                textIndent = paragraphAttributes.indentFirstLine ? paragraphAttributes.indentFirstLine : 0;
                collectedAttrs.textIndent = Utils.convertHmmToLength(textIndent, 'px', 1) + 'px';
            } else {
                // but explicit attributes need to be handled (40792)
                explicitParaAttributes = AttributeUtils.getExplicitAttributes(paragraph);
                explicitParaAttributes = (explicitParaAttributes && explicitParaAttributes.paragraph) || {};
                if (explicitParaAttributes.indentLeft) { leftMargin += explicitParaAttributes.indentLeft; }
                if (explicitParaAttributes.indentRight) { rightMargin += explicitParaAttributes.indentRight; }
                if (_.isNumber(explicitParaAttributes.indentFirstLine) && (explicitParaAttributes.indentFirstLine !== 0 || _.isNumber(explicitParaAttributes.indentLeft))) {
                    var lI = explicitParaAttributes.indentLeft ? explicitParaAttributes.indentLeft - 635 : 635;
                    // first line indent, negative values represent hanging indents
                    var lFl = explicitParaAttributes.indentFirstLine ? Math.abs(explicitParaAttributes.indentFirstLine) :  0;
                    // text indent is left indent minus special indent like first line or hanging
                    textIndent -= lI - lFl;
                    collectedAttrs.textIndent = Utils.convertHmmToLength(textIndent, 'px', 1) + 'px';
                }

                // explicit attribute for left indent needs to handle the indent of a list (47090)
                if (listObject && listObject.indent && explicitParaAttributes && explicitParaAttributes.indentLeft) {
                    leftMargin -= listObject.indent;
                }
            }

            if (textIndent < 0) {
                leftPadding -= textIndent;
                leftMargin += textIndent;
            }

            _.extend(collectedAttrs, {
                paddingLeft: Utils.convertHmmToLength(leftPadding, 'px', 1) + 'px',
                // now set left & right margin: On small devices it must be set to 0 (when smaller than 0),
                // otherwise it may be not visible on small devices because of the draft mode
                marginLeft: (Utils.SMALL_DEVICE && leftMargin < 0) ? '0px' : Utils.convertHmmToLength(leftMargin, 'px', 1) + 'px',
                marginRight: (Utils.SMALL_DEVICE && leftMargin < 0) ? '0px' : Utils.convertHmmToLength(rightMargin, 'px', 1) + 'px',
                textIndent: Utils.convertHmmToLength(textIndent, 'px', 1) + 'px'

            });

            // Overwrite of margin left for lists in draft mode (conversion from fixed unit mm to %)
            if (Utils.SMALL_DEVICE && paragraph.data('draftRatio')) {
                collectedAttrs.marginLeft = (parseInt($(paragraph).css('margin-left'), 10) * paragraph.data('draftRatio')) + '%';
            }

            //no distance between paragraph using the same style if contextualSpacing is set
            var noDistanceToPrev = prevAttributes.paragraph.contextualSpacing && (mergedAttributes.styleId === prevAttributes.styleId),
                noDistanceToNext = paragraphAttributes.contextualSpacing && (mergedAttributes.styleId === nextAttributes.styleId);
            if (noDistanceToPrev) {
                //remove bottom margin from previous paragraph
                prevCollectedAttrs.marginBottom = 0 + 'px';
                collectedAttrs.paddingTop = 0 + 'px';
                topMargin = 0;
            }
            if (noDistanceToNext) {
                collectedAttrs.paddingBottom = 0 + 'px';
                bottomMargin = 0;
            }
            _.extend(collectedAttrs, {

                marginTop: Utils.convertHmmToLength(topMargin, 'px', 1) + 'px',
                marginBottom: Utils.convertHmmToLength(bottomMargin, 'px', 1) + 'px'

            });

            // Important: The next and previous paragraphs are not available at document load.
            // As these both are mandatory for formatting paragraph border and color with margins,
            // add these paragraphs to a store for later formatting, when both are accessible.
            if (!self.isImportFinished()) {

                if (hasBorder(paragraphAttributes) || hasColor(paragraphAttributes)) {
                    docModel.addParagraphToStore(paragraph, mergedAttributes);
                }
            }

            // taking care of implicit paragraph nodes behind tables after loading the document
            // (also handling final empty paragraphs behind tables inside table cell (36537)).
            if (DOM.isIncreasableParagraph(paragraph) && (!self.isImportFinished() || !docModel.selectionInNode(paragraph))) {
                collectedAttrs.height = 0;
            }

            if (!_.browser.IE) {
                collectedAttrs.width = null;
            }

            // apply collected attributes at the end
            prevParagraph.css(prevCollectedAttrs);
            paragraph.css(collectedAttrs);

            // update the size of all tab stops in this paragraph (but only if the paragraph contains tabs (Performance))
            if (paragraph.find(DOM.TAB_NODE_SELECTOR).length > 0) {
                updateTabStops(paragraph, mergedAttributes);
            }

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(paragraph, mergedAttributes);
        }

        /**
         * Will be called for paragraphs used as preview elements in the GUI.
         *
         * @param {jQuery} paragraph
         *  The preview paragraph node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function setParagraphPreviewFormatting(paragraph, mergedAttributes) {

            var // the character styles/formatter
                characterStyles = docModel.getStyleCollection('character');

            // format the text spans contained in the paragraph element
            paragraph.children().each(function () {
                characterStyles.updateElementFormatting(this, { baseAttributes: mergedAttributes, preview: true });
            });
        }

        /**
         * Updates all paragraph nodes that have a color or a border.
         * This method is intended to update all needed paragraphs after page-breaks are applied.
         */
        function updateParagraphBorderAndColor() {

            // must not be called at document loading, since at loading page-breaks
            // are applied before formatting borders/paragraph color
            if (self.isImportFinished()) {

                // find all paragraph nodes that have a color or a border
                // note: find seems to be very fast in all browsers and even in very large documents
                var nodes = docModel.getNode().find('.borderOrColor');

                if (nodes.length > 0) {
                    docModel.implParagraphChanged(nodes);
                }
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated. If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {ParagraphStyles}
         *  A reference to this instance.
         */
        this.updateTabStops = function (paragraph) {
            updateTabStops($(paragraph), this.getElementAttributes(paragraph));
            return this;
        };

        /**
         * Making the function hasBorder public.
         * (take a look at the private function for more details).
         */
        this.hasBorder = function (paraAttr) {
            return hasBorder(paraAttr);
        };

        /**
         * Updates all passed paragraph nodes in case they have:
         * 1) a paragraph color
         * 2) or a border
         *
         * Otherwise nothing happens. So no unnecessary updates are triggered
         * for paragraphs that don't fall into the mentioned category.
         *
         * When this function is called while undoRedo is running, the nodes are added to a queue
         * and are updated after 'undoRedo' is finished. This is needed, because at undoRedo the
         * node may have old attributes, which would cause a visually wrong formatting.
         *
         * This method is intended to update paragraphs with color or a border in special cases,
         * such as paragraph merge, splitt, delete or insert.
         *
         * @param {jQuery|Array} nodes
         *  A jQuery node or an array with jQuery nodes.
         */
        this.updateParagraphBorderAndColorNodes = function (nodes) {

            _.each(nodes, function (node) {
                // make sure its a jQuery node
                var $node = $(node);

                if ($node.hasClass('borderOrColor') && $node && $node.length > 0 && DOM.isParagraphNode($node)) {
                    // add to queue while undoRedo is running, else update the paragraph normal
                    if (docModel.isUndoRedoRunning()) {
                        docModel.addParagraphToStore($node);
                    } else {
                        docModel.implParagraphChanged($node);
                    }
                }
            });
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('paragraph', DEFINITIONS);

        // is triggered after page-break calculation is finished
        self.listenTo(docModel, 'update:borderAndColor', updateParagraphBorderAndColor);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class ParagraphStyles

    // constants --------------------------------------------------------------

    /**
     * Default solid border line for paragraphs and tables.
     *
     * @constant
     */
    ParagraphStyles.SINGLE_BORDER = { style: 'single', width: 17, space: 140, color: Color.AUTO };

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: ParagraphStyles });

});
