/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/renderutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/container/sortedarray',
    'io.ox/office/tk/render/path',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, Logger, SimpleMap, SortedArray, Path, Color, Border, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var IntervalArray = SheetUtils.IntervalArray;

    // pixel resolution for canvas rendering (double resolution for retina displays)
    var CANVAS_RESOLUTION = Utils.RETINA ? 2 : 1;

    // width of a grid line for rendering in the scaled canvas coordinate system
    var GRID_LINE_WIDTH = 1 / CANVAS_RESOLUTION;

    // border line styles, mapped to rendering priority (less numbers are higher priority)
    var BORDER_STYLE_PRIOS = { solid: 1, dashed: 2, dotted: 3, dashDot: 4, dashDotDot: 5 };

    // class BorderDescriptor =================================================

    /**
     * Descriptor for the rendering styles of a single cell border line.
     *
     * @constructor
     *
     * @property {String} style
     *  Effective dash style of the border line: one of 'solid', 'dotted',
     *  'dashed', 'dashDot', or 'dashDotDot'.
     *
     * @property {Number} width
     *  The effective width of the border, in pixels (positive integer). In
     *  case of double lines, this value will not be less than 3, and specifies
     *  the total width of the entire border. In case of hair lines, this value
     *  will be 1.
     *
     * @property {Boolean} hair
     *  Whether the border consists of a hair line (original width is less than
     *  one pixel).
     *
     * @property {Boolean} double
     *  Whether the border consists of two parallel lines.
     *
     * @property {Number} lineWidth
     *  The effective width of a single line in the border, in pixels (positive
     *  integer). In case of double lines, this value will be less than the
     *  total border width (property 'width').
     *
     * @property {Array<Number>|Null} pattern
     *  The canvas dash pattern for the border lines, or null for solid lines.
     *
     * @property {Boolean} solid
     *  Whether the border lines will be rendered as solid lines (true), or
     *  with a dashed line style (false).
     *
     * @property {String} color
     *  The effective CSS line color.
     *
     * @property {Number} y
     *  The luma value (weighted lightness) of the border line color, as
     *  positive floating-point number. The maximum value 1 represents the
     *  color white.
     */
    function BorderDescriptor(docModel, gridColor, border) {

        // initial line width, in pixels (rounded according to canvas pixel resolution, zero is hair line),
        // restricted to minimum cell size to prevent that left/right or top/bottom border lines of a cell overlay each other
        this.width = Math.min(Utils.convertHmmToLength(border.width, 'px', GRID_LINE_WIDTH), SheetUtils.MIN_CELL_SIZE);

        // flag for double lines
        this.double = border.style === 'double';

        // flag for hair lines (initial line width is zero)
        this.hair = !this.double && (this.width === 0);

        // effective line style (solid, dashed, etc.)
        this.style = this.double ? 'solid' : border.style;

        // effective minimum width in pixels (at least one pixel, or 3 pixels for double lines: two lines, one pixel gap)
        this.width = Math.max(this.width, (this.double ? 3 : 1) * GRID_LINE_WIDTH);

        // width of a single line (important for double-lined borders)
        this.lineWidth = this.double ? Math.max(GRID_LINE_WIDTH, Utils.round(this.width / 3, GRID_LINE_WIDTH)) : this.width;

        // effective dash pattern according to style and effective line width
        this.pattern = Border.getBorderPattern(this.style, this.lineWidth);

        // whether the border lines appear as solid lines
        this.solid = !this.pattern;

        // resolve all color details (use semi-transparency for hair lines, but not on retina displays)
        var color = Color.parseJSON(border.color);
        if (this.hair && (GRID_LINE_WIDTH === 1)) { color.transform('alphaMod', 40000); }
        var colorDesc = docModel.resolveColor(color, gridColor.isAuto() ? 'line' : gridColor);

        // effective CSS line color
        this.color = colorDesc.css;

        // luma of the line color
        this.y = colorDesc.y;

    } // class BorderDescriptor

    // static methods ---------------------------------------------------------

    /**
     * Creates a new border descriptor, if the passed border attribute value
     * describes a visible border line, otherwise returns null.
     *
     * @returns {BorderDescriptor|Null}
     *  A new border descriptor, if the passed border attribute value describes
     *  a visible border line, otherwise null.
     */
    BorderDescriptor.create = function (docModel, gridColor, border) {
        return Border.isVisibleBorder(border) ? new BorderDescriptor(docModel, gridColor, border) : null;
    };

    /**
     * Returns the border with stronger visual appearance.
     *
     * @param {BorderDescriptor|Null} border1
     *  The first border style descriptor, or null for invisible borders.
     *
     * @param {BorderDescriptor|Null} border2
     *  The second border style descriptor, or null for invisible borders.
     *
     * @returns {BorderDescriptor|Null}
     *  One of the passed border descriptors with stronger visual appearance.
     *  If both borders are missing (invisible), null will be returned. If one
     *  of the passed borders is missing (invisible), the other border will be
     *  returned. If both borders are visible, one of them will be selected by
     *  checking the following conditions (in the specified order):
     *  (1) Compare border width: Thicker border wins over thinner border.
     *  (2) Compare line style: Double over solid over dashed over dotted over
     *      dash-dot over dash-dot-dot.
     *  (3) Compare line color by luma: Darker color wins over lighter color.
     */
    BorderDescriptor.getStrongerBorder = function (border1, border2) {

        // handle missing borders (return the other border)
        if (!border1) { return border2; }
        if (!border2) { return border1; }

        // compare border width (hair lines first)
        if (border1.hair && !border2.hair) { return border2; }
        if (!border1.hair && border2.hair) { return border1; }

        // compare pixel border width
        if (border1.width > border2.width) { return border1; }
        if (border1.width < border2.width) { return border2; }

        // compare border style (double border wins first)
        if (border1.double && !border2.double) { return border1; }
        if (!border1.double && border2.double) { return border2; }

        // compare remaining border styles by priority
        if (border1.style !== border2.style) {
            var prio1 = BORDER_STYLE_PRIOS[border1.style] || 9999;
            var prio2 = BORDER_STYLE_PRIOS[border2.style] || 9999;
            return (prio1 < prio2) ? border1 : border2;
        }

        // compare luma value (darker color wins!)
        return (border1.y < border2.y) ? border1 : border2;
    };

    // class StyleDescriptor ==================================================

    /**
     * A descriptor containing various rendering information according to the
     * formatting attributes of a column, row, or cell in the spreadsheet
     * document, such as the effective CSS fill color of the cells.
     *
     * @constructor
     *
     * @property {Object} attributes
     *  The merged formatting attribute set represented by this descriptor.
     *
     * @property {Boolean} formatChanged
     *  Whether the number format of the underlying auto-style and the
     *  effective attribute set are not the same. This implies that the cell
     *  display string cached in the cell model cannot be used as it has been
     *  generated based on the number format of the auto-style only.
     *
     * @property {Boolean} explicitTextColor
     *  Whether the text color has been set by an explicit formatting attribute
     *  (and not via the style sheet referred by the auto-style).
     *
     * @property {ParsedFormat} format
     *  The parsed number format object, according to the format code in the
     *  passed attribute set.
     *
     * @property {String|Null} fill
     *  The effective CSS fill color, or null for transparent cells.
     *
     * @property {Object} borders
     *  The effective rendering styles of all borders surrounding a cell, as
     *  instances of the class BorderDescriptor, mapped by the internal short
     *  border identifier ('t', 'b', 'l', or 'r'). Either of these map elements
     *  may be missing or set to null which denotes a missing (invisible)
     *  border line.
     *
     * @property {Font} font
     *  The effective font settings, with scaled font size according to the
     *  current zoom factor of the sheet.
     *
     * @property {Number} rowHeight
     *  The height of a single text line (effective height of an empty cell),
     *  in pixels.
     *
     * @property {String} textColor
     *  The effective CSS text color.
     */
    function StyleDescriptor(attributeSet, formatChanged, explicitTextColor) {

        // the merged formatting attribute set
        this.attributes = attributeSet;

        // whether the number format has been changed in the attribute set
        this.formatChanged = formatChanged;

        // whether the text color has been set explicitly
        this.explicitTextColor = explicitTextColor;

        // the parsed format code
        this.format = null;

        // effective CSS fill color
        this.fill = null;

        // style settings for all border lines
        this.borders = null;

        // scaled font attributes according to current sheet zoom factor
        this.font = null;

        // height of a single text line (effective height of an empty cell)
        this.rowHeight = 0;

        // resolved CSS text color (without number format override color)
        this.textColor = null;

    } // class StyleDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Refreshes this style descriptor according to the attribute set, and the
     * passed sheet settings.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model, used e.g. to resolve theme colors.
     *
     * @param {Object} gridColor
     *  The grid color of the active sheet. Will be used for cell border lines
     *  that are set to automatic line color.
     *
     * @param {Number} sheetZoom
     *  The zoom factor of the active sheet, used to calculate the scaled font
     *  size for the cell contents.
     */
    StyleDescriptor.prototype.update = function (docModel, gridColor, sheetZoom) {

        // the merged character and cell attributes o fthis style descriptor
        var charAttrs = this.attributes.character;
        var cellAttrs = this.attributes.cell;

        // resolve the number format descriptor
        this.format = docModel.getNumberFormatter().getParsedFormatForAttributes(cellAttrs);

        // resolve fill style to color descriptor, and to resulting CSS color value
        var colorDesc = docModel.resolveColor(Color.parseJSON(cellAttrs.fillColor), 'fill');
        this.fill = (colorDesc.a === 0) ? null : colorDesc.css;

        // resolve style settings for all border lines
        this.borders = Utils.makeObject('tblrdu', function (borderKey) {
            var borderName = SheetUtils.getBorderName(borderKey);
            return RenderUtils.BorderDescriptor.create(docModel, gridColor, cellAttrs[borderName]);
        });

        // scaled font attributes according to current sheet zoom factor
        this.font = docModel.getFontCollection().getFont(charAttrs, { zoom: sheetZoom });

        // scaled height of a single text line, in pixels (effective height of an empty cell)
        this.rowHeight = docModel.getRowHeight(charAttrs, sheetZoom);

        // resolve cell text color
        this.textColor = docModel.getCssTextColor(charAttrs.color, [cellAttrs.fillColor]);
    };

    // class MergeDescriptor ==================================================

    /**
     * Descriptor for a merged range covering at least one cell element in the
     * rendering matrix.
     *
     * @constructor
     *
     * @property {Range} mergedRange
     *  The address of the original merged range, as contained in the document.
     *
     * @property {Range} shrunkenRange
     *  The address of the visible part of the merged range (without leading
     *  and trailing hidden columns and rows).
     *
     * @property {String} originKey
     *  The unique address key of the start cell of the original merged range.
     *
     * @property {MatrixCellElement} element
     *  The cell element describing the top-left visible cell of the merged
     *  range, containing the text descriptor for the merged range. This cell
     *  element may not be part of the regular rendering matrix, e.g. if the
     *  merged range starts outside the bounding ranges of the matrix.
     */
    function MergeDescriptor(mergedRange, shrunkenRange) {

        // the original merged range
        this.mergedRange = mergedRange;

        // the visible part of the merged range
        this.shrunkenRange = shrunkenRange;

        // cell key of the original merged range
        this.originKey = mergedRange.start.key();

        // matrix cell element of the top-left cell
        this.element = null;

    } // class MergeDescriptor

    // class WordDescriptor ===================================================

    /**
     * Represents all settings needed to render a single word with leading
     * white-space in a single line of text for a cell. Used especially for
     * horizontally justified text where the width of gaps between the words
     * depends on the total width available for the text line.
     *
     * @constructor
     *
     * @property {Number} space
     *  The width of all white-space characters preceding this word, in pixels,
     *  according to the font settings of the cell, and the sheet zoom factor.
     *
     * @property {String} text
     *  The word text.
     *
     * @property {Number} width
     *  The width of the word (without any white-space), in pixels, according
     *  to the font settings of the cell, and the sheet zoom factor.
     *
     * @property {Number} x
     *  The horizontal position of the word, in pixels, relative to the start
     *  position of the text line containing this word.
     */
    function WordDescriptor(space, text, width) {

        // the total width of leading white-space
        this.space = space;

        // the text contents of this word
        this.text = text;

        // the total width of the word (without white-space)
        this.width = width;

        // the horizontal offset of the word
        this.x = 0;

    } // class WordDescriptor

    // class LineDescriptor ===================================================

    /**
     * Represents all settings needed to render a single line of text in a
     * (multi-line) cell.
     *
     * @constructor
     *
     * @property {String} text
     *  The text to be rendered in this text line.
     *
     * @property {Number} width
     *  The total width of the text, in pixels, according to the font settings
     *  of the cell, and the sheet zoom factor.
     *
     * @property {Number} x
     *  The absolute horizontal position of the text line, in pixels, according
     *  to the position of the cell, the horizontal alignment, and the length
     *  of the text.
     *
     * @property {Number} y
     *  The absolute vertical position of the font base line for this text
     *  line, in pixels, according to the position of the cell, the font
     *  settings, the vertical alignment, and the number of text lines.
     *
     * @property {Array<WordDescriptor>|Null} words
     *  Detailed settings for the single words and the white-space characters
     *  contained in this text line, e.g. for horizontally justified alignment.
     *  If set to null, the entire text line as contained in the property
     *  'text' must be rendered at once.
     */
    function LineDescriptor(text, width) {

        // the text contents of this text line
        this.text = text;

        // the total width of the text line
        this.width = width;

        // the horizontal position of the text line
        this.x = 0;

        // the vertical position of the text line
        this.y = 0;

        // single words in the text line
        this.words = null;

    } // class LineDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Creates an array with descriptors for all words contained in this text
     * line, and assigns it to the property 'words' of this descriptor. The
     * single words will be distributed equally into the specified available
     * width for the text line.
     *
     * @param {Font} font
     *  The effective font settings. The font size must be scaled according to
     *  the current zoom factor of the sheet containing the text cell.
     *
     * @param {String} textDir
     *  The writing direction that specifies whether to calculate the positions
     *  of the words from left to right, or vice versa.
     *
     * @param {Number} availableWidth
     *  The maximum width available for the text line.
     */
    LineDescriptor.prototype.justifyWords = function (font, textDir, availableWidth) {

        // the text contained in this text line descriptor (will be shortened in the while-loop)
        var text = this.text;
        // total width of white-space characters
        var totalSpace = 0;
        // total width of the words without white-space
        var totalWidth = 0;

        // create the array property
        this.words = [];

        // extract all words from the text line
        while (text.length > 0) {

            // extract leading white-space characters, and a following word
            var matches = /^(\s*)(\S*)/.exec(text);
            // width of leading white-space characters
            var space = font.getTextWidth(matches[1]);
            // width of the word without white-space
            var width = font.getTextWidth(matches[2]);

            // push a new word descriptor to the array
            this.words.push(new WordDescriptor(space, matches[2], width));
            totalSpace += space;
            totalWidth += width;

            // remove the processed part from the text line, and proceed with next word
            text = text.substr(matches[0].length);
        }

        // nothing to do, if no white-space is available
        if (totalSpace === 0) {
            this.words = null;
            return;
        }

        // expansion factor for white-space between words
        var spaceFactor = (availableWidth - totalWidth) / totalSpace;
        // reverse positioning of the words in right-to-left mode
        var reverse = textDir === 'rtl';
        // relative offset of the current word
        var offset = reverse ? availableWidth : 0;

        // calculate relative start position of all words (reverse the words in
        // right-to-left writing mode - TODO: is this save in all cases?)
        if (reverse) {
            this.words.forEach(function (wordDesc) {
                offset -= wordDesc.space * spaceFactor + wordDesc.width;
                wordDesc.x = Math.round(offset);
            });
        } else {
            this.words.forEach(function (wordDesc) {
                offset += wordDesc.space * spaceFactor;
                wordDesc.x = Math.round(offset);
                offset += wordDesc.width;
            });
        }

        // set total line width (may be less than availableWidth, e.g. single word with leading space)
        this.width = reverse ? (availableWidth - offset) : offset;
    };

    // class TextDescriptor ===================================================

    /**
     * Represents all settings needed to render text contents of a single cell.
     *
     * @constructor
     *
     * @property {Object} inner
     *  The position of the inner cell rectangle, without trailing grid lines,
     *  expanded to the size of a merged range if existing.
     *
     * @property {Object} clip
     *  The position of the clipping rectangle needed to restrict the content
     *  text to the cell area (e.g. for oversized fonts), or to clip text cells
     *  with horizontally overflowing contents at the boundaries of adjacent
     *  cells.
     *
     * @property {Number} overflowL
     *  The width of the text parts floating into the outer space left of the
     *  inner cell area, in pixels. The number zero represents fitting or
     *  clipped text.
     *
     * @property {Number} overflowR
     *  The width of the text parts floating into the outer space right of the
     *  inner cell area, in pixels. The number zero represents fitting or
     *  clipped text.
     *
     * @property {Array<LineDescriptor>} lines
     *  An array with descriptors for all text lines to be rendered for this
     *  cell.
     *
     * @property {String} fill
     *  The canvas fill style needed to render the text contents with the
     *  color specified in the cell's formatting attributes.
     *
     * @property {Object} font
     *  The canvas font styles needed to render the text contents with the font
     *  settings specified in the cell's formatting attributes. See method
     *  Canvas.setFontStyle() for details.
     *
     * @property {String} direction
     *  The writing direction for correct rendering of BiDi text.
     *
     * @property {Object|Null} decoration
     *  The text decoration styles (underline, and strike-through); or null, if
     *  the text will not be decorated. An existing text decoration descriptor
     *  contains the following properties:
     *  - {Object} decoration.line
     *      The canvas line styles needed to render the text decoration. See
     *      method Canvas.setLineStyle() for details.
     *  - {Path} decoration.path
     *      The canvas rendering path containing the line segments for all text
     *      decorations in all text lines.
     */
    function TextDescriptor(innerRect) {

        // the inner cell rectangle
        this.inner = innerRect;

        // the clipping rectangle
        this.clip = { left: innerRect.left, top: innerRect.top, width: innerRect.width, height: innerRect.height };

        // text overflow width to the left
        this.overflowL = 0;

        // text overflow width to the right
        this.overflowR = 0;

        // settings for text lines
        this.lines = [];

        // the canvas fill style for the text
        this.fill = null;

        // the canvas font style for the text
        this.font = null;

        // additional text decoration settings
        this.decoration = null;

    } // class TextDescriptor

    // class MatrixElement ====================================================

    /**
     * Base class for various elements in the rendering matrix, used for cell
     * elements, and column/row header elements. Contains the original cell
     * formatting attributes for the matrix element, and resolved formatting
     * settings suitable for rendering.
     *
     * @constructor
     *
     * @property {StyleDescriptor} style
     *  A descriptor representing the auto style in the document used to format
     *  the cell area and text contents. If set to null, the matrix element is
     *  in undefined state (nothing will be rendered).
     *
     * @property {Object} [skipBorders]
     *  If specified, a map specifying which border attributes will be set to
     *  invisible regardless of the actual border formatting attribute (all map
     *  properties with a key set to the one-letter border identifier as used
     *  in the 'borders' property of this class, and the value set to true).
     */
    var MatrixElement = _.makeExtendable(function () {

        // the style descriptor (background color, border styles, font settings)
        this.style = null;

        // descriptor specifying which border lines will not be rendered for this element
        this.skipBorders = null;

    }); // class MatrixElement

    // public methods ---------------------------------------------------------

    /**
     * Updates the effective rendering styles of the background area and border
     * lines of this element, according to the passed cell formatting
     * attributes.
     *
     * @param {StyleDescriptor|Null} styleDesc
     *  The style descriptor for this element. If set to null, the element does
     *  not contain any active own formatting (it will not be rendered at all).
     *
     * @param {Object} [skipBorders]
     *  If specified, a map specifying which border attributes will be set to
     *  invisible regardless of the actual border formatting attribute (all map
     *  properties with a key set to the one-letter border identifier as used
     *  in the 'borders' property of this class, and the value set to true).
     */
    MatrixElement.prototype.update = function (styleDesc, skipBorders) {

        // store the new style descriptor
        this.style = styleDesc;

        // store flags for skipped borders
        this.skipBorders = skipBorders;
    };

    // class MatrixHeaderElement ==============================================

    /**
     * Represents a column/row header element in the rendering matrix.
     *
     * @constructor
     *
     * @extends MatrixElement
     *
     * @property {Boolean} columns
     *  Whether this element is a column header element (true), or a row header
     *  element (false).
     *
     * @property {MatrixHeaderElement|Null} p
     *  The previous column/row header element in the rendering matrix.
     *
     * @property {MatrixHeaderElement|Null} n
     *  The next column/row header element in the rendering matrix.
     *
     * @property {MatrixCellElement} f
     *  The first matrix cell element in the column/row represented by this
     *  header element.
     *
     * @property {Number} index
     *  The zero-based column/row index.
     *
     * @property {Number} offset
     *  The absolute position of the first pixel in the column/row, in pixels.
     *
     * @property {Number} size
     *  The size of the column/row, in pixels.
     *
     * @property {Boolean} leading
     *  Whether this header element represents the first visible column/row of
     *  a bounding interval in the rendering matrix. This may be an element in
     *  the middle of the rendering matrix, if the matrix covers multiple
     *  bounding intervals with gaps inbetween.
     *
     * @property {Boolean} trailing
     *  Whether this header element represents the last visible column/row of a
     *  bounding interval in the rendering matrix. This may be an element in
     *  the middle of the rendering matrix, if the matrix covers multiple
     *  bounding intervals with gaps inbetween.
     */
    var MatrixHeaderElement = MatrixElement.extend({ constructor: function (index, columns, prev, next) {

        // base constructor
        MatrixElement.call(this);

        // column/row index of this header element
        this.index = index;

        // orientation
        this.columns = columns;

        // references to adjacent header elements
        this.p = prev;
        this.n = next;

        // first matrix element
        this.f = null;

        // absolute pixel position of the column/row
        this.offset = 0;

        // pixel size of the column/row
        this.size = 0;

        // whether this element is the first in a bounding interval
        this.leading = false;

        // whether this element is the last in a bounding interval
        this.trailing = false;

        // update the references in the adjacent header elements
        if (prev) { prev.n = this; }
        if (next) { next.p = this; }

    } }); // class MatrixHeaderElement

    // public methods ---------------------------------------------------------

    /**
     * Updates the effective rendering styles of this header element, according
     * to the passed column/row descriptor.
     *
     * @param {AutoStyleCache} styleCache
     *  The style cache used to resolve the style descriptor for this element.
     *
     * @param {ColRowDescriptor} entryDesc
     *  The column/row descriptor, as received from the class ColRowCollection.
     */
    MatrixHeaderElement.prototype.update = function (styleCache, entryDesc) {

        // the style descriptor (no formatting for rows without 'customFormat' flag)
        var styleDesc = (this.columns || entryDesc.merged.customFormat) ? styleCache.getStyle(entryDesc.style) : null;

        // update background/border formatting
        MatrixElement.prototype.update.call(this, styleDesc);

        // update specific column/row settings
        this.offset = entryDesc.offset;
        this.size = entryDesc.size;
    };

    /**
     * Unlinks this instance from all adjacent matrix elements, and updates the
     * old adjacent matrix elements accordingly.
     */
    MatrixHeaderElement.prototype.unlink = function () {

        // unlink this element in the adjacent header elements
        if (this.p) { this.p.n = this.n; }
        if (this.n) { this.n.p = this.p; }

        // release the references to all matrix elements
        this.p = this.n = this.f = null;
    };

    // class MatrixCellElement ================================================

    /**
     * Represents a single cell element in the rendering matrix.
     *
     * @constructor
     *
     * @extends MatrixElement
     *
     * @property {Address} address
     *  The address of this cell element in the sheet.
     *
     * @property {String} key
     *  An arbitrary unique map key for the cell element.
     *
     * @property {MatrixHeaderElement} col
     *  The column header element of this cell in the rendering matrix.
     *
     * @property {MatrixHeaderElement} row
     *  The row header element of this cell in the rendering matrix.
     *
     * @property {MatrixCellElement|Null} l
     *  The next cell element to the left in the rendering matrix.
     *
     * @property {MatrixCellElement|Null} r
     *  The next cell element to the right in the rendering matrix.
     *
     * @property {MatrixCellElement|Null} t
     *  The next cell element above in the rendering matrix.
     *
     * @property {MatrixCellElement|Null} b
     *  The next cell element below in the rendering matrix.
     *
     * @property {CellModel|Null} model
     *  The model of the cell covered by this matrix element.
     *
     * @property {Boolean} blank
     *  Whether the cell represented by this matrix element is blank. Will only
     *  be true for undefined or blank cells, but not for non-blank cells
     *  without visible text representation, e.g. a formula cell resulting in
     *  an empty string.
     *
     * @property {Object} rect
     *  The effective inner rectangle of the cell, without the trailing border
     *  lines, expanded to merged ranges.
     *
     * @property {MergeDescriptor|Null} merged
     *  The descriptor for a merged range covering this cell element, if
     *  available; otherwise null.
     *
     * @property {TextDescriptor|Null} text
     *  A descriptor object for the text contents to be rendered for this cell,
     *  including single text lines for multi-line text, and all formatting
     *  settings.
     *
     * @property {Number} contentWidth
     *  The required width of the text contents (regardless of the actual size
     *  or inner padding of the cell represented by this matrix element). If
     *  set to zero, the content width for this cell is not available (e.g., if
     *  the cell is blank). Can be used to determine the optimal width of the
     *  column containing this cell.
     *
     * @property {Number} contentHeight
     *  The required height of the text contents (regardless of the actual size
     *  or inner padding of the cell represented by this matrix element). If
     *  set to zero, the content height is not available (e.g., if the cell is
     *  undefined). Can be used to determine the optimal height of the row
     *  containing this cell.
     *
     * @property {Object|Null} renderProps
     *  Additional rendering properties next to the cell formatting attibutes,
     *  e.g. data bar settings from a conditional formatting rule.
     */
    var MatrixCellElement = MatrixElement.extend({ constructor: function (address, colHeader, rowHeader, l, r, t, b) {

        // base constructor
        MatrixElement.call(this);

        // cell address, and unique map key for the element
        this.address = address;
        this.key = address.key();

        // references to the column/row header elements
        this.col = colHeader;
        this.row = rowHeader;

        // references to adjacent cell elements
        this.l = l;
        this.r = r;
        this.t = t;
        this.b = b;

        // the cell model, and the blank cell flag
        this.model = null;
        this.blank = true;

        // descriptor of the merged range containing this cell (instance of MergeDescriptor)
        this.merged = null;

        // text descriptor
        this.text = null;

        // other rendering properties
        this.renderProps = null;

        // update the references in the header elements
        if (colHeader && !t) { colHeader.f = this; }
        if (rowHeader && !l) { rowHeader.f = this; }

        // update the references in the adjacent cells
        if (l) { l.r = this; }
        if (r) { r.l = this; }
        if (t) { t.b = this; }
        if (b) { b.t = this; }

    } }); // class MatrixCellElement

    // public methods ---------------------------------------------------------

    /**
     * Updates the effective rendering styles of this matrix element, according
     * to the passed cell descriptor.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the cell represented by this instance.
     *
     * @param {CellModel|Null} cellModel
     *  The model of the cell to be rendered. If set to null, the cell is
     *  undefined (column/row default formatting will be used instead).
     *
     * @param {StyleDescriptor|Null} styleDesc
     *  The style descriptor for this element. If set to null, the element does
     *  not contain any active own formatting (it will not be rendered at all).
     *
     * @param {Object|Null} renderProps
     *  Additional rendering properties next to the formatting attibutes of the
     *  passed style descriptor, e.g. data bar settings from a conditional
     *  formatting rule.
     */
    MatrixCellElement.prototype.update = function (sheetModel, cellModel, styleDesc, renderProps) {

        // the document model, and its number formatter
        var docModel = sheetModel.getDocModel();
        var numberFormatter = docModel.getNumberFormatter();

        // whether the cell is covered (hidden) by a merged range
        var covered = this.isMergedCovered();
        // the visible part of the merged range (needed for cell border formatting)
        var shrunkenRange = this.merged ? this.merged.shrunkenRange : null;
        // do not paint inner borders in a merged range
        var skipBorders = shrunkenRange ? {
            l: shrunkenRange.start[0] < this.address[0],
            r: this.address[0] < shrunkenRange.end[0],
            t: shrunkenRange.start[1] < this.address[1],
            b: this.address[1] < shrunkenRange.end[1],
            d: covered,
            u: covered
        } : null;

        // update background/border formatting
        MatrixElement.prototype.update.call(this, styleDesc, skipBorders);

        // set the cell model, and the blank cell flag
        this.model = cellModel;
        this.blank = !cellModel || cellModel.isBlank();

        // ensure that the merge descriptor refers to the correct matrix element; it may change e.g. when the
        // origin of a large merged range (with a temporary matrix element) moves inside the bounding ranges
        if (this.isMergedOrigin()) { this.merged.element = this; }

        // the inner cell rectangle without grid line for rendering (merged ranges may be partly out of rendering matrix, resolve from sheet model)
        var innerRect = this.innerRect = this.merged ? sheetModel.getRangeRectangle(this.merged.mergedRange) :
            (this.col && this.row) ? { left: this.col.offset, top: this.row.offset, width: this.col.size, height: this.row.size } :
            sheetModel.getCellRectangle(this.address);

        // initialize other element properties to represent a blank cell initially
        this.text = null;
        this.contentWidth = this.contentHeight = 0;

        // store the rendering properties (also without display string, special number formats may leave the cell empty!)
        this.renderProps = renderProps;

        // Do not render cells covered by merged ranges, or undefined cells (missing cell model). Conditional
        // formatting (data bars or icon sets) may specify to hide the cell display text. Reformat the display
        // string, if the number format of the auto-style has been overridden (e.g. via conditional formatting).
        var display = null;
        if (covered || this.blank || (renderProps && (renderProps.showValue === false))) {
            display = '';
        } else if (styleDesc && styleDesc.formatChanged) {
            display = numberFormatter.formatValue(styleDesc.format, cellModel.v);
        } else {
            display = cellModel.d; // may be null (railroad track error)
        }

        // exclude grid lines at trailing cell borders
        innerRect.width -= GRID_LINE_WIDTH;
        innerRect.height -= GRID_LINE_WIDTH;

        // nothing more to do for cells without visible display text (blank cells, formulas resulting in empty string, etc.),
        // but do not exit for null values as display string (invalid number format) which will be rendered as 'railroad track error'
        if (!styleDesc || (display === '')) {
            // set content height (used e.g. for optimal row height)
            this.contentHeight = styleDesc ? styleDesc.rowHeight : 0;
            return;
        }

        // height of a single text line (effective height of an empty cell)
        var rowHeight = styleDesc.rowHeight;
        // the merged attribute set
        var attributes = styleDesc.attributes;
        // create a new text descriptor
        var textDesc = this.text = new TextDescriptor(innerRect);
        // effective text orientation (alignment and writing direction)
        var textOrient = SheetUtils.getTextOrientation(cellModel.v, attributes.cell.alignHor, display);
        // the inner horizontal padding for text in cells (according to current zoom)
        var paddingLeft = (renderProps && renderProps.iconSet) ? rowHeight : sheetModel.getEffectiveTextPadding();
        var paddingRight = sheetModel.getEffectiveTextPadding();
        // the available width for the cell text (without left/right padding)
        var availableWidth = Math.max(2, innerRect.width - paddingLeft - paddingRight);
        // resolved and scaled font settings
        var font = styleDesc.font;
        // the effective number format section, according to cell value (treat booleans like strings)
        var section = styleDesc.format.getSection(cellModel.isBoolean() ? '' : cellModel.v);
        // position of the font base line in the text lines (according to effective row height)
        var baseLineOffset = font.getBaseLineOffset(rowHeight);
        // resulting width of text contents (maximum width of text lines)
        var totalWidth = 0;
        // resulting height of text contents (total height of all text lines)
        var totalHeight = 0;
        // width of text overflow left of the cell
        var overflowL = 0;
        // width of text overflow right of the cell
        var overflowR = 0;

        // special processing for text cells: may span over multiple lines, or may overflow out of the cell area
        if (_.isString(display) && cellModel.isText()) {

            // automatic text wrapping: may result in multiple text lines, trimmed whitespace (also in single line text)
            if (SheetUtils.hasWrappingAttributes(attributes)) {

                // whether text will be justified horizontally
                var justify = textOrient.cssTextAlign === 'justify';

                // create settings for each text line
                display.split(/\n/).forEach(function (paragraph) {

                    // replace any control characters with space characters
                    paragraph = Utils.cleanString(paragraph);

                    // collect maximum width of entire paragraph texts before wrapping
                    totalWidth = Math.max(totalWidth, font.getTextWidth(paragraph));

                    // split paragraph to text lines
                    var textLines = font.getTextLines(paragraph, availableWidth);

                    // create settings for each text line in the paragraph
                    textLines.forEach(function (textLine, lineIndex) {

                        // remove leading/trailing whitespace (but not leading whitespace in first line)
                        textLine = textLine.replace((lineIndex === 0) ? /\s+$/ : /^\s+|\s+$/, '');

                        // width of the text
                        var width = font.getTextWidth(textLine);
                        // the new text line descriptor
                        var lineDesc = new LineDescriptor(textLine, width);

                        // push the new line descriptor into the array
                        textDesc.lines.push(lineDesc);

                        // split horizontally justified text into words (but not single characters, or the last line of the paragraph)
                        if (justify && (textLine.length > 1) && (lineIndex + 1 < textLines.length) && (width < availableWidth)) {
                            lineDesc.justifyWords(font, textOrient.textDir, availableWidth);
                        }
                    });
                }, this);

            // text cells without wrapping (text may overflow out of the cell)
            } else {

                // remove line breaks from text (but leading/trailing whitespace
                // remains significant, e.g. for the optimal column width)
                display = Utils.cleanString(display.replace(/\n/g, ''));

                // create a single text line
                totalWidth = font.getTextWidth(display);
                textDesc.lines.push(new LineDescriptor(display, totalWidth));

                // calculate width of the text that overflows from the cell (merged ranges always clip at cell boundaries)
                if (!this.merged && (availableWidth < totalWidth)) {

                    // store width of text overflowing from the cell area, according to alignment
                    switch (textOrient.baseBoxAlign) {
                        case 'left':
                            overflowR = Math.ceil(paddingLeft + totalWidth - innerRect.width);
                            break;
                        case 'center':
                            overflowL = Math.max(0, Math.ceil((totalWidth - innerRect.width - paddingLeft + paddingRight) / 2));
                            overflowR = Math.max(0, Math.ceil((totalWidth - innerRect.width + paddingLeft - paddingRight) / 2));
                            break;
                        case 'right':
                            overflowL = Math.ceil(totalWidth + paddingRight - innerRect.width);
                            break;
                    }
                }
            }

        // other cells: numbers, booleans, error codes
        } else {

            // first, save the width of original display text (used for optimal column width)
            var effectiveWidth = totalWidth = display ? font.getTextWidth(display) : 0;

            // number cells with standard number format: reformat display string dynamically according to available space
            // TODO: for other dynamic formats too!
            if (display && section && (section.category === 'standard') && cellModel.isNumber()) {

                // number formatter returns null, if no valid display string can be found for the available width (e.g. cell too narrow)
                var formatResult = numberFormatter.formatStandardNumberToWidth(cellModel.v, SheetUtils.MAX_LENGTH_STANDARD_CELL, font, availableWidth + 1);

                // store the new display string, and the resulting text width
                display = formatResult ? formatResult.text : null;
                effectiveWidth = formatResult ? formatResult.width : 0;
            }

            // show railroad track error when display string does not fit
            if ((display === null) || (availableWidth < Math.floor(effectiveWidth))) {
                display = Utils.repeatString('#', Math.max(1, Math.floor(availableWidth / font.getTextWidth('#'))));
                effectiveWidth = font.getTextWidth(display);
            }

            // create a single text line
            textDesc.lines.push(new LineDescriptor(display, effectiveWidth));
        }

        // the resulting height of all text lines
        totalHeight = textDesc.lines.length * rowHeight;

        // store width and height of the text box in the matrix element for external usage
        this.contentWidth = totalWidth;
        this.contentHeight = totalHeight;

        // initialize the clipping rectangle (text may be taller than height of cell)
        textDesc.clip.left -= overflowL;
        textDesc.clip.width += overflowL + overflowR;

        // store text overflow width
        textDesc.overflowL = overflowL;
        textDesc.overflowR = overflowR;

        // vertical position of the first text line
        var offsetY = 0;
        // rendering line height (enlarged for vertically adjusted text)
        var lineHeight = rowHeight;

        // calculate vertical start position of the first text line
        switch (attributes.cell.alignVert) {
            case 'top':
                offsetY = innerRect.top;
                break;
            case 'middle':
                offsetY = innerRect.top + Math.round((innerRect.height - totalHeight) / 2);
                break;
            case 'justify':
                offsetY = innerRect.top;
                // enlarge rendering line height if possible
                if ((textDesc.lines.length > 1) && (totalHeight < innerRect.height)) {
                    lineHeight = rowHeight + (innerRect.height - totalHeight) / (textDesc.lines.length - 1);
                }
                break;
            default:
                // bottom alignment, and fall-back for unknown alignments
                offsetY = innerRect.top + innerRect.height - totalHeight;
        }

        // adjust vertical offset to font base line position (as used by canvas renderer)
        offsetY += baseLineOffset + 1;

        // calculate effective rendering positions for all text lines
        textDesc.lines.forEach(function (lineDesc, lineIndex) {

            // calculate horizontal text position according to alignment
            switch (textOrient.baseBoxAlign) {
                case 'center':
                    lineDesc.x = innerRect.left + paddingLeft + (availableWidth - lineDesc.width) / 2;
                    break;
                case 'right':
                    lineDesc.x = innerRect.left + innerRect.width - paddingRight - lineDesc.width;
                    break;
                default:
                    // left alignment, and fall-back for unknown alignments
                    lineDesc.x = innerRect.left + paddingLeft;
            }

            // calculate vertical text position
            lineDesc.y = Math.round(offsetY + lineIndex * lineHeight);
        });

        // CSS text color, calculated from text color and fill color attributes
        var textColor = styleDesc.textColor;

        // color from a number format overrides cell text color; use implicit hyperlink color if specified
        if (section && section.rgbColor) {
            textColor = '#' + section.rgbColor;
        } else if (renderProps && renderProps.hyperlink && !styleDesc.explicitTextColor) {
            textColor = '#' + docModel.getTheme().getSchemeColor('hyperlink', '00f');
        }

        // the fill style for canvas rendering (text will be rendered in fill mode, without outline)
        textDesc.fill = textColor;

        // all font style properties for canvas rendering
        textDesc.font = { font: font.getCanvasFont(), direction: textOrient.textDir };

        // calculate text decoration path settings
        textDesc.decoration = (function () {

            // active underline and strike-through styles
            var underline = attributes.character.underline;
            var strike = attributes.character.strike !== 'none';

            // nothing to do without any text decoration attributes
            if (!underline && !strike) { return null; }

            // width of the decoration lines, depending on the font size
            var width = Math.max(1, Math.round(font.size / 15));
            // the rendering path for the line(s)
            var path = new Path();

            function pushDecorationLine(lineDesc, relOffset) {
                var lineY = lineDesc.y + Math.round(rowHeight * relOffset) - width / 2;
                path.pushLine(lineDesc.x, lineY, lineDesc.x + Math.ceil(lineDesc.width), lineY);
            }

            // add line segments for all text lines to the path
            textDesc.lines.forEach(function (lineDesc) {
                if (underline) { pushDecorationLine(lineDesc, 0.1); }
                if (strike) { pushDecorationLine(lineDesc, -0.2); }
            });

            // return the text decoration settings object (TODO: own underline color in ODF?)
            return { line: { style: textColor, width: width }, path: path };
        }());
    };

    /**
     * Returns whether this cell element is part of a merged range.
     *
     * @returns {Boolean}
     *  Whether this cell element is part of a merged range.
     */
    MatrixCellElement.prototype.isMerged = function () {
        return !!this.merged;
    };

    /**
     * Returns whether this cell element represents the top-left visible cell
     * of a merged range (this may not necessarily be the real origin cell, if
     * the leading columns and/or rows of the merged range are hidden).
     *
     * @returns {Boolean}
     *  Whether this cell element represents the top-left visible cell of a
     *  merged range.
     */
    MatrixCellElement.prototype.isMergedOrigin = function () {
        return this.isMerged() && this.merged.shrunkenRange.start.equals(this.address);
    };

    /**
     * Returns whether this cell element represents a cell covered by a merged
     * range (all cells in the merged range but the visible origin cell, see
     * method MatrixCellElement.isMergedOrigin() for more details).
     *
     * @returns {Boolean}
     *  Whether this cell element represents a cell covered by a merged range.
     */
    MatrixCellElement.prototype.isMergedCovered = function () {
        return this.isMerged() && this.merged.shrunkenRange.start.differs(this.address);
    };

    /**
     * Returns whether this cell element represents a cell covered by a merged
     * range that is located at the visible outer borders of the merged range.
     *
     * @returns {Boolean}
     *  Whether this cell element represents a border cell of a merged range.
     */
    MatrixCellElement.prototype.isMergedBorder = function () {
        return this.isMerged() && this.merged.shrunkenRange.isBorderAddress(this.address);
    };

    /**
     * Returns whether this cell element is covered by a merged range which
     * contains multiple columns (regardless of how many of these columns are
     * currently visible).
     *
     * @returns {Boolean}
     *  Whether this cell element is covered by a merged range with multiple
     *  columns.
     */
    MatrixCellElement.prototype.hasMergedColumns = function () {
        return this.isMerged() && (this.merged.mergedRange.start[0] < this.merged.mergedRange.end[0]);
    };

    /**
     * Returns whether this cell element is covered by a merged range which
     * contains multiple rows (regardless of how many of these rows are
     * currently visible).
     *
     * @returns {Boolean}
     *  Whether this cell element is covered by a merged range with multiple
     *  rows.
     */
    MatrixCellElement.prototype.hasMergedRows = function () {
        return this.isMerged() && (this.merged.mergedRange.start[1] < this.merged.mergedRange.end[1]);
    };

    /**
     * Returns whether this cell element contains overflowing text contents.
     *
     * @param {Boolean} forward
     *  Whether to check for overflowing text to the left (false), or to the
     *  right (true).
     *
     * @returns {Boolean}
     *  Whether this cell element contains text contents overflowing to the
     *  specified direction.
     */
    MatrixCellElement.prototype.hasOverflowText = function (forward) {
        return !!this.text && (this.text[forward ? 'overflowR' : 'overflowL'] > 0);
    };

    /**
     * Returns the effective style of a single border line of this matrix cell
     * element. If the element does not contain own formatting attributes, the
     * default border styles form the row or column will be returned.
     *
     * @param {String} key
     *  The key for the border line to be returned, as used in the property
     *  MatrixElement.borders.
     *
     * @returns {Object|Null}
     *  The effective style of the specified border line; or null, if the
     *  border line is invisible.
     */
    MatrixCellElement.prototype.getBorder = function (key) {
        var borders = this.style ? this.style.borders : this.row.style ? this.row.style.borders : this.col.style.borders;
        return (borders && (!this.skipBorders || !this.skipBorders[key])) ? borders[key] : null;
    };

    /**
     * Unlinks this instance from all adjacent matrix elements, and updates the
     * old adjacent matrix elements accordingly.
     */
    MatrixCellElement.prototype.unlink = function () {

        // unlink this element in the adjacent header elements
        if (this.col && (this.col.f === this)) { this.col.f = this.b; }
        if (this.row && (this.row.f === this)) { this.row.f = this.r; }

        // unlink this element in the adjacent cell elements
        if (this.l) { this.l.r = this.r; }
        if (this.r) { this.r.l = this.l; }
        if (this.t) { this.t.b = this.b; }
        if (this.b) { this.b.t = this.t; }

        // release the references to all matrix elements
        this.col = this.row = null;
        this.l = this.r = this.t = this.b = null;
    };

    // class MatrixHeaderCache ================================================

    /**
     * Helper class representing the column header, or the row header elements
     * of the rendering matrix. The elements will be stored as sorted array,
     * and in an object mapped by column/row index, for maximum performance.
     *
     * @constructor
     *
     * @property {Boolean} columns
     *  Whether this instance represents the column header of a rendering
     *  matrix (true), or the row header (false).
     *
     * @property {SimpleMap<HeaderBoundary>} boundaries
     *  All column/row boundaries represented by this matrix header, mapped by
     *  pane side identifier ('left' and 'right' for the column header; 'top'
     *  and 'bottom' for the row header of the rendering matrix).
     *
     * @property {IntervalArray} intervals
     *  The unified column/row intervals covered by this matrix header.
     *
     * @property {SortedArray<MatrixHeaderElement>} array
     *  An array of matrix header elements, sorted by column/row index.
     *
     * @property {SimpleMap<MatrixHeaderElement>} map
     *  A map with all matrix header elements contained in the cache, mapped by
     *  column/row index.
     */
    function MatrixHeaderCache(columns) {

        // bounding intervals and pixel positions
        this.columns = columns;

        // bounding intervals and pixel positions
        this.boundaries = new SimpleMap();

        // shrunken and merged bounding intervals
        this.intervals = new IntervalArray();

        // array of header elements, sorted by column/row index
        this.array = new SortedArray(function (element) { return element.index; });

        // map of header elements, mapped by column/row index
        this.map = new SimpleMap();

    } // class MatrixHeaderCache

    // public methods ---------------------------------------------------------

    /**
     * Returns all header elements in this cache as plain sorted array.
     *
     * @returns {Array<MatrixHeaderElement>}
     *  All header elements in this cache.
     */
    MatrixHeaderCache.prototype.getElements = function () {
        return this.array.values();
    };

    /**
     * Returns the specified header element.
     *
     * @param {Number} index
     *  The column/row index of the header element.
     *
     * @returns {HeaderElement|Null}
     *  The specified header element if existing, otherwise null.
     */
    MatrixHeaderCache.prototype.get = function (index) {
        return this.map.get(index, null);
    };

    /**
     * Returns the first visible header element in the specified column/row
     * interval.
     *
     * @param {Number} first
     *  The index of the first column/row in the interval.
     *
     * @param {Number} last
     *  The index of the last column/row in the interval.
     *
     * @returns {HeaderElement|Null}
     *  The first visible header element in the interval if existing, otherwise
     *  null.
     */
    MatrixHeaderCache.prototype.findFirst = function (first, last) {

        // for performance, look into the map first (simple JS property access)
        var element = this.map.get(first);
        if (element) { return element; }

        // search for a header element in the sorted array
        var desc = this.array.find(first, 'next');
        return (desc && (desc.index <= last)) ? desc.value : null;
    };

    /**
     * Invokes the passed callback function for all visible header elements
     * covered by the passed column/row interval.
     *
     * @param {Number} first
     *  The index of the first column/row in the interval.
     *
     * @param {Number} last
     *  The index of the last column/row in the interval.
     *
     * @param {Function} callback
     *  The callback function invoked for every visible header element in
     *  ascending order. Receives the following parameters:
     *  (1) {MatrixHeaderElement} element
     *      The header element of the rendering matrix currently visited.
     */
    MatrixHeaderCache.prototype.iterateElements = function (first, last, callback, context) {
        for (var element = this.findFirst(first, last); element && (element.index <= last); element = element.n) {
            callback.call(context, element);
        }
    };

    // static class RenderUtils ===============================================

    /**
     * The static class RenderUtils exports various small helper classes for
     * the rendering matrix. Additionally, the class is a console logger bound
     * to the URL hash flag 'spreadsheet:log-renderer', that logs detailed
     * information about the cell caching and rendering process.
     */
    var RenderUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Global pixel resolution for canvas rendering.
     *
     * @constant
     */
    RenderUtils.CANVAS_RESOLUTION = CANVAS_RESOLUTION;

    /**
     * Width of a grid line for rendering in the scaled canvas coordinate
     * system (the size of a physical canvas pixel in the public coordinate
     * system of the canvas, according to pixel resolution).
     *
     * @constant
     */
    RenderUtils.GRID_LINE_WIDTH = GRID_LINE_WIDTH;

    // logger interface -------------------------------------------------------

    Logger.extend(RenderUtils, { enable: 'spreadsheet:log-renderer', prefix: 'RENDER' });

    // classes ----------------------------------------------------------------

    RenderUtils.BorderDescriptor = BorderDescriptor;
    RenderUtils.StyleDescriptor = StyleDescriptor;
    RenderUtils.MergeDescriptor = MergeDescriptor;
    RenderUtils.MatrixHeaderElement = MatrixHeaderElement;
    RenderUtils.MatrixCellElement = MatrixCellElement;
    RenderUtils.MatrixHeaderCache = MatrixHeaderCache;

    // exports ================================================================

    return RenderUtils;

});
