/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/cellrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/config',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, Config, SimpleMap, Canvas, TriggerObject, TimerMixin, Color, SheetUtils, PaneUtils, RenderUtils) {

    'use strict';

    // mathematical constants
    var PI = Math.PI;
    var PI16 = PI / 6;
    var PI14 = PI / 4;
    var PI12 = PI / 2;
    var PI34 = 3 * PI14;
    var PI54 = 5 * PI14;
    var PI56 = 5 * PI16;
    var PI74 = 7 * PI14;

    // convenience shortcuts
    var Interval = SheetUtils.Interval;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;

    // line width for grid colors in the canvas
    var GRID_LINE_WIDTH = RenderUtils.GRID_LINE_WIDTH;

    // automatic grid color (dark gray instead of black, to make it visible on white and black backgrounds)
    var AUTO_GRID_COLOR = new Color('rgb', '666666');

    // reduction of clipping area size at leading border for canvas rendering,
    // needed to protect the existing outer cell borders
    var LEADING_CLIP_SIZE_REDUCTION = Utils.roundDown(SheetUtils.MIN_CELL_SIZE / 2, GRID_LINE_WIDTH);

    // reduction of clipping area size at trailing border for canvas rendering,
    // needed to protect the existing outer cell borders
    var TRAILING_CLIP_SIZE_REDUCTION = SheetUtils.MIN_CELL_SIZE - LEADING_CLIP_SIZE_REDUCTION;

    // class IconRenderer =====================================================

    /**
     * Renderer for the single icons of an icon set.
     *
     * @constructor
     *
     * @param {Number} size
     *  The effective icon size (width and height) that will be used to render
     *  the icons, in pixels. See static method IconRenderer.getEffectiveSize()
     *  for more details.
     */
    function IconRenderer(size) {

        // the effective icon size
        this.size = size;

        // line width for outlines, according to icon size
        this.LW = this.lineWidth(1);
        // line width for thick lines used for signs with shadow effect
        this.LW2 = this.lineWidth(2.5);
        // line offset needed to keep the outlines on entire pixels
        this.LO = this.LW / 2;

        // inner size (maximum dimension of the path)
        this.SI = this.size - this.LW;

        // leading position for outlines (top or left lines)
        this.C0 = this.coord(0);
        // center position for outlines (centered horizontal or vertical lines)
        this.CC = this.size / 2;
        // trailing position for outlines (for bottom or right lines)
        this.CE = this.mirror(this.C0);

        // radius of a full-size circle
        this.R = this.CC - this.C0;

        // the rendering context to be drawn into (will be set while creating an icon)
        this.context = null;
        // the cache for pre-rendered icons, mapped by icon identifiers
        this.cache = new SimpleMap();

    } // class IconRenderer

    // constants --------------------------------------------------------------

    /**
     * A map with names and RGB values of all colors used in icon sets.
     *
     * @type {Object<String, String>}
     *
     * @constant
     */
    IconRenderer.COLORS = {
        BLACK: '#555',
        BLUE: '#46f',
        GRAY: '#bbb',
        GREEN: '#2a4',
        PINK: '#faa',
        RED: '#e42',
        WHITE: '#fff',
        YELLOW: '#dc2'
    };

    // static methods ---------------------------------------------------------

    /**
     * Returns the effective line width for the specified icon size, and base
     * line width, to be used to render the outlines of icons.
     *
     * @param {Number} size
     *  The icon size, in pixels.
     *
     * @param {Number} width
     *  The base width of the lines to be drawn, in pixels.
     *
     * @returns {Number}
     *  The effective line width to be used to render the outlines of icons.
     */
    IconRenderer.getLineWidth = function (size, width) {
        return Math.ceil(size * width / 25);
    };

    /**
     * Returns the effective size used to render the icons for the passed size.
     * The passed size will be restricted to a minimum of 7 pixels, and may be
     * reduced by one pixel to keep the horizontal/vertical center lines on
     * entire pixels (this prevents blurry lines in the icons).
     *
     * @param {Number} size
     *  The maximum size available for rendering the icons, in pixels.
     *
     * @returns {Number}
     *  The effective icon size used to render the icons.
     */
    IconRenderer.getEffectiveSize = function (size) {

        // use a minimum size of 7 pixels
        var effSize = Math.max(size, 7);

        // get the base line width for outlines
        var lineWidth = IconRenderer.getLineWidth(effSize, 1);

        // reduce icon size to keep the horizontal/vertical center lines on entire pixels
        // (odd icon size for odd line widths; and even size for even line widths)
        var pad = (lineWidth & 1) ? (1 - (effSize & 1)) : (effSize & 1);
        return effSize - pad;
    };

    /**
     * Creates an icon renderer for the passed size in pixels, and stores the
     * renderer instances per effective icon size in an internal cache.
     *
     * @param {Number} size
     *  The maximum size available for rendering the icons, in pixels.
     *
     * @returns {IconRenderer}
     *  The icon renderer for the passed size. The effective icon size used by
     *  the renderer is contained in its property 'size'.
     */
    IconRenderer.create = (function () {

        // creates and caches an icon renderer per effective icon size
        var createRenderer = _.memoize(function (size) {
            return new IconRenderer(size);
        });

        // return the implementation of the method from local scope
        return function (size) {
            return createRenderer(IconRenderer.getEffectiveSize(size));
        };
    }());

    // private methods --------------------------------------------------------

    /**
     * Returns the effective line width for the specified base width, according
     * to the size of the icons to be rendered.
     *
     * @param {Number} width
     *  The base width of the lines to be drawn, in pixels.
     *
     * @returns {Number}
     *  The effective line width to be used to render the outlines of icons.
     */
    IconRenderer.prototype.lineWidth = function (width) {
        return IconRenderer.getLineWidth(this.size, width);
    };

    /**
     * Returns the effective value of a path coordinate. The resulting position
     * will be adjusted so that horizontal/vertical outlines drawn at these
     * coordinate will be kept on entire pixels to prevent blurry lines.
     *
     * @param {Number} pos
     *  The relative position inside the icon, in the interval [0;1]. The value
     *  0 represents the leading border (same value as the property 'C0'). The
     *  value 1 represents the trailing border (same value as the property
     *  'CE').
     *
     * @returns {Number}
     *  The effective value of a path coordinate.
     */
    IconRenderer.prototype.coord = function (pos) {
        return this.LO + Math.round(this.SI * pos);
    };

    /**
     * Returns the effective value of a path coordinate. Works similarly as the
     * method IconRenderer.coord(), but does not adjust the effective position
     * to entire pixels.
     *
     * @param {Number} pos
     *  The relative position inside the icon, in the interval [0;1]. The value
     *  0 represents the leading border (same value as the property 'C0'). The
     *  value 1 represents the trailing border (same value as the property
     *  'CE').
     *
     * @returns {Number}
     *  The effective value of a path coordinate.
     */
    IconRenderer.prototype.exact = function (pos) {
        return this.LO + this.SI * pos;
    };

    /**
     * Mirrors the passed effective coordinate which helps to create symmetric
     * icons. Calling the method IconRenderer.coord() with mirrored relative
     * positions may not result in the desired symmetric effective coordinates
     * due to internal rounding effects.
     *
     * @param {Number} coord
     *  The effective corrdinate, as returned for example by the methods
     *  IconRenderer.coord() or IconRenderer.exact().
     *
     * @returns {Number}
     *  The opposite coordinate.
     */
    IconRenderer.prototype.mirror = function (coord) {
        return this.size - coord;
    };

    /**
     * Returns a new empty path instance.
     *
     * @returns {Path}
     *  A new empty path instance.
     */
    IconRenderer.prototype.path = function () {
        return this.context.createPath();
    };

    /**
     * Draws (fills and/or strokes) the specified path into the internal
     * rendering context.
     *
     * @param {Path} path
     *  The canvas path to be drawn.
     *
     * @param {String|Null} color
     *  The name of a color, as defined in the color map IconRenderer.COLORS;
     *  or null to retain the current fill color.
     *
     * @param {String} [mode='all']
     *  The rendering mode for the path (one of 'all', 'stroke', or 'fill').
     *  See method Canvas.drawPath() for details.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawPath = function (path, color, mode) {
        if (color) { this.context.setFillStyle(IconRenderer.COLORS[color]); }
        this.context.setLineStyle({ width: this.LW, style: '#333' }).drawPath(path, mode);
        return this;
    };

    /**
     * Draws the specified label into the internal rendering context, as thick
     * lines with a black shadow effect.
     *
     * @param {Path} path
     *  The canvas path to be drawn.
     *
     * @param {String} color
     *  The name of a color to be used to draw the lines, as defined in the
     *  color map IconRenderer.COLORS.
     *
     * @param {Boolean} [shadow=false]
     *  Whether to draw the shadow effect.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawLabel = function (path, color, shadow) {
        this.context.setLineStyle({ width: this.LW2, style: IconRenderer.COLORS[color] });
        if (shadow) { this.context.setShadowStyle({ color: 'black', blur: this.LW2 }); }
        this.context.drawPath(path, 'stroke');
        return this;
    };

    /**
     * Draws the specified symbol into the internal rendering context.
     *
     * @param {Object} symbolId
     *  The identifier of a symbol.
     *
     * @param {Any} symbolArg
     *  Additional arguments needed to draw the symbol. The data type depends
     *  on the specified symbol. Usually, this argument consists of one or more
     *  color names used to fill the areas of the symbol.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawSymbol = (function () {

        // all symbol renderer implementations, expecting an initialized rendering context
        var SYMBOL_MAP = {

            arrowDown: function (color) {
                var c0 = this.C0;
                var c3 = this.coord(0.3);
                var c5 = this.CC;
                var c7 = this.mirror(c3);
                var ce = this.CE;
                this.drawPath(this.path().pushPolygon(c5, ce, c0, c5, c3, c5, c3, c0, c7, c0, c7, c5, ce, c5), color);
            },

            arrowRight: function (color) {
                this.drawSymbolRot270('arrowDown', color);
            },

            arrowUp: function (color) {
                this.drawSymbolRot180('arrowDown', color);
            },

            arrowDiagDown: function (color) {
                var c0 = this.C0;
                var c15 = this.coord(0.15);
                var c28 = this.coord(0.28);
                var c85 = this.mirror(c15);
                var d = (c85 - c28 - c15 + c0) / 2;
                this.drawPath(this.path().pushPolygon(c15, c85, c85, c85, c85, c15, c85 - d, c15 + d, c28, c0, c0, c28, c15 + d, c85 - d), color);
            },

            arrowDiagUp: function (color) {
                this.drawSymbolRot270('arrowDiagDown', color);
            },

            flag: function (color) {
                var c0 = this.C0;
                var c1 = this.coord(0.1);
                var c2 = this.coord(0.2);
                var c3 = this.coord(0.3);
                var c6 = 2 * c3 - c0;
                var c9 = this.mirror(c1);
                var ce = this.CE;
                this.drawPath(this.path().pushLineChain(c2, c0, c9, c3, c2, c6), color);
                this.drawPath(this.path().pushRect(c1, c0, c2 - c1, ce - c0), 'BLACK');
            },

            star: function (colors) {
                // uppermost tip
                var c1x = this.CC, c1y = this.C0;
                // upper left/right tips
                var c2xl = this.C0, c2xr = this.CE, c2y = this.coord(0.363);
                // lower left/right tips
                var c3xl = this.coord(0.155), c3xr = this.mirror(c3xl), c3y = this.coord(0.951);
                // upper left/right inner angles
                var c4xl = this.exact(0.345), c4xr = this.mirror(c4xl), c4y = this.exact(0.313);
                // lower left/right inner angles
                var c5xl = this.exact(0.25), c5xr = this.mirror(c5xl), c5y = this.exact(0.607);
                // lowermost inner angle
                var c6x = this.CC, c6y = this.exact(0.789);

                // path for right background area
                var pathR = this.path().pushPolygon(c1x, c1y, c4xr, c4y, c2xr, c2y, c5xr, c5y, c3xr, c3y, c6x, c6y);
                // path of the full star
                var pathO = this.path().pushPolygon(c1x, c1y, c4xr, c4y, c2xr, c2y, c5xr, c5y, c3xr, c3y, c6x, c6y, c3xl, c3y, c5xl, c5y, c2xl, c2y, c4xl, c4y);

                this.drawPath(pathO, colors[0], 'fill');
                this.drawPath(pathR, colors[1], 'fill');
                this.drawPath(pathO, null, 'stroke');
            },

            circle: function (color) {
                this.drawPath(this.path().pushCircle(this.CC, this.CC, this.R), color);
            },

            quarter: function (args) {
                var c0 = this.C0;
                var c5 = this.CC;
                var ce = this.CE;
                var fillPath = this.path().moveTo(c5, c0);
                var strokePath = this.path().pushCircle(c5, c5, this.R).moveTo(c5, c0);
                switch (args.quarters) {
                    case 1:
                        fillPath.arc(c5, c5, this.R, -PI12, 0).lineTo(c5, c5);
                        strokePath.lineTo(c5, c5).lineTo(ce, c5);
                        break;
                    case 2:
                        fillPath.arc(c5, c5, this.R, -PI12, PI12);
                        strokePath.lineTo(c5, ce);
                        break;
                    case 3:
                        fillPath.arc(c5, c5, this.R, -PI12, PI).lineTo(c5, c5);
                        strokePath.lineTo(c5, c5).lineTo(c0, c5);
                        break;
                }
                this.drawPath(this.path().pushCircle(c5, c5, this.R), args.color1, 'fill');
                this.drawPath(fillPath.close(), args.color2, 'fill');
                this.drawPath(strokePath, null, 'stroke');
            },

            trafficLight: function (color) {
                var c15 = this.coord(0.15);
                var c85 = this.mirror(c15);
                var r1 = c15 - this.C0;
                var r2 = this.CC - this.coord(0.1);
                this.drawPath(this.path().arc(c15, c15, r1, PI, -PI12).arc(c85, c15, r1, -PI12, 0).arc(c85, c85, r1, 0, PI12).arc(c15, c85, r1, PI12, PI).close(), 'BLACK');
                this.drawPath(this.path().pushCircle(this.CC, this.CC, r2), color);
            },

            rating: function (colors) {
                var c5 = this.CC;
                var w = Math.floor((c5 - this.C0) / 2);
                var c0 = c5 - 2 * w;
                var c25 = c5 - w;
                var c75 = c5 + w;
                var ce = c5 + 2 * w;
                var h = ce - c0;
                this.drawPath(this.path().pushRect(c0, c0 + 3 * w, w, h - 3 * w), colors[0]);
                this.drawPath(this.path().pushRect(c25, c0 + 2 * w, w, h - 2 * w), colors[1]);
                this.drawPath(this.path().pushRect(c5, c0 + w, w, h - w), colors[2]);
                this.drawPath(this.path().pushRect(c75, c0, w, h), colors[3]);
            },

            boxes: function (colors) {
                var c0 = this.C0;
                var c15 = this.coord(0.15);
                var c5 = this.CC;
                var c85 = this.mirror(c15);
                var ce = this.CE;
                var r = c15 - this.C0;
                this.drawPath(this.path().moveTo(c5, c5).lineTo(c5, ce).arc(c15, c85, r, PI12, PI).lineTo(c0, c5).close(), colors[0]);
                this.drawPath(this.path().moveTo(c5, c5).lineTo(ce, c5).arc(c85, c85, r, 0, PI12).lineTo(c5, ce).close(), colors[1]);
                this.drawPath(this.path().moveTo(c5, c5).lineTo(c0, c5).arc(c15, c15, r, PI, -PI12).lineTo(c5, c0).close(), colors[2]);
                this.drawPath(this.path().moveTo(c5, c5).lineTo(c5, c0).arc(c85, c15, r, -PI12, 0).lineTo(ce, c5).close(), colors[3]);
            },

            squareSign: function (color) {
                var c15 = this.coord(0.15);
                var c5 = this.CC;
                var c85 = this.mirror(c15);
                var r = c15 - this.C0;
                this.drawPath(this.path().arc(c5, c15, r, -PI34, -PI14).arc(c85, c5, r, -PI14, PI14).arc(c5, c85, r, PI14, PI34).arc(c15, c5, r, PI34, -PI34).close(), color);
            },

            triangleSign: function (color) {
                var c15 = this.coord(0.15);
                var c5 = this.CC;
                var c75 = this.coord(0.75);
                var c85 = this.mirror(c15);
                var r = c15 - this.C0;
                this.drawPath(this.path().arc(c5, c15, r, -PI56, -PI16).arc(c85, c75, r, -PI16, PI12).arc(c15, c75, r, PI12, -PI56).close(), color);
            },

            cross: function (color) {
                var c0 = this.C0;
                var c2 = this.coord(0.2);
                var c5 = this.CC;
                var c8 = this.mirror(c2);
                var ce = this.CE;
                var d = c2 - c0;
                var c3 = c5 - d;
                var c7 = c5 + d;
                this.drawPath(this.path().pushPolygon(c2, c0, c0, c2, c3, c5, c0, c8, c2, ce, c5, c7, c8, ce, ce, c8, c7, c5, ce, c2, c8, c0, c5, c3), color);
            },

            exclamation: function (color) {
                var c0 = this.C0;
                var c35 = this.coord(0.35);
                var c5 = this.CC;
                var c6 = this.coord(0.6);
                var ce = this.CE;
                var r = c5 - c35;
                this.drawPath(this.path().pushRect(c35, c0, 2 * r, c6 - c0).pushCircle(c5, ce - r, r), color);
            },

            check: function (color) {
                var c0 = this.C0;
                var c4 = this.coord(0.4);
                var c6 = this.mirror(c4);
                var ce = this.CE;
                var d = this.coord(0.2) - c0;
                var o = Math.floor(d / 2);
                this.drawPath(this.path().pushPolygon(c0, c6 - o, c4, ce - o, ce, c4 - o, ce - d, c4 - d - o, c4, ce - 2 * d - o, c0 + d, c6 - d - o), color);
            },

            minus: function (color) {
                var c0 = this.C0;
                var c4 = this.coord(0.38);
                var c6 = this.mirror(c4);
                this.drawPath(this.path().pushRect(c0, c4, this.SI, c6 - c4), color);
            },

            triangleDown: function (color) {
                var c0 = this.C0;
                var c25 = this.coord(0.25);
                var c5 = this.CC;
                var c75 = c25 + c5 - c0;
                var ce = this.CE;
                this.drawPath(this.path().pushPolygon(c0, c25, c5, c75, ce, c25), color);
            },

            triangleUp: function (color) {
                this.drawSymbolRot180('triangleDown', color);
            },

            crossLabel: function () {
                var c3 = this.coord(0.3);
                var c7 = this.mirror(c3);
                this.drawLabel(this.path().pushLine(c3, c3, c7, c7).pushLine(c3, c7, c7, c3), 'WHITE', true);
            },

            exclamationLabel: function () {
                var c25 = this.coord(0.25);
                var c5 = this.CC;
                var c55 = this.coord(0.55);
                var c75 = this.mirror(c25);
                this.drawLabel(this.path().pushLine(c5, c25, c5, c55).pushLine(c5, c75, c5, c75 + 1), 'WHITE', true);
            },

            checkLabel: function () {
                var c25 = this.coord(0.25);
                var c75 = this.mirror(c25);
                var c5 = (c75 + c25) / 2;
                var d = (c75 - c25) / 3;
                this.drawLabel(this.path().pushLineChain(c25, c5, c25 + d, c5 + d, c75, c5 - d), 'WHITE', true);
            },

            smileyFace: function (type) {

                // eyes
                var cxl = this.coord(0.35);
                var cxr = this.mirror(cxl);
                var cy = this.coord(0.4);
                this.drawPath(this.path().pushCircle(cxl, cy, 1).pushCircle(cxr, cy, 1), 'BLACK');

                // mouth
                var r = this.coord(0.35);
                switch (type) {
                    case 'neutral':
                        cxl = this.coord(0.35);
                        cxr = this.mirror(cxl);
                        cy = this.coord(0.65);
                        this.drawPath(this.path().pushLine(cxl, cy, cxr, cy), null, 'stroke');
                        break;
                    case 'good':
                        cy = this.coord(0.4);
                        this.drawPath(this.path().pushArc(this.CC, cy, r, PI14, PI34), null, 'stroke');
                        break;
                    case 'bad':
                        cy = this.CE;
                        this.drawPath(this.path().pushArc(this.CC, cy, r, PI54, PI74), null, 'stroke');
                        break;
                }
            }
        };

        // return the implementation of the method from local scope
        return function (symbolId, symbolArg) {
            SYMBOL_MAP[symbolId].call(this, symbolArg);
            return this;
        };
    }());

    /**
     * Rotates the rendering context by 180 degrees, and draws the specified
     * symbol.
     *
     * @param {Object} symbolId
     *  The identifier of a symbol to be drawn rotated.
     *
     * @param {Any} symbolArg
     *  Additional arguments needed to draw the symbol. See description of the
     *  method IconRenderer.drawSymbol() for details.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawSymbolRot180 = function (symbolId, symbolArg) {
        this.context.render(function () {
            this.context.translate(0, this.size).scale(1, -1);
            this.drawSymbol(symbolId, symbolArg);
        }, this);
        return this;
    };

    /**
     * Rotates the rendering context by 270 degrees clockwise, and draws the
     * specified symbol.
     *
     * @param {Object} symbolId
     *  The identifier of a symbol to be drawn rotated.
     *
     * @param {Any} symbolArg
     *  Additional arguments needed to draw the symbol. See description of the
     *  method IconRenderer.drawSymbol() for details.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawSymbolRot270 = function (symbolId, symbolArg) {
        this.context.render(function () {
            this.context.translate(0, this.size).rotate(-PI12);
            this.drawSymbol(symbolId, symbolArg);
        }, this);
        return this;
    };

    // public methods ---------------------------------------------------------

    /**
     * Draws the specified icon from an icon set.
     *
     * @param {ContextWrapper} context
     *  The rendering context to draw the icon into.
     *
     * @param {Number} x
     *  The horizontal position of the icon in the passed context.
     *
     * @param {Number} y
     *  The vertical position of the icon in the passed context.
     *
     * @param {String} iconSetId
     *  The identifier of an icon set, as used in document operations (ignoring
     *  the character case).
     *
     * @param {Number} iconIndex
     *  The zero-based index of the icon from the icon set to be drawn.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawIcon = (function () {

        // maps identifiers of all supported icons to arrays of symbol descriptors
        var ICON_MAP = {

            // arrows
            redArrowDown:        [{ id: 'arrowDown',     arg: 'RED'    }],
            yellowArrowDiagDown: [{ id: 'arrowDiagDown', arg: 'YELLOW' }],
            yellowArrowRight:    [{ id: 'arrowRight',    arg: 'YELLOW' }],
            yellowArrowDiagUp:   [{ id: 'arrowDiagUp',   arg: 'YELLOW' }],
            greenArrowUp:        [{ id: 'arrowUp',       arg: 'GREEN'  }],
            grayArrowDown:       [{ id: 'arrowDown',     arg: 'GRAY'   }],
            grayArrowDiagDown:   [{ id: 'arrowDiagDown', arg: 'GRAY'   }],
            grayArrowRight:      [{ id: 'arrowRight',    arg: 'GRAY'   }],
            grayArrowDiagUp:     [{ id: 'arrowDiagUp',   arg: 'GRAY'   }],
            grayArrowUp:         [{ id: 'arrowUp',       arg: 'GRAY'   }],

            // flags
            redFlag:    [{ id: 'flag', arg: 'RED'    }],
            yellowFlag: [{ id: 'flag', arg: 'YELLOW' }],
            greenFlag:  [{ id: 'flag', arg: 'GREEN'  }],

            // circles
            redCircle:          [{ id: 'circle',  arg: 'RED'    }],
            yellowCircle:       [{ id: 'circle',  arg: 'YELLOW' }],
            greenCircle:        [{ id: 'circle',  arg: 'GREEN'  }],
            grayCircle:         [{ id: 'circle',  arg: 'GRAY'   }],
            blackCircle:        [{ id: 'circle',  arg: 'BLACK'  }],
            pinkCircle:         [{ id: 'circle',  arg: 'PINK'   }],
            redCrossCircle:     [{ id: 'circle',  arg: 'RED'    }, { id: 'crossLabel'       }],
            yellowExclamCircle: [{ id: 'circle',  arg: 'YELLOW' }, { id: 'exclamationLabel' }],
            greenCheckCircle:   [{ id: 'circle',  arg: 'GREEN'  }, { id: 'checkLabel'       }],
            grayQuarter0:       [{ id: 'circle',  arg: 'WHITE'  }],
            grayQuarter1:       [{ id: 'quarter', arg: { quarters: 1, color1: 'WHITE', color2: 'GRAY' } }],
            grayQuarter2:       [{ id: 'quarter', arg: { quarters: 2, color1: 'WHITE', color2: 'GRAY' } }],
            grayQuarter3:       [{ id: 'quarter', arg: { quarters: 3, color1: 'WHITE', color2: 'GRAY' } }],
            grayQuarter4:       [{ id: 'circle',  arg: 'GRAY'   }],

            // smileys
            greenSmileyGood:     [{ id: 'circle',  arg: 'GREEN'  }, { id: 'smileyFace', arg: 'good'    }],
            yellowSmileyGood:    [{ id: 'circle',  arg: 'YELLOW' }, { id: 'smileyFace', arg: 'good'    }],
            yellowSmileyNeutral: [{ id: 'circle',  arg: 'YELLOW' }, { id: 'smileyFace', arg: 'neutral' }],
            yellowSmileyBad:     [{ id: 'circle',  arg: 'YELLOW' }, { id: 'smileyFace', arg: 'bad'     }],
            redSmileyBad:        [{ id: 'circle',  arg: 'RED'    }, { id: 'smileyFace', arg: 'bad'     }],

            // traffic lights
            redTrafficLight:    [{ id: 'trafficLight', arg: 'RED'    }],
            yellowTrafficLight: [{ id: 'trafficLight', arg: 'YELLOW' }],
            greenTrafficLight:  [{ id: 'trafficLight', arg: 'GREEN'  }],

            // ratings
            whiteStar:      [{ id: 'star',   arg: ['WHITE',  'WHITE']  }],
            halfYellowStar: [{ id: 'star',   arg: ['YELLOW', 'WHITE']  }],
            yellowStar:     [{ id: 'star',   arg: ['YELLOW', 'YELLOW'] }],
            blueRating0:    [{ id: 'rating', arg: ['GRAY', 'GRAY', 'GRAY', 'GRAY'] }],
            blueRating1:    [{ id: 'rating', arg: ['BLUE', 'GRAY', 'GRAY', 'GRAY'] }],
            blueRating2:    [{ id: 'rating', arg: ['BLUE', 'BLUE', 'GRAY', 'GRAY'] }],
            blueRating3:    [{ id: 'rating', arg: ['BLUE', 'BLUE', 'BLUE', 'GRAY'] }],
            blueRating4:    [{ id: 'rating', arg: ['BLUE', 'BLUE', 'BLUE', 'BLUE'] }],
            blueBoxes0:     [{ id: 'boxes',  arg: ['GRAY', 'GRAY', 'GRAY', 'GRAY'] }],
            blueBoxes1:     [{ id: 'boxes',  arg: ['BLUE', 'GRAY', 'GRAY', 'GRAY'] }],
            blueBoxes2:     [{ id: 'boxes',  arg: ['BLUE', 'BLUE', 'GRAY', 'GRAY'] }],
            blueBoxes3:     [{ id: 'boxes',  arg: ['BLUE', 'BLUE', 'BLUE', 'GRAY'] }],
            blueBoxes4:     [{ id: 'boxes',  arg: ['BLUE', 'BLUE', 'BLUE', 'BLUE'] }],

            // signs
            redSquareSign:      [{ id: 'squareSign',   arg: 'RED'    }],
            yellowTriangleSign: [{ id: 'triangleSign', arg: 'YELLOW' }],
            redCross:           [{ id: 'cross',        arg: 'RED'    }],
            yellowExclam:       [{ id: 'exclamation',  arg: 'YELLOW' }],
            greenCheck:         [{ id: 'check',        arg: 'GREEN'  }],
            yellowMinus:        [{ id: 'minus',        arg: 'YELLOW' }],
            redTriangleDown:    [{ id: 'triangleDown', arg: 'RED'    }],
            greenTriangleUp:    [{ id: 'triangleUp',   arg: 'GREEN'  }]
        };

        // maps identifiers of all supported icon sets (lower-case!) to arrays of icon identifiers (see ICON_MAP)
        var ICON_SET_MAP = {

            // 3 icons
            '3arrows':         ['redArrowDown', 'yellowArrowRight', 'greenArrowUp'],
            '3arrowsgray':     ['grayArrowDown', 'grayArrowRight', 'grayArrowUp'],
            '3flags':          ['redFlag', 'yellowFlag', 'greenFlag'],
            '3trafficlights1': ['redCircle', 'yellowCircle', 'greenCircle'],
            '3trafficlights2': ['redTrafficLight', 'yellowTrafficLight', 'greenTrafficLight'],
            '3signs':          ['redSquareSign', 'yellowTriangleSign', 'greenCircle'],
            '3symbols':        ['redCrossCircle', 'yellowExclamCircle', 'greenCheckCircle'],
            '3symbols2':       ['redCross', 'yellowExclam', 'greenCheck'],
            '3stars':          ['whiteStar', 'halfYellowStar', 'yellowStar'],
            '3triangles':      ['redTriangleDown', 'yellowMinus', 'greenTriangleUp'],
            '3smilies':        ['yellowSmileyGood', 'yellowSmileyNeutral', 'yellowSmileyBad'], // ODF only, with wrong spelling 'Smilies'
            '3colorsmilies':   ['greenSmileyGood', 'yellowSmileyNeutral', 'redSmileyBad'], // ODF only, with wrong spelling 'Smilies'

            // 4 icons
            '4arrows':         ['redArrowDown', 'yellowArrowDiagDown', 'yellowArrowDiagUp', 'greenArrowUp'],
            '4arrowsgray':     ['grayArrowDown', 'grayArrowDiagDown', 'grayArrowDiagUp', 'grayArrowUp'],
            '4redtoblack':     ['blackCircle', 'grayCircle', 'pinkCircle', 'redCircle'],
            '4rating':         ['blueRating1', 'blueRating2', 'blueRating3', 'blueRating4'],
            '4trafficlights':  ['blackCircle', 'redCircle', 'yellowCircle', 'greenCircle'],

            // 5 icons
            '5arrows':         ['redArrowDown', 'yellowArrowDiagDown', 'yellowArrowRight', 'yellowArrowDiagUp', 'greenArrowUp'],
            '5arrowsgray':     ['grayArrowDown', 'grayArrowDiagDown', 'grayArrowRight', 'grayArrowDiagUp', 'grayArrowUp'],
            '5rating':         ['blueRating0', 'blueRating1', 'blueRating2', 'blueRating3', 'blueRating4'],
            '5quarters':       ['grayQuarter0', 'grayQuarter1', 'grayQuarter2', 'grayQuarter3', 'grayQuarter4'],
            '5boxes':          ['blueBoxes0', 'blueBoxes1', 'blueBoxes2', 'blueBoxes3', 'blueBoxes4']
        };

        // return the implementation of the method from local scope
        return function (context, x, y, iconSetId, iconIndex) {

            // the icon set (array of icon identifiers)
            var iconSet = ICON_SET_MAP[iconSetId.toLowerCase()];
            if (!iconSet) { return; }

            // the icon identifier
            var iconId = iconSet[iconIndex];
            if (!iconId) { return; }

            // resolve an existing icon from the internal cache, or create a new canvas element
            var iconCanvas = this.cache.getOrCreate(iconId, function () {

                // internal helper canvas needed to render icons independently from
                var iconCanvas = new Canvas().initialize({ width: this.size, height: this.size });

                // render the icon into the canvas
                iconCanvas.render(function (iconContext, width, height) {

                    // initialize the rendering context
                    iconContext.clearRect(0, 0, width, height);
                    iconContext.setLineStyle({ cap: 'round', join: 'round' });

                    // draw all symbols into the internal rendering context
                    this.context = iconContext;
                    ICON_MAP[iconId].forEach(function (symbolDesc) {
                        this.drawSymbol(symbolDesc.id, symbolDesc.arg);
                    }, this);
                    this.context = null;
                }, this);

                return iconCanvas;
            }, this);

            // copy the icon to the specified rendering context, either from canvas, or from image
            context.drawImage(iconCanvas, x, y);
            return this;
        };
    }());

    // class CellRenderer =====================================================

    /**
     * Renders the cell contents, cell formatting, and the cell grid lines into
     * a single grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this cell layer renderer.
     */
    function CellRenderer(gridPane) {

        // self reference
        var self = this;

        // document model and view, and the rendering cache of the document view
        var docView = gridPane.getDocView();
        var docModel = docView.getDocModel();
        var renderCache = docView.getRenderCache();
        var app = docView.getApp();

        // current zoom factor, for scaling of canvas element
        var sheetZoom = 1;

        // canvas element for all cell contents
        var cellCanvas = new Canvas({ classes: 'cell-layer' });

        // the layer node containing outlines for matrix formulas
        var matrixLayerNode = gridPane.createLayerNode('matrix-layer');

        // the layer node containing outlines for table ranges
        var tableLayerNode = gridPane.createLayerNode('table-layer');

        // the cell range and absolute position covered by the layer nodes in the sheet area
        var layerRange = null;
        var layerRectangle = null;

        // all pending cell ranges to be rendered debounced
        var pendingRanges = new RangeArray();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Renders the grid lines between the cells into the canvas element.
         */
        var renderGridLayer = RenderUtils.profileMethod('rendering grid layer', function (clip, range) {

            // nothing to do, if grid line are not enabled in the active sheet
            if (!docView.getSheetViewAttribute('showGrid')) { return; }

            // render grid lines into the canvas
            cellCanvas.render(clip, function (context) {

                // the model of the active sheet
                var sheetModel = docView.getSheetModel();
                // the current grid color
                var gridColor = sheetModel.getGridColor();
                // the path object containing all grid line positions
                var path = context.createPath();
                // all merged ranges with more than one visible column
                var mergedColRanges = new RangeArray();
                // all merged ranges with more than one visible row
                var mergedRowRanges = new RangeArray();
                // sub-pixel offset to keep grid lines on canvas pixels
                var subPxOffset = -GRID_LINE_WIDTH / 2;

                // fill the arrays of merged ranges in one pass
                renderCache.iterateMergedRanges(function (mergeDesc) {
                    var shrunkenRange = mergeDesc.shrunkenRange;
                    if (!shrunkenRange.singleCol()) { mergedColRanges.push(shrunkenRange); }
                    if (!shrunkenRange.singleRow()) { mergedRowRanges.push(shrunkenRange); }
                });

                // adds either all vertical or all horizontal grid lines to the path
                function addGridLines(columns) {

                    // the column/row collection in the other direction (for intervals of merged ranges)
                    var collection = columns ? docView.getRowCollection() : docView.getColCollection();
                    // merged ranges spanning over multiple columns/rows (causing multiple line segments)
                    var mergedRanges = columns ? mergedColRanges : mergedRowRanges;
                    // layer start offset in rendering direction
                    var layerOffset = columns ? layerRectangle.top : layerRectangle.left;
                    // layer size in rendering direction
                    var layerSize = columns ? layerRectangle.height : layerRectangle.width;
                    // function to push a line to the path
                    var pushLineFunc = columns ?
                        function (x, y1, y2) { path.pushLine(x, y1 - GRID_LINE_WIDTH, x, y2); } :
                        function (y, x1, x2) { path.pushLine(x1 - GRID_LINE_WIDTH, y, x2, y); };

                    // restrict layer size to sheet size
                    layerSize = Math.min(layerSize, collection.getTotalSize() - layerOffset);

                    // add the grid lines for all visible columns/rows to the path
                    renderCache.iterateInterval(range, columns, function (entry) {

                        // use the end position of the current column/row, translate by half a pixel to paint on whole pixels
                        var offset = entry.offset + entry.size + subPxOffset;
                        // all merged ranges that will break the grid line
                        var coveredMergedRanges = mergedRanges.filter(function (mergedRange) {
                            return (mergedRange.getStart(columns) <= entry.index) && (entry.index < mergedRange.getEnd(columns));
                        });
                        // the merged and sorted column/row intervals of the merged ranges
                        var mergedIntervals = coveredMergedRanges.intervals(!columns).merge();
                        // the coordinates of the line segments, as flat array
                        var lineCoords = null;

                        // calculate the line start/end coordinates
                        lineCoords = mergedIntervals
                            // get the pixel positions of the merged column/row intervals
                            .map(collection.getIntervalPosition, collection)
                            // filter hidden merged ranges
                            .filter(function (position) { return position.size > 0; })
                            // adjust layer offset in positions and convert to (nested) integer array
                            .map(function (position) { return [position.offset, position.offset + position.size]; });

                        // convert to a flat array, add start and end position of the layer rectangle
                        lineCoords = _.flatten(lineCoords);
                        lineCoords.unshift(layerOffset);
                        lineCoords.push(layerOffset + layerSize);

                        // draw all line segments
                        for (var index = 0, length = lineCoords.length; index < length; index += 2) {
                            if (lineCoords[index] < lineCoords[index + 1]) {
                                pushLineFunc(offset, lineCoords[index], lineCoords[index + 1]);
                            }
                        }
                    });
                }

                // add all grid lines to the path
                addGridLines(false);
                addGridLines(true);

                // paint everything with 20% opacity
                context.setGlobalAlpha(0.2);
                context.setLineStyle({ style: docModel.resolveColor(gridColor, AUTO_GRID_COLOR).css, width: GRID_LINE_WIDTH });
                context.drawPath(path, 'stroke');
            });
        });

        /**
         * Renders the fill styles of all cells into the canvas element.
         */
        var renderFillLayer = RenderUtils.profileMethod('rendering fill layer', function (clip, range) {

            // render cell background areas into the canvas
            cellCanvas.render(clip, function (context) {

                // all columns with an explicit fill color, mapped by column index
                var colFills = {};
                // all rows with an explicit fill color (or transparent), mapped by row index
                var rowFills = {};
                // all cells that need to be cleared (in order to delete default column/row formatting)
                var clearCells = [];
                // all cells that need to be filled
                var fillCells = [];

                // paints or clears a rectangle in the canvas (also fills the trailing pixel of the preceding left/top cells)
                function drawRect(left, top, width, height, style) {
                    if (style) {
                        context.setFillStyle(style);
                        context.drawRect(left - GRID_LINE_WIDTH, top - GRID_LINE_WIDTH, width + GRID_LINE_WIDTH, height + GRID_LINE_WIDTH, 'fill');
                    } else {
                        context.clearRect(left - GRID_LINE_WIDTH, top - GRID_LINE_WIDTH, width + GRID_LINE_WIDTH, height + GRID_LINE_WIDTH);
                    }
                }

                // paints the rectangle of the passed cell data
                function drawCellElement(element) {
                    var rect = element.innerRect;
                    var fill = element.style ? element.style.fill : null;
                    drawRect(rect.left, rect.top, rect.width + GRID_LINE_WIDTH, rect.height + GRID_LINE_WIDTH, fill);
                }

                // render entire filled columns (skip transparent columns, canvas has been cleared)
                renderCache.iterateInterval(range, true, function (element) {
                    var fill = element.style ? element.style.fill : null;
                    if (fill) {
                        drawRect(element.offset, layerRectangle.top, element.size, layerRectangle.height, fill);
                        colFills[element.index] = fill;
                    }
                });

                // render entire filled rows (but skip rows without attributes property, which do not have the 'customFormat' flag)
                renderCache.iterateInterval(range, false, function (element) {
                    if (element.style) {
                        // render transparent rows too, need to clean areas with column formatting
                        drawRect(layerRectangle.left, element.offset, layerRectangle.width, element.size, element.style.fill);
                        rowFills[element.index] = element.style.fill;
                    }
                });

                // Collect all cell to be filled or cleared (transparent cells may need to clear the default column/row
                // background). Collect cells to be cleared in a separate array to be able to draw them before the filled
                // cells which want to expand the filled area by 1 pixel to the left/top into their predecessors.
                renderCache.iterateCells(range, function (element) {

                    // skip undefined cells (column/row default formatting remains as is)
                    if (!element.style) { return; }

                    // target array to push the current cell data object to
                    var targetArray = element.style.fill ? fillCells : clearCells;

                    // bug 40429: always render merged ranges over default column/row formatting
                    if (element.isMergedOrigin()) {
                        targetArray.push(element);
                        return;
                    }

                    // column and row index of the current cell
                    var col = element.address[0];
                    var row = element.address[1];
                    // current default fill color from column or row (already rendered above)
                    var defaultFillStyle = (row in rowFills) ? rowFills[row] : (col in colFills) ? colFills[col] : null;

                    // Bug 37041: Improve performance by checking the default column/row fill color
                    // (do not render cells that do not change the existing fill color).
                    if (defaultFillStyle !== element.style.fill) {
                        targetArray.push(element);
                    }
                }, { origins: true });

                // first, clear all transparent cells (done before drawing all filled cells, otherwise
                // the trailing pixel of preceding filled cells would be cleared by these cells)
                clearCells.forEach(drawCellElement);
                fillCells.forEach(drawCellElement);
            });
        });

        /**
         * Renders the border styles of all cells into the canvas element.
         */
        var renderBorderLayer = RenderUtils.profileMethod('rendering border layer', function (clip, range) {

            // render cell background areas into the canvas
            cellCanvas.render(clip, function (context) {

                // current border line to be painted next (will be expanded across multiple cells before)
                var cachedBorder = null;

                // returns whether the passed border descriptors are considered to be equal for expanding
                function equalBorders(border1, border2) {
                    // never expand dashed borders to prevent moving dashes/dots during partial rendering
                    return border1.solid && border2.solid && (border1.width === border2.width) && (border1.double === border2.double) && (border1.hair === border2.hair) && (border1.color === border2.color);
                }

                // paints the border line stored in the variable 'cachedBorder', and resets it
                function paintCachedBorder() {

                    if (!cachedBorder) { return; }

                    // shortcut to the border style
                    var border = cachedBorder.border;
                    // shortcuts to start/end coordinate of the line
                    var start = cachedBorder.start;
                    var end = cachedBorder.end;
                    // sub-pixel offset to paint the line on entire canvas pixels
                    var subPxOffset = -((border.width / 2) % GRID_LINE_WIDTH);
                    // a new context path object
                    var path = context.createPath();

                    // pushes a single line to the rendering path
                    function pushLine(offset) {
                        if (cachedBorder.columns) {
                            path.pushLine(offset, start - GRID_LINE_WIDTH, offset, end);
                        } else {
                            path.pushLine(start - GRID_LINE_WIDTH, offset, end, offset);
                        }
                    }

                    // initialize context line style for the border line
                    context.setLineStyle({ style: border.color, width: border.lineWidth, pattern: border.pattern });

                    // add all border lines to the path
                    if (border.double) {
                        var leadOffset = cachedBorder.offset - Math.round((border.width - border.lineWidth) / 2 - subPxOffset) - subPxOffset;
                        pushLine(leadOffset);
                        pushLine(leadOffset + Math.round(border.width / 3 * 2));
                    } else {
                        pushLine(cachedBorder.offset + subPxOffset);
                    }

                    // draw the border, reset the 'cachedBorder' variable
                    context.drawPath(path, 'stroke');
                    cachedBorder = null;
                }

                // renders all border lines of the passed range in one direction
                function renderBorderLines(columns, leading) {

                    // property name for the border style of the current cell
                    var borderProp = columns ? (leading ? 'l' : 'r') : (leading ? 't' : 'b');
                    // property name for the opposite border style of the adjacent cell
                    var oppositeProp = columns ? (leading ? 'r' : 'l') : (leading ? 'b' : 't');

                    // visit all cells
                    renderCache.iterateCells(range, function (element) {

                        // the border style of the cell
                        var elementBorder = element.getBorder(borderProp);
                        // the next element containing the concurrent border line
                        var adjacentElement = element[borderProp];
                        // the border style of the adjacent cell
                        var adjacentBorder = adjacentElement && adjacentElement.getBorder(oppositeProp);
                        // get the stronger of the two borders
                        var border = RenderUtils.BorderDescriptor.getStrongerBorder(elementBorder, adjacentBorder);

                        // no border available: nothing to do
                        if (!border) { return; }

                        // matrix header element in drawing direction
                        var mainHeader = columns ? element.col : element.row;
                        // line coordinate for main drawing direction
                        var mainOffset = mainHeader.offset + (leading ? 0 : mainHeader.size);
                        // matrix header element in opposite direction (for start/end coordinate of the line)
                        var oppositeHeader = columns ? element.row : element.col;
                        // start/end coordinates of the line from opposite header
                        var startOffset = oppositeHeader.offset;
                        var endOffset = startOffset + oppositeHeader.size;

                        // try to expand the existing cached border (never expand dashed borders to prevent moving dashes/dots during partial rendering)
                        if (cachedBorder && (cachedBorder.columns === columns) && (cachedBorder.offset === mainOffset) && (cachedBorder.end === startOffset) && equalBorders(cachedBorder.border, border)) {
                            cachedBorder.end = endOffset;
                        } else {
                            // paint current cached border, and start a new cached border
                            paintCachedBorder();
                            cachedBorder = { offset: mainOffset, start: startOffset, end: endOffset, border: border, columns: columns };
                        }

                    }, { columns: columns, first: leading });
                }

                // paint vertical borders of all cells (first left borders of left cells, then right borders of all cells)
                renderBorderLines(true, true);
                renderBorderLines(true, false);

                // paint horizontal borders of all cells (first top borders of top cells, then bottom borders of all cells)
                renderBorderLines(false, true);
                renderBorderLines(false, false);

                // paint the remaining border
                paintCachedBorder();
            });
        });

        /**
         * Renders data bars of conditional formatting below the text layer.
         */
        var renderDataBarLayer = RenderUtils.profileMethod('rendering data bar layer', function (clip, range) {

            // sub-pixel offset to paint the axis line on entire canvas pixels
            var subPxOffset = GRID_LINE_WIDTH / 2;

            // returns the rounded pixel position inside the specified pixel interval
            function getPixelX(firstX, lastX, offset) {
                return Utils.round(firstX + (lastX - firstX) * Utils.minMax(offset, 0, 1), GRID_LINE_WIDTH);
            }

            // render data bar cells into the canvas
            cellCanvas.render(clip, function (context) {
                renderCache.iterateCells(range, function (element) {

                    // the data bar properties
                    var dataBar = element.renderProps && element.renderProps.dataBar;
                    if (!dataBar) { return; }

                    // the inner rectangle of the cell, without trailing border lines
                    var rect = element.innerRect;
                    // the first and last pixel index in X direction to be drawn (keep one pixel padding in the cell)
                    var firstX = rect.left + 1;
                    var lastX = rect.left + rect.width - 2;
                    // the rounded pixel position of the axis line
                    var axisX = getPixelX(firstX, lastX, -dataBar.min / (dataBar.max - dataBar.min));
                    // the rounded pixel position of the first/last filled bar pixel
                    var barX = getPixelX(firstX, lastX, (dataBar.num - dataBar.min) / (dataBar.max - dataBar.min));

                    // draw the axis line unless it is located on the first or last visible pixel
                    if (axisX === firstX) {
                        axisX = firstX - 1;
                    } else if (axisX === lastX) {
                        axisX = lastX + 1;
                    } else {
                        context.setLineStyle({ style: 'rgba(0,0,0,.5)', width: GRID_LINE_WIDTH, pattern: 2 });
                        context.drawLine(axisX + subPxOffset, rect.top, axisX + subPxOffset, rect.top + rect.height);
                    }

                    // calculate the left  X coordinates and absolute width according to bar direction
                    var x = Math.min(barX, axisX + 1);
                    var width = Math.abs(axisX - barX);

                    // draw the data bar if it is visible (covers at least one pixel next to the axis)
                    if (width > 0) {
                        if (dataBar.color2) {
                            context.setFillStyle(context.createLinearGradient(axisX, 0, barX, 0, [
                                { offset: 0, color: dataBar.color1 },
                                { offset: 1, color: dataBar.color2 }
                            ]));
                        } else {
                            context.setFillStyle(dataBar.color1);
                        }
                        context.drawRect(x, rect.top + 1, width, rect.height - 2, 'fill');
                    }
                });
            });
        });

        /**
         * Renders the cell content texts into the canvas element.
         */
        var renderTextLayer = RenderUtils.profileMethod('rendering text layer', function (clip, range) {

            // render text cells into the canvas
            cellCanvas.render(clip, function (context) {

                // Collect text cells separated by writing direction, to be able to render BiDi text correctly.
                // The property 'direction' of the canvas rendering context is hardly supported in real-life,
                // but as a workaround, setting the 'dir' element attribute at the canvas element will do the
                // job. In order to prevent changing that attribute per text cell (performance!), texts will be
                // rendered separated by writing direction, resulting in two DOM modifications only.
                var textDescMap = { ltr: [], rtl: [] };
                renderCache.iterateTextCells(range, function (element) {
                    textDescMap[element.text.font.direction].push(element.text);
                });

                // ensure to render the text at font base line position
                context.setFontStyle({ align: 'left', baseline: 'alphabetic' });

                // render text cells separated by writing direction
                _.each(textDescMap, function (textDescs, textDir) {

                    // set 'dir' attribute at the canvas element which will influence
                    // the writing direction of the rendering context (see above)
                    cellCanvas.getNode().attr('dir', textDir);

                    // draw the texts and additional decorations into the canvas
                    textDescs.forEach(function (textDesc) {
                        context.clip(textDesc.clip, function () {

                            // initialize all settings for canvas rendering (text will be rendered in fill mode, without outline)
                            context.setFillStyle(textDesc.fill).setFontStyle(textDesc.font);

                            // draw the text lines (or single words in justified alignment mode)
                            textDesc.lines.forEach(function (lineDesc) {
                                if (lineDesc.words) {
                                    lineDesc.words.forEach(function (wordDesc) {
                                        context.drawText(wordDesc.text, lineDesc.x + wordDesc.x, lineDesc.y, 'fill');
                                    });
                                } else {
                                    context.drawText(lineDesc.text, lineDesc.x, lineDesc.y, 'fill');
                                }
                            });

                            // render text decoration
                            if (textDesc.decoration) {
                                context.setLineStyle(textDesc.decoration.line).drawPath(textDesc.decoration.path, 'stroke');
                            }
                        });
                    });
                });
            });
        });

        /**
         * Renders the icons of an icon set from conditional formatting above
         * the text layer.
         */
        var renderIconSetLayer = RenderUtils.profileMethod('rendering icon set layer', function (clip, range) {

            // bug 48857: in ODS, icon size depends on row height, not on font size (icons always top-aligned)
            var odf = app.isODF();

            // render icon set cells into the canvas
            cellCanvas.render(clip, function (context) {
                renderCache.iterateCells(range, function (element) {

                    // the icon set properties
                    var iconSet = element.renderProps && element.renderProps.iconSet;
                    if (!iconSet) { return; }

                    // the complete style descriptor of the cell
                    var style = element.style;
                    // the inner rectangle of the cell, without trailing border lines
                    var rect = element.innerRect;
                    // leave some space to the cell borders (width of lines used in the icon)
                    var padding = IconRenderer.getLineWidth(rect.height, 1);
                    // the effective size of the available space, according to file format
                    var availSize = odf ? Math.min(rect.width, rect.height) : style.rowHeight;
                    // create a new, or reuse an existing, icon renderer
                    var renderer = IconRenderer.create(availSize - 2 * padding);

                    // calculate the vertical position (always top-aligned in ODF documents)
                    var offsetY = odf ? padding : (function () {
                        switch (style.attributes.cell.alignVert) {
                            case 'top':
                            case 'justify':
                                return padding;
                            case 'middle':
                                return Math.ceil((rect.height - renderer.size) / 2);
                            default: // bottom alignment, and fall-back for unknown alignments
                                return rect.height - padding - renderer.size;
                        }
                    }());

                    context.clip(rect, function () {
                        renderer.drawIcon(context, rect.left + padding, rect.top + offsetY, iconSet.id, iconSet.index);
                    });
                });
            });
        });

        /**
         * Renders additional debugging information for the cells.
         */
        var renderDebugLayer = Config.DEBUG ? RenderUtils.profileMethod('rendering debug highlighting layer', function (clip, range) {

            // nothing to do, if formula highlighting is not enabled
            if (!docView.isHighlightFormulasMode()) { return; }

            // render icon set cells into the canvas
            cellCanvas.render(clip, function (context) {

                function drawMarker(rect, color, text, anchor) {
                    var l = rect.left, t = rect.top;
                    context.setFillStyle(color).drawPath(context.createPath().pushPolygon(l, t, l + 8, t, l, t + 8), 'fill');
                    if (text) { context.drawText(text, l + 3, t + 3, 'fill'); }
                    if (anchor) { context.setFillStyle('white').drawCircle(l + 2.5, t + 2.5, 1.5, 'fill'); }
                }

                // the cell collection of the active sheet
                var cellCollection = docView.getCellCollection();

                // set font style for marker labels
                context.setFontStyle({ font: '8pt monospace', align: 'left', baseline: 'top' });

                // process all cells in the rendering cache
                renderCache.iterateCells(range, function (element) {

                    // visualize shared formula cells
                    if (element.model && element.model.isSharedFormula()) {
                        drawMarker(element.innerRect, '#080', String(element.model.si), element.model.isSharedAnchor());
                        return;
                    }

                    // visualize normal formula cells
                    if (element.model && element.model.isNormalFormula()) {
                        drawMarker(element.innerRect, '#00a');
                        return;
                    }

                    // resolve token descriptor for all cells covered by a matrix formula
                    var tokenDesc = cellCollection.getTokenArray(element.address, { fullMatrix: true });
                    if (tokenDesc && tokenDesc.matrixRange) {
                        var isMatrixAnchor = tokenDesc.matrixRange.startsAt(element.address);
                        drawMarker(element.innerRect, '#a00', isMatrixAnchor ? tokenDesc.matrixRange.toString() : null, isMatrixAnchor);
                    }
                });
            });
        }) : _.noop;

        /**
         * Creates the HTML mark-up for the range elements of the passed table
         * ranges.
         *
         * @param {Array<TableModel>|TableModel> tableModels
         *  An array with models of table ranges, or a single table model.
         */
        function createTableOutlineMarkup(tableModels) {

            // collect all table ranges (expand auto-filter to available content range)
            var tableRanges = RangeArray.map(tableModels, function (tableModel) {

                // the original table range, as contained in the model
                var tableRange = tableModel.getRange();
                // the data range, expanded at bottom border for auto filters
                var dataRange = tableModel.getDataRange();

                // expand resulting range to data range (dynamically expanded for auto filters)
                if (dataRange) { tableRange = tableRange.boundary(dataRange); }

                // add name of the table range
                tableRange.name = tableModel.getName();
                return tableRange;
            });

            // create the HTML mark-up for all visible table ranges
            var markup = '';
            gridPane.iterateRangesForRendering(tableRanges, function (range, rectangle) {
                markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                if (range.name) {
                    markup += ' title="' + Utils.escapeHTML(range.name) + '"';
                } else {
                    markup += ' data-autofilter="true"';
                }
                markup += '></div>';
            }, { alignToGrid: true });

            return markup;
        }

        /**
         * Creates and inserts the DOM elements for the outlines of all visible
         * table ranges.
         */
        var renderTableOutlinesDebounced = this.createDebouncedMethod(_.noop, function () {

            // get all table ranges in the active sheet
            var tableModels = docView.getTableCollection().getAllTables({ autoFilter: true });

            // create the HTML mark-up for all visible table ranges
            tableLayerNode[0].innerHTML = createTableOutlineMarkup(tableModels);
        }, { infoString: 'CellRenderer: renderTableOutlines', app: app });

        /**
         * Updates the outline node of the auto filter. The auto filter changes
         * dynamically after adding or deleting cell contents below at its
         * bottom border.
         */
        function updateAutoFilterOutline() {

            // remove the current auto filter outline node
            tableLayerNode.find('>[data-autofilter]').remove();

            // create and insert the new HTML mark-up for the auto filter
            var tableModel = docView.getTableCollection().getAutoFilter();
            if (tableModel) { tableLayerNode.append(createTableOutlineMarkup(tableModel)); }
        }

        /**
         * Returns a copy of the passed column/row interval, which will be
         * expanded to include the next available visible entry in both
         * directions.
         *
         * @param {Interval} interval
         *  The column/row interval to be expanded.
         *
         * @param {Boolean} columns
         *  Whether the passed interval is a column interval (true), or a row
         *  interval (false).
         *
         * @returns {Interval}
         *  The expanded interval. If there are no other visible columns/rows
         *  available before or after the interval, the respective index will
         *  not be changed.
         */
        function expandIntervalToNextVisible(interval, columns) {

            // the column/row collection
            var collection = columns ? docView.getColCollection() : docView.getRowCollection();
            // preceding visible column/row entry
            var prevEntry = collection.getPrevVisibleEntry(interval.first - 1);
            // following visible column/row entry
            var nextEntry = collection.getNextVisibleEntry(interval.last + 1);

            return new Interval(prevEntry ? prevEntry.index : interval.first, nextEntry ? nextEntry.index : interval.last);
        }

        /**
         * Renders the cell background styles, the grid lines, the cell border
         * lines, and the cell display texts of the passed cell range into the
         * canvas element. The grid lines will be rendered above the cell
         * background rectangles, but below the border lines. The text contents
         * will be rendered on top of everything else.
         *
         * @param {Range} range
         *  The address of the cell range to be rendered into the canvas.
         */
        var renderCellLayers = RenderUtils.profileMethod('CellRenderer.renderCellLayers()', function (range) {

            // expand the passed range by one column/row in each direction to be able
            // to paint or delete thick borders covering the pixels of adjacent cells
            var colInterval = expandIntervalToNextVisible(range.colInterval(), true);
            var rowInterval = expandIntervalToNextVisible(range.rowInterval(), false);
            var renderRange = Range.createFromIntervals(colInterval, rowInterval);

            // restrict to own layer range
            renderRange = renderRange.intersect(layerRange);
            if (!renderRange) { return; }
            RenderUtils.log('original range: ' + range);
            RenderUtils.log('render range: ' + renderRange);

            // the rectangle of the rendering range, for clipping
            var clipRect = docView.getRangeRectangle(renderRange);

            // reduce the clipping rectangle by half of the minimum cell size (which is the maximum
            // border width) in each direction to not clear the outer borders of the rendered range
            if (layerRange.start[0] < renderRange.start[0]) {
                clipRect.left += LEADING_CLIP_SIZE_REDUCTION;
                clipRect.width -= LEADING_CLIP_SIZE_REDUCTION;
            }
            if (renderRange.end[0] < layerRange.end[0]) {
                clipRect.width -= TRAILING_CLIP_SIZE_REDUCTION;
            }
            if (layerRange.start[1] < renderRange.start[1]) {
                clipRect.top += LEADING_CLIP_SIZE_REDUCTION;
                clipRect.height -= LEADING_CLIP_SIZE_REDUCTION;
            }
            if (renderRange.end[1] < layerRange.end[1]) {
                clipRect.height -= TRAILING_CLIP_SIZE_REDUCTION;
            }

            // clear the rendering area
            cellCanvas.render(function (context) {
                context.clearRect(clipRect);
            });

            // render all cell background rectangles, grid lines, border lines, and texts
            renderFillLayer(clipRect, renderRange);
            renderGridLayer(clipRect, renderRange);
            renderBorderLayer(clipRect, renderRange);
            renderDataBarLayer(clipRect, renderRange);
            renderTextLayer(clipRect, renderRange);
            renderIconSetLayer(clipRect, renderRange);
            renderDebugLayer(clipRect, renderRange);

            // update the auto filter outline range which may change after every cell change
            updateAutoFilterOutline();
        });

        /**
         * Registers the specified cell ranges to be rendered in a background
         * task.
         *
         * @param {RangeArray|Range} ranges
         *  The addresses of all cell ranges to be rendered, or the address of
         *  a single cell range.
         */
        var registerPendingRanges = (function () {

            // direct callback: register passed ranges
            function registerRanges(ranges) {
                RenderUtils.log('CellRenderer.registerPendingRanges(): (1) store pending ranges');
                pendingRanges.append(ranges);
                RenderUtils.log('pending ranges: ' + pendingRanges);
            }

            // deferred callback: renders all registered ranges into the canvas element
            function renderRanges() {

                // the boundary range of all pending ranges to be rendered
                var renderRange =  pendingRanges.boundary();

                // performance: wait for more dirty ranges in the rendering cache
                // (it will trigger an update event that causes to run this method again)
                if (!renderRange || renderCache.hasDirtyRanges(renderRange)) { return; }

                // for simplicity, render all cells in the entire bounding range
                RenderUtils.takeTime('CellRenderer.registerPendingRanges(): (2) render pending ranges', function () {
                    renderCellLayers(renderRange);
                });

                // reset the pending ranges for next rendering cycle
                pendingRanges.clear();
            }

            // debounced version of the method CellRenderer.registerPendingRanges()
            var registerPendingRangesDebounced = self.createDebouncedMethod(registerRanges, renderRanges, { infoString: 'CellRenderer: registerPendinRanges', app: app });

            // the actual method CellRenderer.registerPendingRanges() to be returned from the local scope
            function registerPendingRanges(ranges) {

                // bug 46268: render cell immediately if there are no pending ranges, and if the passed ranges
                // consist of a single cell only (e.g. after leaving simple cell edit mode)
                if (pendingRanges.empty() && (ranges.length === 1) && (ranges.first().cells() === 1)) {
                    registerRanges(ranges);
                    renderRanges();
                } else {
                    registerPendingRangesDebounced(ranges);
                }
            }

            return registerPendingRanges;
        }());

        /**
         * Handler for the 'update:matrix' events triggered from the rendering
         * cache. Renders the specified cell ranges into the cell layers.
         */
        function updateMatrixHandler(event, ranges) {
            registerPendingRanges(ranges);
        }

        /**
         * Updates the rendering layers after specific document view attributes
         * have been changed.
         */
        function changeViewAttributesHandler(event, attributes) {

            // highlight formula cells in debug mode
            if ('highlightFormulas' in attributes) {
                registerPendingRanges(layerRange);
            }
        }

        /**
         * Updates the rendering layers after specific sheet view attributes
         * have been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // visibility of cell grid has changed (grid color already handled via rendering cache)
            if ('showGrid' in attributes) {
                registerPendingRanges(layerRange);
                matrixLayerNode.toggle(attributes.showGrid);
                tableLayerNode.toggle(attributes.showGrid);
            }
        }

        // protected methods (renderer API) -----------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('CellRenderer.setLayerRange()', function (newLayerRange, newLayerRectangle, dirtyRanges) {

            // the model of the active sheet
            var sheetModel = docView.getSheetModel();
            // current zoom factor of active sheet
            var newSheetZoom = sheetModel.getEffectiveZoom();
            // the exact position of the new empty space in the rendering layer
            var emptyRectangles = null;
            // the new ranges that became visible in the rendering layer
            var renderRanges = null;

            // initialize canvas element (automatic scrolling of retained contents)
            RenderUtils.takeTime('initialize canvas element', function () {

                // render entire rectangle, if rendering layer was hidden before
                if (!layerRectangle) {

                    cellCanvas.initialize(newLayerRectangle, RenderUtils.CANVAS_RESOLUTION);
                    emptyRectangles = [newLayerRectangle];
                    renderRanges = new RangeArray(newLayerRange);

                } else {

                    // the zoom scaling factor, according to old and new zoom factor
                    var scale =  newSheetZoom / sheetZoom;

                    // 'scroll' canvas element to the new layer rectangle
                    emptyRectangles = cellCanvas.relocate(newLayerRectangle, scale);

                    // get cell ranges to be rendered from empty rectangles (layer rectangle does not match range boundaries)
                    renderRanges = RangeArray.map(emptyRectangles, function (rectangle) {
                        return sheetModel.getRangeFromRectangle(rectangle, { pixel: true });
                    });

                    // add the passed dirty ranges
                    renderRanges.append(dirtyRanges);

                    // Bug 38311: Clear canvas areas that are outside the sheet limits. Parent
                    // container node is in state overflow:visible in order to show elements
                    // that are party outside the sheet area, e.g. selection range elements
                    // with their tracking handles. If the lower or right boundary of the sheet
                    // is visible, the canvas may contain some rendering artifacts that will
                    // not be overdrawn or cleared later.
                    var sheetWidth = sheetModel.getColCollection().getTotalSize();
                    var sheetHeight = sheetModel.getRowCollection().getTotalSize();
                    var oversizeWidth = newLayerRectangle.left + newLayerRectangle.width - sheetWidth;
                    var oversizeHeight = newLayerRectangle.top + newLayerRectangle.height - sheetHeight;
                    if ((oversizeWidth > 0) || (oversizeHeight > 0)) {
                        cellCanvas.render(function (context) {
                            if (oversizeWidth > 0) {
                                context.clearRect(sheetWidth, newLayerRectangle.top, oversizeWidth, newLayerRectangle.height);
                            }
                            if (oversizeHeight > 0) {
                                context.clearRect(newLayerRectangle.left, sheetHeight, newLayerRectangle.width, oversizeHeight);
                            }
                        });
                    }
                }
            });

            // store new layer range and rectangle for convenience
            layerRange = newLayerRange;
            layerRectangle = newLayerRectangle;
            sheetZoom = newSheetZoom;

            // immediately render the grid lines to reduce the 'empty space' effect when scrolling quickly
            emptyRectangles.forEach(function (clipRect, index) {
                renderGridLayer(clipRect, renderRanges[index]);
            });

            // register the new ranges for debounced rendering (needed to render missing parts of existing
            // cells at the old layer boundaries which will not be notified from the rendering cache)
            registerPendingRanges(renderRanges);

            // repaint all table range outlines
            renderTableOutlinesDebounced();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRange = layerRectangle = null;
            // bugfix #47674: Performance problem when switching sheets
            // cellCanvas.initialize({ width: 0, height: 0 });
            matrixLayerNode.empty();
            tableLayerNode.empty();
            pendingRanges.clear();
        };

        // initialization -----------------------------------------------------

        // insert the canvas element into the DOM
        cellCanvas.getNode().insertBefore(matrixLayerNode);

        // start rendering after any cell elements in the rendering cache have changed
        gridPane.listenToWhenVisible(this, docView.getRenderCache(), 'update:matrix', updateMatrixHandler);

        // update cell layer after changing document/sheet view attributes
        gridPane.listenToWhenVisible(this, docView, 'change:viewattributes', changeViewAttributesHandler);
        gridPane.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // render table range outlines
        gridPane.listenToWhenVisible(this, docView, 'insert:table delete:table change:table', renderTableOutlinesDebounced);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            cellCanvas.destroy();
            self = gridPane = docModel = docView = renderCache = null;
            cellCanvas = matrixLayerNode = tableLayerNode = pendingRanges = null;
        });

    } // class CellRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CellRenderer });

});
