/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/headercontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Button, ContextMenu, Labels, gt) {

    'use strict';

    // class HeaderContextMenu ================================================

    /**
     * A context menu for header panes. Provides menu actions to manipulate
     * entire columns/rows in the active sheet.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {HeaderPane} headerPane
     *  The header pane that contains this instance.
     *
     * @param {Boolean} columns
     *  Whether this context menu is for columns (true), or for rows (false).
     */
    function HeaderContextMenu(headerPane, columns) {

        var // self reference
            self = this,

            // the document model and view
            docView = headerPane.getDocView(),
            docModel = docView.getDocModel();

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, headerPane.getNode(), {
            selector: '.cell', // only on cell header nodes (not on resizers)
            // do not show context menu at all in locked sheets, or during cell edit mode
            enableKey: 'sheet/operation/unlocked/cell',
            delay: 200
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts column-specific menu entries into the context menu.
         */
        function initColumnMenu() {
            var selRanges = docView.getSelectedRanges(),
                multiCols = (selRanges.colIntervals().size() > 1);

            self.destroyAllGroups();

            self.addGroup('column/insert', new Button((multiCols ? Labels.INSERT_COLUMNS_OPTIONS : Labels.INSERT_COLUMN_OPTIONS)))
                .addGroup('column/delete', new Button((multiCols ? Labels.DELETE_COLUMNS_OPTIONS : Labels.DELETE_COLUMN_OPTIONS)))
                .addSeparator()
                .addGroup('column/width/optimal', new Button({ label: gt('Set optimal column width') }))
                .addGroup('column/hide', new Button({ label: (multiCols ? gt('Hide columns') : gt('Hide column')) }))
                .addGroup('column/show', new Button({ label: gt('Show hidden columns') }));

            self.updateAllGroups();
        }

        /**
         * Inserts row-specific menu entries into the context menu.
         */
        function initRowMenu() {
            var selRanges = docView.getSelectedRanges(),
                multiRows = (selRanges.rowIntervals().size() > 1);

            self.destroyAllGroups();

            self.addGroup('row/insert', new Button((multiRows ? Labels.INSERT_ROWS_OPTIONS : Labels.INSERT_ROW_OPTIONS)))
                .addGroup('row/delete', new Button((multiRows ? Labels.DELETE_ROWS_OPTIONS : Labels.DELETE_ROW_OPTIONS)))
                .addSeparator()
                .addGroup('row/height/optimal', new Button({ label: gt('Set optimal row height') }))
                .addGroup('row/hide', new Button({ label: (multiRows ? gt('Hide rows') : gt('Hide row')) }))
                .addGroup('row/show', new Button({ label: gt('Show hidden rows') }));

            self.updateAllGroups();
        }

        /**
         * Selects the column/row that has been clicked with the right mouse
         * button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            var // the source event that caused to open the context menu
                sourceEvent = event.sourceEvent,
                // the index of the column/row to be selected
                index = Utils.getElementAttributeAsInteger(sourceEvent.target, 'data-index', -1);

            // functor to check whether the column/row index is contained in a range address
            function rangeContainsIndex(range) {
                return docModel.isFullRange(range, columns) && range.containsIndex(index, columns);
            }

            // Do nothing if no valid cell node has been clicked
            if ((index < 0)) {
                event.preventDefault();
                return;
            }

            // select the clicked column/row if it is not contained in the selection
            if (!docView.getSelectedRanges().some(rangeContainsIndex)) {
                docView.grabFocus();
                headerPane.trigger('select:start', index, 'select');
                headerPane.trigger('select:end', index, true);
            }
        }

        // public methods ----
        this.getXY = function (event) {
            if (event.sourceEvent.pageX && event.sourceEvent.pageY) {
                return {
                    pageX: event.sourceEvent.pageX,
                    pageY: event.sourceEvent.pageY
                };
            } else {
                var xy = $(event.sourceEvent.target).offset();
                return {
                    pageX: (xy.left + ($(event.sourceEvent.target).width() / 2)),
                    pageY: (xy.top + ($(event.sourceEvent.target).height() / 2))
                };
            }
        };

        // initialization -----------------------------------------------------

        // lazy initialization of the menu items according to the orientation of the header pane
        this.on('popup:beforeshow', columns ? initColumnMenu : initRowMenu);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = headerPane = docModel = docView = null;
        });

    } // class HeaderContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: HeaderContextMenu });

});
