/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/drawingcontextmenu', [
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/view/labels'
], function (Button, ContextMenu, DrawingFrame, Labels) {

    'use strict';

    // class DrawingContextMenu ================================================

    /**
     * A context menu for drawing objects. Provides menu actions to manipulate
     * the selected drawing objects in the active sheet.
     *
     * @constructor
     *
     * @extends ContextMenu
     */
    function DrawingContextMenu(gridPane, drawingLayerNode) {

        var // self reference
            self = this,

            // the spreadsheet view
            docView = gridPane.getDocView();

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, drawingLayerNode, {
            enableKey: 'drawing/operation', // show context menu for editable drawings only
            delay: 200,
            lazyInitHandler: initMenu
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts all menu entries into this context menu.
         */
        function initMenu() {
            self.addGroup('drawing/delete', new Button({ label: Labels.DELETE_DRAWING_LABEL, tooltip: Labels.DELETE_DRAWING_TOOLTIP }));
        }

        /**
         * Selects the drawing object that has been clicked with the right
         * mouse button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {
            var // the source event that caused to open the context menu
                sourceEvent = event.sourceEvent,
                drawing = DrawingFrame.getDrawingNode(sourceEvent.target),
                drawingPos = null;

            drawingPos = [drawing.index()];
            docView.selectDrawing(drawingPos);
        }

        // public methods ----
        this.getXY = function (event) {
            if (event.pageX && event.pageY) {
                return {
                    pageX: event.pageX,
                    pageY: event.pageY
                };
            } else {
                var xy = $(event.sourceEvent.target).offset();
                return {
                    pageX: (xy.left + ($(event.sourceEvent.target).width() / 2)),
                    pageY: (xy.top + ($(event.sourceEvent.target).height() / 2))
                };
            }
        };

        // initialization -----------------------------------------------------

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = drawingLayerNode = null;
        });

    } // class DrawingContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: DrawingContextMenu });

});
