/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/cornerpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, TriggerObject, TimerMixin, SheetUtils) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address;

    // class CornerPane =======================================================

    /**
     * Represents the top-left corner in the spreadsheet view.
     *
     * Triggers the following events:
     * - 'select:all'
     *      After this corner pane has been clicked which will select the
     *      entire sheet. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys. This
     *              should move the current active cell in the selection to the
     *              first visible cell in the selection.
     *          - 'extend': Extend current active range (SHIFT). This should
     *              keep the current active cell in the selection unmodified.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this corner pane.
     */
    function CornerPane(docView) {

        var // self reference
            self = this,

            // the container node of this corner pane
            rootNode = $('<div class="corner-pane">'),

            // the cell DOM node used to calculate the size
            cellNode = $('<div class="cell noI18n">');

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Handles mouse clicks and selects the entire sheet.
         */
        function clickHandler(event) {

            if ((event.type === 'touchstart') || (event.button === 0)) {
                self.trigger('select:all', event.shiftKey ? 'extend' : 'select');
            }

            // return focus to active grid pane, after the event has been processed
            self.executeDelayed(function () { docView.grabFocus(); }, { infoString: 'CornerPane: clickHandler', app: docView.getApp() });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the document view that contains this corner pane.
         *
         * @returns {SpreadsheetView}
         *  The document view that contains this corner pane.
         */
        this.getDocView = function () {
            return docView;
        };

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Initializes the size of this corner pane.
         *
         * @param {Number} maxRow
         *  The maximum index of the row to be displayed in row header panes.
         *
         * @returns {CornerPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (maxRow) {

            var // the exact node size (fractional pixels)
                nodeSize = null;

            // prepare the cell node text and font size
            cellNode
                .text(Address.stringifyRow(maxRow))
                .css({ fontSize: docView.getHeaderFontSize() + 'pt', lineHeight: 'normal' })
                .appendTo(rootNode);

            // copy size of cell node to root node, and remove the cell node
            nodeSize = Utils.getCeilNodeSize(cellNode);
            rootNode.css({ width: nodeSize.width + 2, height: nodeSize.height + 2 });
            cellNode.remove();

            return this;
        };

        // initialization -----------------------------------------------------

        // immediately react on mousedown (do not wait for click)
        rootNode.on('mousedown touchstart', clickHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = rootNode = cellNode = null;
        });

    } // class CornerPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CornerPane });

});
