/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/font',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/range3d',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/range3darray',
    'io.ox/office/spreadsheet/utils/addressset',
    'io.ox/office/spreadsheet/utils/intervalset',
    'io.ox/office/spreadsheet/utils/rangeset',
    'io.ox/office/spreadsheet/utils/changedescriptor',
    'io.ox/office/spreadsheet/utils/movedescriptor',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Logger, LocaleData, Font, ErrorCode, Address, Interval, Range, Range3D, AddressArray, IntervalArray, RangeArray, Range3DArray, AddressSet, IntervalSet, RangeSet, ChangeDescriptor, MoveDescriptor, gt) {

    'use strict';

    // maps single-character border keys to the attribute names of cell borders
    var BORDER_ATTRIBUTE_NAMES = {
        l: 'borderLeft',
        r: 'borderRight',
        t: 'borderTop',
        b: 'borderBottom',
        d: 'borderDown',
        u: 'borderUp',
        v: 'borderInsideVert',
        h: 'borderInsideHor'
    };

    // static class SheetUtils ================================================

    /**
     * The static class SheetUtils contains various helper methods for cell
     * addresses, cell range addresses, and column/row index intervals.
     * Additionally, the class is a console logger bound to the URL hash flag
     * 'spreadsheet:log-models', that logs detailed information about the
     * document model instances and collections.
     */
    var SheetUtils = {};

    // logger interface -------------------------------------------------------

    Logger.extend(SheetUtils, { enable: 'spreadsheet:log-models', prefix: 'MODEL' });

    // helper classes ---------------------------------------------------------

    // export address/interval/range helper classes for convenience
    SheetUtils.ErrorCode = ErrorCode;
    SheetUtils.Address = Address;
    SheetUtils.Interval = Interval;
    SheetUtils.Range = Range;
    SheetUtils.Range3D = Range3D;
    SheetUtils.AddressArray = AddressArray;
    SheetUtils.IntervalArray = IntervalArray;
    SheetUtils.RangeArray = RangeArray;
    SheetUtils.Range3DArray = Range3DArray;
    SheetUtils.AddressSet = AddressSet;
    SheetUtils.IntervalSet = IntervalSet;
    SheetUtils.RangeSet = RangeSet;
    SheetUtils.ChangeDescriptor = ChangeDescriptor;
    SheetUtils.MoveDescriptor = MoveDescriptor;

    // constants --------------------------------------------------------------

    /**
     * Maximum length of a defined name.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_NAME_LENGTH = 255;

    /**
     * The maximum number of cells to be filled with one range operation, or
     * auto-fill operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_FILL_CELL_COUNT = 20000;

    /**
     * The maximum number of entire columns/rows to be filled with one
     * auto-fill operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT = 100;

    /**
     * Maximum number of merged ranges to be created with a single operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_MERGED_RANGES_COUNT = 1000;

    /**
     * The maximum number of cells to be filled when unmerging a range.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_UNMERGE_CELL_COUNT = 5000;

    /**
     * The maximum number of columns to be changed with one row operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_CHANGE_COLS_COUNT = 5000;

    /**
     * The maximum number of rows to be changed with one row operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_CHANGE_ROWS_COUNT = 5000;

    /**
     * The maximum number of columns/rows that can be sorted.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_SORT_LINES_COUNT = 5000;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the standard number format (in cells).
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_LENGTH_STANDARD_CELL = 11;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the standard number format (in cell edit mode).
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_LENGTH_STANDARD_EDIT = 21;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the standard number format (in formula interpretation, e.g. when
     * converting numbers to strings for functions expecting strings).
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_LENGTH_STANDARD_FORMULA = 20;

    /**
     * Minimum effective size of a cell, in pixels. Also used as maximum width
     * of cell border lines, to prevent that the left/right borders or the top/
     * bottom borders of a cell overlay each other.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MIN_CELL_SIZE = 5;

    /**
     * Minimum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MIN_ZOOM = 0.5;

    /**
     * Maximum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_ZOOM = 8;

    /**
     * Whether multi-selection (multiple ranges at the same time) is supported.
     *
     * @constant
     * @type Boolean
     */
    SheetUtils.MULTI_SELECTION = !Utils.TOUCHDEVICE;

    // public methods ---------------------------------------------------------

    /**
     * Convenience shortcut that returns a rejected promise with a specific
     * message code as object with 'cause' property.
     *
     * @param {String} msgCode
     *  The message code the returned promise will be rejected with.
     *
     * @returns {jQuery.Promise}
     *  A promise that has already been rejected with an object containing
     *  a string property 'cause' set to the passed message code.
     */
    SheetUtils.makeRejected = function (msgCode) {
        return $.Deferred().reject({ cause: msgCode });
    };

    /**
     * Returns a unique map key for the passed label of a defined name.
     *
     * @param {String} label
     *  The original label of a defined name.
     *
     * @returns {String}
     *  A unique map key for the passed label of a defined name.
     */
    SheetUtils.getNameKey = function (label) {
        return label.toUpperCase();
    };

    /**
     * Returns a unique map key for the passed table name. Can also be used for
     * the names of table columns.
     *
     * @param {String} tableName
     *  The original name of a table range, or a table column.
     *
     * @returns {String}
     *  A unique map key for the passed table name, or table column name.
     */
    SheetUtils.getTableKey = function (tableName) {
        return tableName.toUpperCase();
    };

    /**
     * Returns the localized sheet name with the specified index, ready to be
     * used in the document model (without any I18N marker characters used for
     * UI debugging).
     *
     * @param {Number} index
     *  The one-based (!) sheet index to be inserted into the sheet name.
     *
     * @returns {String}
     *  The generated sheet name (e.g. 'Sheet1', 'Sheet2', etc.).
     */
    SheetUtils.getSheetName = function (index) {
        //#. Default sheet names in spreadsheet documents. Should be equal to the sheet
        //#. names used in existing spreadsheet applications (Excel, OpenOffice Calc, ...).
        //#. %1$d is the numeric index inside the sheet name
        //#. Resulting names e.g. "Sheet1", "Sheet2", etc.
        //#. No white-space between the "Sheet" label and sheet index!
        //#, c-format
        return _.noI18n.fix(gt('Sheet%1$d', _.noI18n(index)));
    };

    /**
     * Returns the translated word for 'Table' to be used for new table ranges,
     * ready to be used in the document model (without any I18N marker
     * characters used for UI debugging).
     *
     * @returns {String}
     *  The translated word for 'Table'.
     */
    SheetUtils.getTableName = function () {
        //#. Default name for table ranges (cell ranges in a sheet with filter settings) in spreadsheet
        //#. documents. Will be used to create serially numbered tables, e.g. "Table1", "Table2", etc.
        return _.noI18n.fix(gt.pgettext('filter', 'Table'));
    };

    /**
     * Returns the translated word for 'Column', ready to be used in the
     * document model (without any I18N marker characters used for UI
     * debugging).
     *
     * @returns {String}
     *  The translated word for 'Column'.
     */
    SheetUtils.getTableColName = function () {
        //#. Default name for columns in filtered ranges in spreadsheet documents. Will be
        //#. used to create serially numbered columns, e.g. "Column1", "Column2", etc.
        return _.noI18n.fix(gt.pgettext('filter', 'Column'));
    };

    /**
     * Returns whether the passed merged attribute set will force to split the
     * text contents of a cell into multiple lines, either by the cell
     * attribute 'wrapText' set to true, or by horizontally or vertically
     * justified alignment.
     *
     * @param {Object} attributeSet
     *  A merged (complete) cell attribute set.
     *
     * @returns {Boolean}
     *  Whether the attribute set will force to split the text contents of a
     *  cell into multiple lines.
     */
    SheetUtils.hasWrappingAttributes = function (attributeSet) {
        var cellAttrs = attributeSet.cell;
        return cellAttrs.wrapText || (cellAttrs.alignHor === 'justify') || (cellAttrs.alignVert === 'justify');
    };

    /**
     * Returns the effective text orientation settings for the passed cell
     * value, formatted display string, and horizontal alignment.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  A scalar cell value. Influences the behavior of the 'auto' alignment.
     *  The value null represents a blank cell.
     *
     * @param {String} align
     *  The horizontal alignment, as used in the cell attribute 'alignHor'. The
     *  value 'auto' will cause special behavior based on the passed cell value
     *  and display string.
     *
     * @param {String} [display]
     *  The formatted display string for the passed cell value. Can be omitted,
     *  or set to null, to return the text orientation based on the cell value
     *  only.
     *
     * @returns {Object}
     *  A descriptor containing various text orientation properties:
     *  - {String} textDir
     *      The effective writing direction in the single text lines of the
     *      display text, as value supported by the HTML element attribute
     *      'dir' (either 'ltr' or 'rtl').
     *  - {String} cssTextAlign
     *      The effective horizontal CSS text alignment, as value supported by
     *      the CSS property 'text-align'. For automatic alignment (parameter
     *      'align' is set to 'auto'), the resulting text alignment depends on
     *      the data type of the passed value. Numbers will be aligned to the
     *      right, boolean values and error codes will be centered, and strings
     *      will be aligned according to their own writing direction (NOT the
     *      passed display text which may differ from the string, e.g. by using
     *      a specific number format). Will be one of the following value:
     *      - 'left': The text lines are aligned to the left in their bounding
     *          box.
     *      - 'center': The text lines are centered in their bounding box.
     *      - 'right': The text lines are aligned to the right in their
     *          bounding box.
     *      - 'justify': The text lines will be expanded to the width of the
     *          bounding box. The alignment of a single text line, or of the
     *          the last text line in multi-line cells is dependent on the
     *          default writing direction of the text.
     *  - {String} baseBoxAlign
     *      The resolved horizontal alignment of the single text lines inside
     *      their bounding box. Will be equal to the property 'cssTextAlign'
     *      except for justified alignment (property cssTextAlign contains the
     *      value 'justify') which will be resolved to 'left' or 'right',
     *      depending on the default writing direction of the text value (NOT
     *      the display string).
     */
    SheetUtils.getTextOrientation = function (value, align, display) {

        // converts the passed text direction to left/right alignment
        function getAlignFromDir(dir) {
            return (dir === 'rtl') ? 'right' : 'left';
        }

        // the effective writing direction of the display text
        var displayDir = _.isString(display) ? Font.getTextDirection(display) : LocaleData.DIR;
        // the effective CSS text alignment
        var cssTextAlign = null;
        // base alignment of the text lines in the cell box
        var baseBoxAlign = null;

        switch (align) {
            case 'left':
            case 'center':
            case 'right':
            case 'justify':
                cssTextAlign = align;
                break;

            default: // automatic alignment

                // strings (depends on writing direction of the text result, not display string!)
                if (_.isString(value)) {
                    cssTextAlign = getAlignFromDir((value === display) ? displayDir : Font.getTextDirection(value));

                // numbers (always right aligned, independent from display string)
                } else if (_.isNumber(value)) {
                    cssTextAlign = 'right';

                // blank cells (depending on system writing direction)
                } else if (_.isNull(value)) {
                    cssTextAlign = getAlignFromDir(Font.DEFAULT_TEXT_DIRECTION);

                // booleans and error codes (always centered)
                } else {
                    cssTextAlign = 'center';
                }
        }

        // base box alignment in justified cells depends on writing direction of display string
        baseBoxAlign = (cssTextAlign === 'justify') ? getAlignFromDir(displayDir) : cssTextAlign;

        // return the orientation descriptor object
        return { textDir: displayDir, cssTextAlign: cssTextAlign, baseBoxAlign: baseBoxAlign };
    };

    /**
     * Returns a single-character key for an outer cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the key of a vertical border (true), or of a
     *  horizontal border (false).
     *
     * @param {Boolean} leading
     *  Whether to return the key of a leading border (true), or of a trailing
     *  border (false).
     *
     * @returns {String}
     *  One of the border keys 't', 'b', 'l', or 'r'.
     */
    SheetUtils.getOuterBorderKey = function (columns, leading) {
        return columns ? (leading ? 'l' : 'r') : (leading ? 't' : 'b');
    };

    /**
     * Returns a single-character key for an inner cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the key of the vertical inner border (true), or of
     *  the horizontal inner border (false).
     *
     * @returns {String}
     *  One of the border keys 'h', or 'v'.
     */
    SheetUtils.getInnerBorderKey = function (columns) {
        return columns ? 'v' : 'h';
    };

    /**
     * Returns the attribute name of a cell border.
     *
     * @param {String} key
     *  The single-character key of a cell border. MUST be one of 't', 'b',
     *  'l', 'r', 'd', 'u', 'h', or 'v'.
     *
     * @returns {String}
     *  The name of the specified border attribute.
     */
    SheetUtils.getBorderName = function (key) {
        return BORDER_ATTRIBUTE_NAMES[key];
    };

    /**
     * Returns the attribute name of an outer cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the name of a vertical border attribute (true), or of
     *  a horizontal border attribute (false).
     *
     * @param {Boolean} leading
     *  Whether to return the name of a leading border attribute (true), or of
     *  a trailing border attribute (false).
     *
     * @returns {String}
     *  One of the border attribute names 'borderTop', 'borderBottom',
     *  'borderLeft', or 'borderRight'.
     */
    SheetUtils.getOuterBorderName = function (columns, leading) {
        return BORDER_ATTRIBUTE_NAMES[SheetUtils.getOuterBorderKey(columns, leading)];
    };

    /**
     * Returns the attribute name of an inner cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the name of the vertical inner border attribute
     *  (true), or of the horizontal inner border attribute (false).
     *
     * @returns {String}
     *  One of the border attribute names 'borderInsideHor', or
     *  'borderInsideVert'.
     */
    SheetUtils.getInnerBorderName = function (columns) {
        return BORDER_ATTRIBUTE_NAMES[SheetUtils.getInnerBorderKey(columns)];
    };

    /**
     * Returns the effective horizontal padding between cell grid lines and the
     * text contents of the cell, according to the passed zoom factor.
     *
     * @param {Number} zoom
     *  The zoom factor to calculate the cell content padding for.
     *
     * @returns {Number}
     *  The effective horizontal cell content padding, in pixels.
     */
    SheetUtils.getTextPadding = function (zoom) {
        // 2 pixels padding for 100% zoom, rounded up early
        return Math.floor(2 * zoom + 0.75);
    };

    /**
     * Returns the total size of all horizontal padding occupied in a cell that
     * cannot be used for the cell contents, according to the passed zoom
     * factor. This value includes the text padding (twice, for left and right
     * border), and additional space needed for the grid lines.
     *
     * @param {Number} zoom
     *  The zoom factor to calculate the cell content padding for.
     *
     * @returns {Number}
     *  The total size of the horizontal cell content padding, in pixels.
     */
    SheetUtils.getTotalCellPadding = function (zoom) {
        // padding at left and right cell border, and 1 pixel for the grid line
        return 2 * SheetUtils.getTextPadding(zoom) + 1;
    };

    /**
     * Returns whether the passed split mode identifier includes the dynamic
     * split mode. Returns also true, if the split mode is 'frozenSplit', i.e.
     * the effective split mode is frozen but will thaw to dynamic split mode.
     *
     * @param {String} splitMode
     *  A split mode identifier (one of 'split', 'frozen', or 'frozenSplit').
     *
     * @returns {Boolean}
     *  Whether the passed split mode identifier includes the dynamic split
     *  mode (either 'split' or 'frozenSplit').
     */
    SheetUtils.isDynamicSplit = function (splitMode) {
        return /^(split|frozenSplit)$/.test(splitMode);
    };

    /**
     * Returns whether the passed split mode identifier includes the frozen
     * split mode.
     *
     * @param {String} splitMode
     *  A split mode identifier (one of 'split', 'frozen', or 'frozenSplit').
     *
     * @returns {Boolean}
     *  Whether the passed split mode identifier includes the frozen split
     *  mode (either 'frozen' or 'frozenSplit').
     */
    SheetUtils.isFrozenSplit = function (splitMode) {
        return /^(frozen|frozenSplit)$/.test(splitMode);
    };

    // exports ================================================================

    return SheetUtils;

});
