/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/addressarray', [
    'io.ox/office/tk/container/typedarray',
    'io.ox/office/spreadsheet/utils/address'
], function (TypedArray, Address) {

    'use strict';

    // class AddressArray =====================================================

    /**
     * Represents an array of cell addresses. The array elements are instances
     * of the class Address.
     *
     * @constructor
     *
     * @extends Array
     */
    var AddressArray = TypedArray.create(Address);

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for this array of cell addresses that can be
     * used for example as key in an associative map. This method is faster
     * than the method AddressArray.toString().
     *
     * @returns {String}
     *  A unique string key for this instance.
     */
    AddressArray.prototype.key = function () {
        return this.map(function (address) { return address.key(); }).join(' ');
    };

    /**
     * Returns whether a cell address in this array is equal to the passed cell
     * address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether this array contains the passed cell address.
     */
    AddressArray.prototype.contains = function (address) {
        return this.some(function (a) { return a.equals(address); });
    };

    /**
     * Returns a shallow copy of this cell address array that does not contain
     * any duplicate cell addresses.
     *
     * @returns {AddressArray}
     *  A copy of this cell address array without any duplicates.
     */
    AddressArray.prototype.unify = function () {
        var result = new AddressArray(), map = {};
        this.forEach(function (address) {
            var key = address.key();
            if (!map[key]) {
                result.push(address);
                map[key] = true;
            }
        });
        return result;
    };

    // exports ================================================================

    return AddressArray;

});
