/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/dimension', function () {

    'use strict';

    // class Dimension ========================================================

    /**
     * Representation the dimension of a formula operand, i.e. the number of
     * rows and columns of a matrix or cell range.
     *
     * @constructor
     *
     * @property {Number} rows
     *  The number of rows of the operand (positive integer).
     *
     * @property {Number} cols
     *  The number of columns of the operand (positive integer).
     */
    function Dimension(rows, cols) {

        this.rows = rows;
        this.cols = cols;

    } // class Dimensions

    // static methods ---------------------------------------------------------

    /**
     * Returns the dimension of the passed cell range address.
     *
     * @param {Range|Range3D} range
     *  A cell range address.
     *
     * @returns {Dimension}
     *  The dimension of the passed cell range address.
     */
    Dimension.createFromRange = function (range) {
        return new Dimension(range.rows(), range.cols());
    };

    /**
     * Creates a new dimension instance representing the bounding size (the
     * maximum of the row numbers and column numbers of the passed dimensions).
     *
     * @param {Dimension|Null} dim1
     *  The first dimension instance. If omitted or set to null, the second
     *  passed dimension will be returned.
     *
     * @param {Dimension|Null} dim2
     *  The second dimension instance. If omitted or set to null, the first
     *  passed dimension will be returned.
     *
     * @returns {Dimension|Null}
     *  The resulting boundary dimension. If both parameters are set to null,
     *  this method will return null.
     */
    Dimension.boundary = function (dim1, dim2) {
        if (!dim1) { return dim2 || null; }
        if (!dim2) { return dim1 || null; }
        return new Dimension(Math.max(dim1.rows, dim2.rows), Math.max(dim1.cols, dim2.cols));
    };

    // public method ----------------------------------------------------------

    /**
     * Returns a clone of this instance.
     *
     * @returns {Dimension}
     *  A clone of this instance.
     */
    Dimension.prototype.clone = function () {
        return new Dimension(this.rows, this.cols);
    };

    /**
     * Returns whether the dimension represented by this instance and the
     * passed dimension are equal.
     *
     * @param {Dimension} dim
     *  The other dimension to compare with this instance.
     *
     * @returns {Boolean}
     *  Whether this dimension and the passed dimension are equal.
     */
    Dimension.prototype.equals = function (dim) {
        return (this.rows === dim.rows) && (this.cols === dim.cols);
    };

    /**
     * Returns the number of elements contained in an operand of the current
     * dimension.
     *
     * @returns {Number}
     *  The number of elements represented by this instance.
     */
    Dimension.prototype.size = function () {
        return this.rows * this.cols;
    };

    /**
     * Returns the string representation of this instance for debug logging.
     *
     * @returns {String}
     *  The string representation of this instance for debug logging.
     */
    Dimension.prototype.toString = function () {
        return this.rows + 'x' + this.cols;
    };

    // exports ================================================================

    return Dimension;

});
