/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/cellref', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;

    // private global functions ===============================================

    /**
     * Returns a relocated column/row index.
     *
     * @param {Number} index
     *  The original column/row index.
     *
     * @param {Number} offset
     *  The distance to move the passed column/row index.
     *
     * @param {Number} maxIndex
     *  The maximum allowed column/row index in a cell address.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the column/row indexes at the sheet boundaries.
     *
     * @returns {Number}
     *  The resulting column/row index; or -1, if the passed column/row index
     *  cannot be relocated.
     */
    function relocateIndex(index, offset, maxIndex, wrapReferences) {
        if (Math.abs(offset) > maxIndex) { return -1; }
        index += offset;
        return wrapReferences ?
            ((index < 0) ? (index + maxIndex + 1) : (index > maxIndex) ? (index - maxIndex - 1) : index) :
            (((index < 0) || (index > maxIndex)) ? -1 : index);
    }

    // class CellRef ==========================================================

    /**
     * A structure representing the address of a single cell in a reference
     * token (e.g. the particles '$A$1' or '$B$2' in Sheet1!$A$1:$B$2).
     *
     * @constructor
     *
     * @property {Number} col
     *  The column index.
     *
     * @property {Number} row
     *  The row index.
     *
     * @property {Boolean} absCol
     *  Whether the column reference is absolute (as in $C2).
     *
     * @property {Boolean} absRow
     *  Whether the row reference is absolute (as in C$2).
     */
    function CellRef(col, row, absCol, absRow) {

        this.col = col;
        this.row = row;
        this.absCol = absCol;
        this.absRow = absRow;

    } // class CellRef

    // static methods ---------------------------------------------------------

    /**
     * Creates a cell reference structure for the passed cell address.
     *
     * @param {Address} address
     *  The cell address.
     *
     * @property {Boolean} absCol
     *  Whether the column reference is absolute (as in $C2).
     *
     * @property {Boolean} absRow
     *  Whether the row reference is absolute (as in C$2).
     */
    CellRef.create = function (address, absCol, absRow) {
        return new CellRef(address[0], address[1], absCol, absRow);
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a deep clone of this cell reference structure.
     *
     * @returns {CellRef}
     *  A deep clone of this cell reference structure.
     */
    CellRef.prototype.clone = function () {
        return new CellRef(this.col, this.row, this.absCol, this.absRow);
    };

    /**
     * Returns whether this cell reference and the passed cell reference are
     * equal.
     *
     * @param {CellRef} cellRef
     *  The other cell reference to be compared with this cell reference.
     *
     * @returns {Boolean}
     *  Whether this cell reference and the passed cell reference are equal.
     */
    CellRef.prototype.equals = function (cellRef) {
        return (this.col === cellRef.col) && (this.row === cellRef.row) && (this.absCol === cellRef.absCol) && (this.absRow === cellRef.absRow);
    };

    /**
     * Returns the cell address represented by this cell reference.
     *
     * @returns {Address}
     *  The cell address represented by this cell reference.
     */
    CellRef.prototype.toAddress = function () {
        return new Address(this.col, this.row);
    };

    /**
     * Returns whether the column or row index of this cell reference is marked
     * to be absolute.
     *
     * @param {Boolean} columns
     *  Whether to check the column index (true), or the row index (false).
     *
     * @returns {Boolean}
     *  Whether the column or row index of this cell reference is marked to be
     *  absolute.
     */
    CellRef.prototype.isAbs = function (columns) {
        return columns ? this.absCol : this.absRow;
    };

    /**
     * Returns the column or row index of this cell reference.
     *
     * @param {Boolean} columns
     *  Whether to retirn the column index (true), or the row index (false).
     *
     * @returns {Number}
     *  The column or row index of this cell reference.
     */
    CellRef.prototype.getIndex = function (columns) {
        return columns ? this.col : this.row;
    };

    /**
     * Relocates the relative column/row components in this cell reference
     * in-place.
     *
     * @param {SpreadsheetDocument} docModel
     *  The spreadsheet document used to validate the column and row indexes.
     *
     * @param {Address} refAddress
     *  The source reference address of the formula containing the token.
     *
     * @param {Address} targetAddress
     *  The target reference address for relocation.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the column/row indexes at the sheet boundaries.
     *
     * @returns {Boolean}
     *  Whether relocation was successful, and the address of this reference
     *  is still valid.
     */
    CellRef.prototype.relocate = function (docModel, refAddress, targetAddress, wrapReferences) {

        // relocate relative column
        if (!this.absCol) {
            this.col = relocateIndex(this.col, targetAddress[0] - refAddress[0], docModel.getMaxCol(), wrapReferences);
            if (this.col < 0) { return false; }
        }

        // relocate relative row
        if (!this.absRow) {
            this.row = relocateIndex(this.row, targetAddress[1] - refAddress[1], docModel.getMaxRow(), wrapReferences);
            if (this.row < 0) { return false; }
        }

        return true;
    };

    /**
     * Returns the complete name of the column part of this cell reference.
     *
     * @returns {String}
     *  The complete name of the column part of this cell reference.
     */
    CellRef.prototype.colText = function () {
        return (this.absCol ? '$' : '') + Address.stringifyCol(this.col);
    };

    /**
     * Returns the complete name of the row part of this cell reference.
     *
     * @returns {String}
     *  The complete name of the row part of this cell reference.
     */
    CellRef.prototype.rowText = function () {
        return (this.absRow ? '$' : '') + Address.stringifyRow(this.row);
    };

    /**
     * Returns the complete name of this cell reference.
     *
     * @returns {String}
     *  The complete name of this cell reference.
     */
    CellRef.prototype.refText = function () {
        return this.colText() + this.rowText();
    };

    /**
     * Returns the complete name of the column part of this cell reference, for
     * R1C1 notation.
     *
     * @param {String} prefixChars
     *  The prefix characters (two characters) used for R1C1 notation.
     *
     * @param {Address} refAddress
     *  The reference address needed to create column offsets as used in R1C1
     *  notation, e.g. R1C[-1].
     *
     * @returns {String}
     *  The complete name of the column part of this cell reference, for R1C1
     *  notation.
     */
    CellRef.prototype.colTextRC = function (prefixChars, refAddress) {
        return prefixChars[1] + (this.absCol ? (this.col + 1) : (this.col !== refAddress[0]) ? ('[' + (this.col - refAddress[0]) + ']') : '');
    };

    /**
     * Returns the complete name of the row part of this cell reference, for
     * R1C1 notation.
     *
     * @param {String} prefixChars
     *  The prefix characters (two characters) used for R1C1 notation.
     *
     * @param {Address} refAddress
     *  The reference address needed to create row offsets as used in R1C1
     *  notation, e.g. R[-1]C1.
     *
     * @returns {String}
     *  The complete name of the row part of this cell reference, for R1C1
     *  notation.
     */
    CellRef.prototype.rowTextRC = function (prefixChars, refAddress) {
        return prefixChars[0] + (this.absRow ? (this.row + 1) : (this.row !== refAddress[1]) ? ('[' + (this.row - refAddress[1]) + ']') : '');
    };

    /**
     * Returns the complete name of this cell reference, for R1C1 notation.
     *
     * @param {String} prefixChars
     *  The prefix characters (two characters) used for R1C1 notation.
     *
     * @param {Address} refAddress
     *  The reference address needed to create column or row offsets as used in
     *  R1C1 notation, e.g. R[-1]C[-1].
     *
     * @returns {String}
     *  The complete name of this cell reference.
     */
    CellRef.prototype.refTextRC = function (prefixChars, refAddress) {
        return this.rowTextRC(prefixChars, refAddress) + this.colTextRC(prefixChars, refAddress);
    };

    /**
     * Parses a column index in A1 notation, and adds it to this cell reference
     * structure.
     *
     * @param {String} colText
     *  The column name in A1 notation, e.g. 'A' for the first column.
     *
     * @param {String|Null} absFlag
     *  The absolute marker. Each non-empty string will be recognized as
     *  absolute column index. An empty string, or any other falsy value will
     *  be recognized as relative column index.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @returns {Boolean}
     *  Whether the passed values represent a valid column index.
     */
    CellRef.prototype.parseCol = function (colText, absFlag, maxCol) {
        this.col = Address.parseCol(colText);
        this.absCol = !!absFlag;
        return (this.col >= 0) && (this.col <= maxCol);
    };

    /**
     * Parses a row index in A1 notation, and adds it to this cell reference
     * structure.
     *
     * @param {String} rowText
     *  The row name in A1 notation, e.g. '1' for the first row.
     *
     * @param {String|Null} absFlag
     *  The absolute marker. Each non-empty string will be recognized as
     *  absolute row index. An empty string, or any other falsy value will be
     *  recognized as relative row index.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {Boolean}
     *  Whether the passed values represent a valid row index.
     */
    CellRef.prototype.parseRow = function (rowText, absFlag, maxRow) {
        this.row = Address.parseRow(rowText);
        this.absRow = !!absFlag;
        return (this.row >= 0) && (this.row <= maxRow);
    };

    /**
     * Parses a column index in R1C1 notation, and adds it to this cell
     * reference structure.
     *
     * @param {String|Null} absText
     *  The absolute column name in R1C1 notation, e.g. '1' for the first
     *  column; or any other falsy value, if the column index is relative (see
     *  parameter 'relText').
     *
     * @param {String|Null} relText
     *  The relative column name in R1C1 notation without brackets, e.g. '-1'
     *  for the preceding column. The empty string, or any other falsy value
     *  will be recognized as reference to the current column (as well as the
     *  string '0', e.g. in 'R1C[0]', 'R1C[]', and 'R1C').
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @param {Number} refCol
     *  The index of the reference column needed to resolve a column offset as
     *  used in R1C1 notation (e.g. in R1C[-1]) to the effective column index.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the column index at the sheet boundaries when resolving
     *  a column offset.
     *
     * @returns {Boolean}
     *  Whether the passed values represent a valid column index.
     */
    CellRef.prototype.parseColRC = function (absText, relText, maxCol, refCol, wrapReferences) {
        this.col = absText ? (parseInt(absText, 10) - 1) : relText ? relocateIndex(refCol, parseInt(relText, 10), maxCol, wrapReferences) : refCol;
        this.absCol = !!absText;
        return (this.col >= 0) && (this.col <= maxCol);
    };

    /**
     * Parses a row index in R1C1 notation, and adds it to this cell reference
     * structure.
     *
     * @param {String|Null} absText
     *  The absolute row name in R1C1 notation, e.g. '1' for the first row; or
     *  any other falsy value, if the row index is relative (see parameter
     *  'relText').
     *
     * @param {String|Null} relText
     *  The relative row name in R1C1 notation without brackets, e.g. '-1' for
     *  the preceding row. The empty string, or any other falsy value will be
     *  recognized as reference to the current row (as well as the string '0',
     *  e.g. in 'R[0]C1', 'R[]C1', and 'RC1').
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @param {Number} refRow
     *  The index of the reference row needed to resolve a row offset as used
     *  in R1C1 notation (e.g. in R[-1]C1) to the effective row index.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the row index at the sheet boundaries when resolving a
     *  row offset.
     *
     * @returns {Boolean}
     *  Whether the passed values represent a valid row index.
     */
    CellRef.prototype.parseRowRC = function (absText, relText, maxRow, refRow, wrapReferences) {
        this.row = absText ? (parseInt(absText, 10) - 1) : relText ? relocateIndex(refRow, parseInt(relText, 10), maxRow, wrapReferences) : refRow;
        this.absRow = !!absText;
        return (this.row >= 0) && (this.row <= maxRow);
    };

    /**
     * Returns a text description of this cell reference for debug logging.
     */
    CellRef.prototype.toString = CellRef.prototype.refText;

    // exports ================================================================

    return CellRef;

});
