/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/textfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, LocaleData, ErrorCode, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with text.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var MathUtils = FormulaUtils.Math;
    var Scalar = FormulaUtils.Scalar;

    // shortcuts to mathematical functions
    var pow10 = MathUtils.pow10;

    // private global functions ===============================================

    function prepareNumberForDecimal(number, decimal) {
        if (decimal < 0) {
            var roundMulti = pow10(decimal);
            if (roundMulti < Number.MIN_VALUE) {
                number = 0;
            } else {
                number = Math.round(roundMulti * number) / roundMulti;
            }
        }
        return number;
    }

    // exports ================================================================

    return {

        ASC: {
            category: 'text',
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        BAHTTEXT: {
            category: 'text',
            name: { odf: 'COM.MICROSOFT.BAHTTEXT' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: (function () {

                var BT_0 = '\u0E28\u0E39\u0E19\u0E22\u0E4C'; // used for number zero, not for digits
                var BT_1_L = '\u0E2B\u0E19\u0E36\u0E48\u0E07'; // leading one (number one, or one hundred, one thousand, etc.)
                var BT_1_T = '\u0E40\u0E2D\u0E47\u0E14'; // trailing one (in 11, 21, 31, 201, 2001, etc.)

                // all digits but zero, used as prefixes for powers of 10 (with the 'leading one' variant)
                var BT_DIGITS = [null, BT_1_L, '\u0E2A\u0E2D\u0E07', '\u0E2A\u0E32\u0E21', '\u0E2A\u0E35\u0E48', '\u0E2B\u0E49\u0E32', '\u0E2B\u0E01', '\u0E40\u0E08\u0E47\u0E14', '\u0E41\u0E1B\u0E14', '\u0E40\u0E01\u0E49\u0E32'];

                var BT_10 = '\u0E2A\u0E34\u0E1A'; // number 10, also used as suffix for twenty, thirty, etc.
                var BT_20 = '\u0E22\u0E35\u0E48'; // prefix for twenty: two-ten (BT_20 + BT_10)
                var BT_1E2 = '\u0E23\u0E49\u0E2D\u0E22'; // hundreds
                var BT_1E3 = '\u0E1E\u0E31\u0E19'; // thousands
                var BT_1E4 = '\u0E2B\u0E21\u0E37\u0E48\u0E19'; // ten thousands
                var BT_1E5 = '\u0E41\u0E2A\u0E19'; // hundred thousands
                var BT_1E6 = '\u0E25\u0E49\u0E32\u0E19'; // millions
                var BT_MINUS = '\u0E25\u0E1A';
                var BT_BAHT = '\u0E1A\u0E32\u0E17';
                var BT_SATANG = '\u0E2A\u0E15\u0E32\u0E07\u0E04\u0E4C';
                var BT_NO_SATANG = '\u0E16\u0E49\u0E27\u0E19'; // suffix for zero Satang
                var BT_NO_SATANG_M = '\u0E16\u0E49'; // suffix for negative rounded zero Baht, zero Satang

                // Generates the text for a single decimal digit.
                function convertDigit(number, pow10, pow10Name) {
                    var digit = Math.floor(number / pow10) % 10;
                    return (digit > 0) ? (BT_DIGITS[digit] + pow10Name) : '';
                }

                // Generates the text for entire Baht for a number less than one million.
                function convertNumber(number, leading, million) {

                    var result = '';

                    // 100k, 10k, 1k, and 100 are simple combinations of digit and decimal suffix
                    result += convertDigit(number, 100000, BT_1E5);
                    result += convertDigit(number, 10000, BT_1E4);
                    result += convertDigit(number, 1000, BT_1E3);
                    result += convertDigit(number, 100, BT_1E2);

                    // create the text for tens
                    var tens = Math.floor(number / 10) % 10;
                    if (tens === 1) {
                        result += BT_10; // no prefix before 'ten' for numbers 10 to 19
                    } else if (tens === 2) {
                        result += BT_20 + BT_10; // use alternative word for digit 2 in 'twenty'
                    } else if (tens >= 3) {
                        result += BT_DIGITS[tens] + BT_10; // digits 3 to 9 are regular
                    }

                    // append the text for the last digit
                    if (leading && (number === 1)) {
                        result += BT_1_L; // 'leading one' variant only if nothing precedes the number one
                    } else {
                        var ones = number % 10;
                        if (ones === 1) {
                            result += BT_1_T; // use the 'trailing one' variant
                        } else if (ones >= 2) {
                            result += BT_DIGITS[ones];
                        }
                    }

                    return million ? (result + BT_1E6) : result;
                }

                return function (number) {

                    // split the input value into Baht and Satang
                    var minus = number < 0;
                    var satang = Math.round(Math.abs(number) * 100);
                    var baht = Math.floor(satang / 100);
                    satang %= 100;

                    // special case: (positive or negative) zero Baht, zero Satang (Excel returns different suffixes?!)
                    if ((baht === 0) && (satang === 0)) {
                        return minus ? (BT_MINUS + BT_0 + BT_BAHT + BT_NO_SATANG_M) : (BT_0 + BT_BAHT + BT_NO_SATANG);
                    }

                    // the function result
                    var result = '';

                    // build the text for Baht
                    if (baht > 0) {
                        var million = false;
                        while (baht > 0) {
                            result = convertNumber(baht % 1e6, baht < 1e6, million) + result;
                            baht = Math.floor(baht / 1e6);
                            million = true;
                        }
                        result += BT_BAHT;
                    }

                    // prepend the minus
                    if (minus) { result = BT_MINUS + result; }

                    // append the text for Satang
                    return result + ((satang > 0) ? (convertNumber(satang, true, false) + BT_SATANG) : BT_NO_SATANG);
                };
            }())
        },

        CHAR: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (value) {
                if ((value < 1) || (value > 255)) { throw ErrorCode.VALUE; }
                // TODO: not exact, needs code page conversion
                return String.fromCharCode(value);
            }
        },

        CLEAN: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                // TODO: other Unicode characters?
                return value.replace(/[\x00-\x1f\x81\x8d\x8f\x90\x9d]/g, '');
            }
        },

        CODE: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                // TODO: not exact, needs code page conversion
                return (value.length === 0) ? ErrorCode.VALUE : value.charCodeAt(0);
            }
        },

        CONCAT: {
            category: 'text',
            name: { ooxml: '_xlfn.CONCAT', odf: null },
            minParams: 1,
            type: 'val',
            signature: 'any|mat:pass',
            resolve: function () {
                // the aggregation function (throws early, if the string length exceeds the limit)
                function aggregate(str1, str2) { return this.concatStrings(str1, str2); }
                return this.aggregateStrings(arguments, '', aggregate, _.identity);
            }
        },

        CONCATENATE: {
            category: 'text',
            minParams: 1,
            type: 'val',
            // each parameter is a single value (function does not concatenate cell ranges or matrixes!)
            signature: 'val:str',
            resolve: function () {
                return this.concatStrings.apply(this, arguments);
            }
        },

        DBCS: {
            category: 'text',
            name: { odf: 'JIS' },
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        DOLLAR: {
            category: 'text',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: (function () {

                var getReplaceString = _.memoize(function (places) {
                    return (places > 0) ? ('.' + Utils.repeatString('0', places)) : '';
                });

                return function (number, places) {

                    if (this.isMissingOperand(1)) {
                        places = 2;
                    } else if (places > 127) {
                        throw ErrorCode.VALUE;
                    }

                    number = prepareNumberForDecimal(number, places);

                    var formatCode = this.numberFormatter.getCurrencyCode();
                    formatCode = formatCode.replace(/\.00/g, getReplaceString(Math.max(0, places)));
                    var parsedFormat = this.numberFormatter.getParsedFormat(formatCode);
                    return this.numberFormatter.formatValue(parsedFormat, number);
                };
            }())
        },

        EXACT: {
            category: 'text',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: function (value1, value2) {
                return value1 === value2; // exact case-sensitive comparison
            }
        },

        FIND: {
            category: 'text',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: function (searchText, text, index) {
                if (_.isUndefined(index)) { index = 1; }
                this.checkStringIndex(index);
                index -= 1;
                // do not search, if index is too large (needed if searchText is empty)
                if (index > text.length - searchText.length) { throw ErrorCode.VALUE; }
                // FIND searches case-sensitive
                index = text.indexOf(searchText, index);
                if (index < 0) { throw ErrorCode.VALUE; }
                return index + 1;
            }
        },

        FINDB: {
            category: 'text',
            hidden: true,
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: 'FIND' // TODO: own implementation for multi-byte character sets?
        },

        FIXED: {
            category: 'text',
            minParams: 1,
            maxParams: 3,
            type: 'val',
            signature: 'val:num val:int val:bool',
            resolve: function (number, places, noGroups) {

                if (this.isMissingOperand(1)) {
                    places = 2;
                } else if (places > 127) {
                    throw ErrorCode.VALUE;
                }

                number = prepareNumberForDecimal(number, places);

                var intCode = noGroups ? '0' : '#,##0';
                var decCode = (places > 0) ? ('.' + Utils.repeatString('0', places)) : '';
                var parsedFormat = this.numberFormatter.getParsedFormat(intCode + decCode);
                return this.numberFormatter.formatValue(parsedFormat, number);
            }
        },

        LEFT: {
            category: 'text',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(0, count);
            }
        },

        LEFTB: {
            category: 'text',
            hidden: true,
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: 'LEFT' // TODO: own implementation for multi-byte character sets?
        },

        LEN: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: _.property('length')
        },

        LENB: {
            category: 'text',
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: 'LEN' // TODO: own implementation for multi-byte character sets?
        },

        LOWER: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) { return text.toLowerCase(); }
        },

        MID: {
            category: 'text',
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:int val:int',
            resolve: function (text, start, count) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                return text.substr(start - 1, count);
            }
        },

        MIDB: {
            category: 'text',
            hidden: true,
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:int val:int',
            resolve: 'MID' // TODO: own implementation for multi-byte character sets?
        },

        NUMBERVALUE: {
            category: 'text',
            name: { ooxml: '_xlfn.NUMBERVALUE' },
            minParams: 1,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:str',
            resolve: function (text, decSep, groupSep) {

                if (text.length === 0) { return '0'; }

                if (this.isMissingOperand(1)) {
                    decSep = LocaleData.DEC;
                } else if (decSep.length === 0) {
                    throw ErrorCode.VALUE;
                }

                if (this.isMissingOperand(2)) {
                    groupSep = LocaleData.GROUP;
                } else if (groupSep.length === 0) {
                    throw ErrorCode.VALUE;
                }

                decSep = decSep[0];
                groupSep = groupSep[0];
                if (decSep === groupSep) { throw ErrorCode.VALUE; }

                var portions = text.split(decSep);
                if (portions.length > 2) { throw ErrorCode.VALUE; }

                var groupRE = new RegExp(_.escapeRegExp(groupSep), 'g');
                if ((portions.length === 2) && groupRE.test(portions[1])) { throw ErrorCode.VALUE; }

                portions = portions.map(function (portion) { return portion.replace(groupRE, ''); });

                var preparedText = portions.join(LocaleData.DEC);

                var percent = /%+/.exec(preparedText);

                if (percent) {
                    percent = percent[0];
                    preparedText = preparedText.replace(percent, '%');
                }

                var form = this.numberFormatter.parseFormattedValue(preparedText);
                var value = form.value;

                if (!_.isNumber(value)) { throw ErrorCode.VALUE; }

                if (percent && percent.length > 1) {
                    value /= pow10(2 * percent.length - 2);
                }

                return value;
            }
        },

        PHONETIC: {
            category: 'text',
            name: { odf: null },
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single'
        },

        PROPER: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // TODO: the reg-exp does not include all characters that count as letters
                return text.replace(/[A-Za-z\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02B8\u02BB-\u02C1\u02D0-\u02D1\u02E0-\u02E4\u02EE\u0370-\u0373\u0376-\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0523]+/g, Utils.capitalize);
            }
        },

        REPLACE: {
            category: 'text',
            minParams: 4,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:int val:int val:str',
            resolve: function (oldText, start, count, newText) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                start -= 1;
                return Utils.replaceSubString(oldText, start, start + count, newText);
            }
        },

        REPLACEB: {
            category: 'text',
            hidden: true,
            minParams: 4,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:int val:int val:str',
            resolve: 'REPLACE' // TODO: own implementation for multi-byte character sets?
        },

        REPT: {
            category: 'text',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, repeat) {
                return this.repeatString(text, repeat);
            }
        },

        RIGHT: {
            category: 'text',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(-count);
            }
        },

        RIGHTB: {
            category: 'text',
            hidden: true,
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: 'RIGHT' // TODO: own implementation for multi-byte character sets?
        },

        ROT13: {
            category: 'text',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.ROT13' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                return text.replace(/[a-zA-Z]/g, function (a) {
                    return String.fromCharCode(((a = a.charCodeAt()) < 91 ? 78 : 110) > a ? a + 13 : a - 13);
                });
            }
        },

        SEARCH: {
            category: 'text',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: function (searchText, text, index) {

                // prepare and check the passed index
                if (_.isUndefined(index)) { index = 1; }
                this.checkStringIndex(index);
                index -= 1;

                // return index, if searchText is empty or a 'match all' selector
                if (/^\**$/.test(searchText)) {
                    if (text.length <= index) { throw ErrorCode.VALUE; }
                    return index + 1;
                }

                // shorten the passed text according to the start index
                text = text.substr(index);

                var // create the regular expression
                    regExp = this.convertPatternToRegExp(searchText),
                    // find first occurrence of the pattern
                    matches = regExp.exec(text);

                // nothing found: throw #VALUE! error code
                if (!_.isArray(matches)) { throw ErrorCode.VALUE; }

                // return the correct index of the search text
                return matches.index + index + 1;
            }
        },

        SEARCHB: {
            category: 'text',
            hidden: true,
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: 'SEARCH' // TODO: own implementation for multi-byte character sets?
        },

        SUBSTITUTE: {
            category: 'text',
            minParams: 3,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:str val:str val:int',
            resolve: function (text, replaceText, newText, index) {

                // check passed index first (one-based index)
                if (_.isNumber(index) && (index <= 0)) { throw ErrorCode.VALUE; }

                // do nothing, if the text to be replaced is empty
                if (replaceText.length === 0) { return text; }

                var // the regular expression that will find the text(s) to be replaced (SUBSTITUTE works case-sensitive)
                    regExp = new RegExp(_.escapeRegExp(replaceText), 'g');

                // replace all occurrences
                if (_.isUndefined(index)) {
                    return text.replace(regExp, newText);
                }

                // replace the specified occurrence of the text
                var splits = text.split(regExp);
                if (index < splits.length) {
                    splits[index - 1] += newText + splits[index];
                    splits.splice(index, 1);
                    return splits.join(replaceText);
                }

                // index too large: return original text
                return text;
            }
        },

        T: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val',
            resolve: function (value) {
                return _.isString(value) ? value : '';
            }
        },

        TEXT: {
            category: 'text',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            recalc: 'once', // result depends on GUI language
            signature: 'val:num val:str',
            resolve: function (number, formatText) {
                var parsedFormat = this.numberFormatter.getParsedFormat(formatText, { grammarId: 'ui' });
                var result = this.numberFormatter.formatValue(parsedFormat, number);
                if (result === null) { throw ErrorCode.VALUE; }
                return result;
            }
        },

        TEXTJOIN: {
            category: 'text',
            name: { ooxml: '_xlfn.TEXTJOIN', odf: null },
            minParams: 3,
            type: 'val',
            signature: 'val:str val:bool any|mat:pass',
            resolve: function (delimiter, skipEmpty) {

                // offset of the last visited string
                var lastOffset = 0;

                // concatenate the strings with the delimiter
                function aggregate(text1, text2, index, offset) {
                    // silently skip empty strings
                    if (text2.length === 0) { return text1; }
                    // repeat delimiter for sequences of blank cells
                    var delimCount = !skipEmpty ? (offset - lastOffset) : (lastOffset < offset) ? 1 : 0;
                    var delim = (delimCount > 0) ? this.repeatString(delimiter, delimCount) : '';
                    lastOffset = offset;
                    return this.concatStrings(text1, delim, text2);
                }

                // finally, add trailing separators for blank cells
                function finalize(text, count, size) {
                    return skipEmpty ? text : (text + this.repeatString(delimiter, size - lastOffset - 1));
                }

                return this.aggregateStrings(_.toArray(arguments).slice(2), '', aggregate, finalize);
            }
        },

        TRIM: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // TRIM removes space characters (U+0020) only, no other white-space or control characters
                return text.replace(/^ +| +$/g, '').replace(/ {2,}/g, ' ');
            }
        },

        UNICHAR: {
            category: 'text',
            name: { ooxml: '_xlfn.UNICHAR' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (value) {
                if ((value < 1) || (value > 0xFFFF)) { throw ErrorCode.VALUE; }
                if ((value >= 0xD800) && (value <= 0xDFFF)) { throw ErrorCode.NA; }
                if ((value >= 0xFDD0) && (value <= 0xFDEF)) { throw ErrorCode.NA; }
                if (value >= 0xFFFE) { throw ErrorCode.NA; }
                return String.fromCharCode(value);
            }
        },

        UNICODE: {
            category: 'text',
            name: { ooxml: '_xlfn.UNICODE' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                return (value.length === 0) ? ErrorCode.VALUE : value.charCodeAt(0);
            }
        },

        UPPER: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) { return text.toUpperCase(); }
        },

        USDOLLAR: {
            category: 'text',
            name: { odf: null },
            hidden: true,
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: 'DOLLAR'
        },

        VALUE: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val',
            resolve: function (value) {
                // text can be converted to number, but boolean values will result in #VALUE!
                switch (Scalar.getType(value)) {
                    case Scalar.Type.NUMBER: return value;
                    case Scalar.Type.STRING: return this.convertToNumber(value);
                    case Scalar.Type.NULL:   return 0; // reference to empty cell
                }
                throw ErrorCode.VALUE;
            }
        }
    };

});
