/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/logicalfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/dimension'
], function (ErrorCode, Dimension) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with boolean
     * values.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // standard options for logical aggregation functions (AND, OR, XOR)
    var AGGREGATE_OPTIONS = { // id: CSS5
        valMode: 'convert', // value operands: convert strings and numbers to booleans
        matMode: 'skip', // matrix operands: skip all strings
        refMode: 'skip', // reference operands: skip all strings
        emptyParam: true, // empty parameters count as FALSE
        complexRef: true // accept multi-range and multi-sheet references
    };

    // private global functions ===============================================

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all boolean values (and other values convertible to boolean
     * values) of all function parameters to the result of the function. See
     * method FormulaContext.aggregateBooleans() for more details about the
     * parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all boolean values of all its operands to the aggregation callback
     *  function. If no boolean values have been found at all, the spreadsheet
     *  function will result in the #VALUE! error code, instead of the initial
     *  result.
     */
    function implementLogicalAggregation(initial, aggregate) {

        // no booleans available results in the #VALUE! error code (not the initial value)
        function finalize(result, count) {
            if (count === 0) { throw ErrorCode.VALUE; }
            return result;
        }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function logicalAggregator() {
            return this.aggregateBooleans(arguments, initial, aggregate, finalize, AGGREGATE_OPTIONS);
        };
    }

    // exports ================================================================

    return {

        AND: {
            category: 'logical',
            minParams: 1,
            type: 'val',
            signature: 'any|mat:pass',
            // empty parameters count as FALSE: AND(TRUE,) = FALSE
            resolve: implementLogicalAggregation(true, function (bool1, bool2) { return bool1 && bool2; })
        },

        FALSE: {
            category: 'logical',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(false)
        },

        IF: {
            category: 'logical',
            minParams: { ooxml: 2, odf: 1 },
            maxParams: 3,
            type: 'any',
            signature: 'val:bool|mat:pass any:lazy|deps:pass|mat:pass any:lazy|deps:pass|mat:pass',
            resolve: function (cond) {

                var count = this.getOperandCount();

                // Special behavior: If first parameter is a matrix, the result will be combined element-by-element
                // from the second and third parameter, otherwise the result will be passed through unmodified.
                //
                // Example (simple cell formulas):
                // =ISREF(IF(TRUE,A1:B2)) results in TRUE (the reference A1:B2 directly)
                // =ISREF(IF({TRUE},A1:B2)) results in FALSE (a matrix built from A1:B2)
                // =IF({1|0},{2|2},{3|3}) results in {2|3} (result matrix built element-by-element)
                //
                if ((this.getContextType() !== 'val') && this.getOperand(0).isMatrix()) {

                    // the dimension of the result matrix is the boundary of all parameters
                    var matrixDim = this.getOperands(0).reduce(function (dim, operand) {
                        return Dimension.boundary(dim, operand.getDimension({ errorCode: ErrorCode.NA })); // #N/A thrown for complex references
                    }, null);

                    return this.aggregateMatrix(matrixDim, function () {
                        var condElem = this.getOperand(0, 'val:bool');
                        var index = condElem ? 1 : 2;
                        return (index < count) ? this.getOperand(index, 'val:any') : condElem;
                    });
                }

                // otherwise, return one of the unconverted operands
                var index = cond ? 1 : 2;
                return (index < count) ? this.getOperand(index) : cond;
            }
        },

        IFERROR: {
            category: 'logical',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:any val:any', // always return scalar values (in difference to IF)
            resolve: function (value1, value2) {
                return (value1 instanceof ErrorCode) ? value2 : value1;
            }
        },

        IFNA: {
            category: 'logical',
            name: { ooxml: '_xlfn.IFNA' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:any val:any', // always return scalar values (in difference to IF)
            resolve: function (value1, value2) {
                return (value1 === ErrorCode.NA) ? value2 : value1;
            }
        },

        IFS: {
            category: 'logical',
            name: { ooxml: '_xlfn.IFS', odf: null },
            minParams: 2,
            repeatParams: 2,
            type: 'any',
            signature: 'val:bool any:lazy|deps:pass',
            resolve: function () {
                for (var index = 0, length = this.getOperandCount(); index < length; index += 2) {
                    if (this.getOperandValue(index)) { return this.getOperand(index + 1); }
                }
                throw ErrorCode.NA;
            }
        },

        NOT: {
            category: 'logical',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:bool',
            resolve: function (value) { return !value; }
        },

        OR: {
            category: 'logical',
            minParams: 1,
            type: 'val',
            signature: 'any|mat:pass',
            // empty parameters count as FALSE (but do not skip to catch empty-only parameters)
            resolve: implementLogicalAggregation(false, function (bool1, bool2) { return bool1 || bool2; })
        },

        TRUE: {
            category: 'logical',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(true)
        },

        XOR: {
            category: 'logical',
            name: { ooxml: '_xlfn.XOR' },
            minParams: 1,
            type: 'val',
            signature: 'any|mat:pass',
            // empty parameters count as FALSE (but do not skip to catch empty-only parameters)
            resolve: implementLogicalAggregation(false, function (bool1, bool2) { return bool1 !== bool2; })
        }
    };

});
