/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/datetimefuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/dateutils'
], function (Utils, ErrorCode, DateUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with date and
     * time.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, dates and times are represented by
     * floating-point numbers (integral part is number of days, counted from a
     * specific 'null date', fractional part is the time of a day). The
     * client-side formula engine provides the function signature type
     * 'val:date', and uses the JavaScript class Date containing the UTC date
     * and time received from spreadsheet cells, or other functions.
     *
     *************************************************************************/

    // standard options for date parameter iterators (method FormulaContext.iterateNumbers())
    var RCONVERT_ALL_SKIP_EMPTY = { // id: RRR0
        valMode: 'rconvert', // value operands: convert strings to numbers (but not booleans)
        matMode: 'rconvert', // matrix operands: convert strings to numbers (but not booleans)
        refMode: 'rconvert', // reference operands: convert strings to numbers (but not booleans)
        emptyParam: false, // skip empty parameter
        complexRef: false, // do not accept multi-range and multi-sheet references
        dates: true, // convert numbers to dates
        floor: true // remove time components from all dates
    };

    // private global functions ===============================================

    /**
     * Implementation helper for the sheet functions WEEKNUM and ISOWEEKNUM.
     *
     * @param {Date} date
     *  The date the week number will be calculated for, as UTC date value.
     *
     * @param {Number} firstDayOfWeek
     *  Determines the start day of the week: 0 for Sunday, 1 for Monday, ...,
     *  or 6 for Saturday.
     *
     * @param {Boolean} iso8601
     *  Whether to use ISO 8601 mode (week of January 1st is the first week, if
     *  at least 4 days are in the new year), or US mode (weeks are counted
     *  strictly inside the year, first and last week in the year may consist
     *  of less than 7 days).
     *
     * @returns {Number}
     *  The resulting week number for the passed date.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the date mode is not valid.
     */
    function getWeekNum(date, firstDayOfWeek, iso8601) {

        // returns the date of the first day of the first week, according to specified first day in week
        function getNullDateOfYear(year) {

            // base date (that is always in the first week of the year; ISO: Jan-04, otherwise: Jan-01)
            var baseDate = DateUtils.makeUTCDate({ Y: year, M: 0, D: iso8601 ? 4 : 1 });
            // week day of the base date
            var baseDay = baseDate.getUTCDay();

            // first day of first week in the year, according to date mode
            return DateUtils.addDaysToDate(baseDate, -Utils.mod(baseDay - firstDayOfWeek, 7));
        }

        var // the year of the passed date
            currYear = date.getUTCFullYear(),
            // first day of first week in the year, according to date mode
            nullDate = getNullDateOfYear(currYear);

        // adjustment for ISO mode: passed date may be counted for the previous year
        // (for Jan-01 to Jan-03), or for the next year (for Dec-29 to Dec-31)
        if (iso8601) {
            if (date < nullDate) {
                // passed date is counted for previous year (e.g. Jan-01-2000 is in week 52 of 1999)
                nullDate = getNullDateOfYear(currYear - 1);
            } else {
                var nextNullDate = getNullDateOfYear(currYear + 1);
                if (nextNullDate <= date) {
                    // passed date is counted for next year (e.g. Dec-31-2001 is in week 1 of 2002)
                    nullDate = nextNullDate;
                }
            }
        }

        return Math.floor((date - nullDate) / DateUtils.MSEC_PER_WEEK) + 1;
    }

    // workday helpers ----------------------------------------------------

    var WEEKEND_CODES = { 1: /* default */ '0000011', 2: '1000001', 3: '1100000', 4: '0110000', 5: '0011000', 6: '0001100', 7: '0000110', 11: '0000001', 12: '1000000', 13: '0100000', 14: '0010000', 15: '0001000', 16: '0000100', 17: '0000010' };

    function getWeekendFromParam(weekend) {
        if (_.isNull(weekend) || _.isUndefined(weekend)) {
            weekend = WEEKEND_CODES['1'];
        } else if (_.isBoolean(weekend)) {
            if (weekend) {
                weekend = WEEKEND_CODES['1'];
            } else {
                throw ErrorCode.NUM;
            }
        } else if (_.isNumber(weekend)) {
            weekend = WEEKEND_CODES[weekend];
            if (!weekend) { throw ErrorCode.NUM; }
        }
        if (!_.isString(weekend) || !/^[01]{7}$/.test(weekend)) {
            throw ErrorCode.VALUE;
        }
        return weekend;
    }

    function getWorkDaysPerWeek(weekendCode) {
        return (weekendCode.match(/0/g) || '').length;
    }

    function getWorkPieces(startDate, endDate, holidays) {
        //FIXME: parse to full days

        var pieces = [];
        holidays = _.filter(holidays, function (holiday) {
            return holiday >= startDate && holiday <= endDate;
        });

        holidays.sort();

        var lastWorkDay = startDate;
        for (var i = 0; i < holidays.length; i++) {
            var holiday = holidays[i];
            if (lastWorkDay && DateUtils.getDaysDiff(lastWorkDay, holiday) > 0) {
                pieces.push({ from: lastWorkDay, to: DateUtils.addDaysToDate(holiday, -1) });
                lastWorkDay = DateUtils.addDaysToDate(holiday, 1);
            } else {
                lastWorkDay = null;
            }
        }
        if (!lastWorkDay) {
            lastWorkDay = DateUtils.addDaysToDate(_.last(holidays), 1);
        }
        if (DateUtils.getDaysDiff(lastWorkDay, endDate) >= 0) {
            pieces.push({ from: lastWorkDay, to: endDate });
        }
        return pieces;
    }

    function getWorkDays(startDate, endDate, weekend, workDaysPerWeek) {
        var startWeekDay = DateUtils.getWeekDay(startDate);
        var endWeekDay = DateUtils.getWeekDay(endDate);

        var startMonday = DateUtils.getWeekStart(startDate);
        var endMonday = DateUtils.getWeekStart(endDate);

        var workDays = 0;
        var i = 0;

        if (DateUtils.getDaysDiff(startMonday, endMonday) === 0) {
            //same week and also one timeslot
            for (i = startWeekDay; i <= endWeekDay; i++) {
                if (weekend[i] === '0') {
                    workDays++;
                }
            }
        } else {
            //first week slot
            for (i = startWeekDay; i <= 6; i++) {
                if (weekend[i] === '0') {
                    workDays++;
                }
            }

            //last week slot
            for (i = 0; i <= endWeekDay; i++) {
                if (weekend[i] === '0') {
                    workDays++;
                }
            }

            //all weeks between
            var endSunday = DateUtils.addDaysToDate(endMonday, -1);
            var weeksDiff = DateUtils.getWeeksDiff(startMonday, endSunday);
            workDays += weeksDiff * workDaysPerWeek;

        }
        return workDays;
    }

    function findNextWorkday(startDate, endDate, weekend, workDaysPerWeek, prefix, daysCounter) {
        if (prefix < 0) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
        }

        //---- begin of perf optimization ---------

        var sliceWeeks = Math.floor(DateUtils.getDaysDiff(startDate, endDate) / 7) * prefix;
        var restWeeks = Math.floor(daysCounter.value / workDaysPerWeek);

        var weeks = Math.abs(sliceWeeks) < Math.abs(restWeeks) ? sliceWeeks : restWeeks;

        if (weeks < -1 || weeks > 1) {
            startDate = DateUtils.addDaysToDate(startDate, (weeks - prefix) * 7);
            daysCounter.value -= (weeks - prefix) * workDaysPerWeek;
        }

        //---- end of perf optimization ---------

        var dayCount = DateUtils.getDaysDiff(startDate, endDate) * prefix;
        for (var i = 0; i <= dayCount; i++) {
            var currentDate = DateUtils.addDaysToDate(startDate, i * prefix);
            var currentWeekDay = DateUtils.getWeekDay(currentDate);
            if (weekend[currentWeekDay] === '0') {
                daysCounter.value -= prefix;
                if (daysCounter.value === 0) {
                    return currentDate;
                }
            }
        }
    }

    function networkDaysIntl(context, startDate, endDate, weekendParam, holidaysParam) {
        var weekend = getWeekendFromParam(weekendParam);
        var workDaysPerWeek = getWorkDaysPerWeek(weekend);
        var holidays = context.getNumbersAsArray(holidaysParam, RCONVERT_ALL_SKIP_EMPTY);

        if (workDaysPerWeek === 0) {
            return 0;
        }
        var prefix = 1;

        // order start and end date
        if (endDate < startDate) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
            prefix = -1;
        }

        var workDays = 0;

        if (holidays.length) {
            var workPieces = getWorkPieces(startDate, endDate, holidays);
            _.each(workPieces, function (workPiece) {
                workDays += getWorkDays(workPiece.from, workPiece.to, weekend, workDaysPerWeek);
            });
        } else {
            workDays += getWorkDays(startDate, endDate, weekend, workDaysPerWeek);
        }
        return workDays * prefix;
    }

    function workDayIntl(context, startDate, days, weekendParam, holidaysParam) {

        var weekend = getWeekendFromParam(weekendParam);
        var workDaysPerWeek = getWorkDaysPerWeek(weekend);
        var holidays = context.getNumbersAsArray(holidaysParam, RCONVERT_ALL_SKIP_EMPTY);

        if (workDaysPerWeek === 0) {
            throw ErrorCode.VALUE;
        }
        if (days === 0) {
            return startDate;
        }
        var prefix = 1;
        var endDate = null;
        var daysCounter = {
            value: days
        };

        var holidayCount = holidays.length;
        var maxDays = Math.ceil((Math.abs(days) / workDaysPerWeek) * 7) + holidayCount;

        //the safe way
        maxDays *= 2;

        // order start and end date
        if (days < 0) {
            prefix = -1;
            endDate = startDate;
            startDate = DateUtils.addDaysToDate(startDate, maxDays * prefix);
            //first day is always ignored!!!
            endDate = DateUtils.addDaysToDate(endDate, prefix);
        } else {
            endDate = DateUtils.addDaysToDate(startDate, maxDays * prefix);
            //first day is always ignored!!!
            startDate = DateUtils.addDaysToDate(startDate, prefix);
        }

        var workPieces = null;
        if (holidays) {
            workPieces = getWorkPieces(startDate, endDate, holidays);
            if (prefix < 0) {
                workPieces.reverse();
            }
        } else {
            workPieces = [{ from: startDate, to: endDate }];
        }

        for (var i = 0; i < workPieces.length; i++) {
            var workPiece = workPieces[i];
            var result = findNextWorkday(workPiece.from, workPiece.to, weekend, workDaysPerWeek, prefix, daysCounter);
            if (result) {
                return result;
            }
        }
        throw ErrorCode.REF;
    }

    function parseDateTime(context, text) {
        var result = context.numberFormatter.parseFormattedValue(text);
        var parsedFormat = context.numberFormatter.getParsedFormat(result.format);
        if (parsedFormat.isAnyDateTime()) {
            return result.value;
        }
        throw ErrorCode.VALUE;
    }

    // exports ================================================================

    return {

        DATE: {
            category: 'datetime',
            minParams: 3,
            maxParams: 3,
            type: 'val',
            format: 'date',
            signature: 'val:int val:int val:int',
            resolve: function (year, month, day) {
                // Excel adds 1900 to all years less than 1900; e.g. 1899 becomes 3799 (TODO: different behavior for ODF?)
                if (year < 1900) { year += 1900; }
                // create a UTC date object (Date object expects zero-based month)
                return DateUtils.makeUTCDate({ Y: year, M: month - 1, D: day });
            }
        },

        DATEDIF: {
            category: 'datetime',
            hidden: true,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DATEVALUE: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (dateText) {
                return Math.floor(parseDateTime(this, dateText));
            }
        },

        DAY: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) { return date.getUTCDate(); }
        },

        DAYS: {
            category: 'datetime',
            name: { ooxml: '_xlfn.DAYS' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:day val:day',
            resolve: function (endDate, startDate) {
                // DAYS() expects end date before start date
                return DateUtils.getDaysDiff(startDate, endDate);
            }
        },

        DAYS360: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day val:bool',
            resolve: function (startDate, endDate, euroMode) {
                // euroMode missing or FALSE: use 30/360 US mode
                return DateUtils.getDays360(startDate, endDate, euroMode ? 'eur' : 'us:days');
            }
        },

        DAYSINMONTH: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.DAYSINMONTH' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) {
                return DateUtils.getDaysInMonth(date.getUTCFullYear(), date.getUTCMonth());
            }
        },

        DAYSINYEAR: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.DAYSINYEAR' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) {
                return DateUtils.getDaysInYear(date.getUTCFullYear());
            }
        },

        EASTERSUNDAY: {
            category: 'datetime',
            // AOO 4.4 writes 'EASTERSUNDAY' (https://bz.apache.org/ooo/show_bug.cgi?id=126519)
            name: { ooxml: null, odf: ['ORG.OPENOFFICE.EASTERSUNDAY', 'EASTERSUNDAY'] },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            format: 'date',
            signature: 'val:int',
            resolve: function (year) {

                // adjust two-digit years, check validity (TODO: correct error code?)
                year = this.numberFormatter.expandYear(year);
                if ((year < 1583) || (year > 9999)) { throw ErrorCode.NUM; }

                // implementation of the Gauss algorithm, adapted from OOo source code (sc/source/core/tool/interpr2.cxx)
                var n = year % 19,
                    b = Math.floor(year / 100),
                    c = year % 100,
                    d = Math.floor(b / 4),
                    e = b % 4,
                    f = Math.floor((b + 8) / 25),
                    g = Math.floor((b - f + 1) / 3),
                    h = (19 * n + b - d - g + 15) % 30,
                    i = Math.floor(c / 4),
                    k = c % 4,
                    l = (32 + 2 * e + 2 * i - h - k) % 7,
                    m = Math.floor((n + 11 * h + 22 * l) / 451),
                    o = h + l - 7 * m + 114;

                return DateUtils.makeUTCDate({ Y: year, M: Math.floor(o / 31) - 1, D: o % 31 + 1 });
            }
        },

        EDATE: {
            category: 'datetime',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            format: 'date',
            signature: 'val:day val:int',
            resolve: function (date, months) {
                // set day in result to the day of the passed original date
                return DateUtils.addMonthsToDate(date, months);
            }
        },

        EOMONTH: {
            category: 'datetime',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            format: 'date',
            signature: 'val:day val:int',
            resolve: function (date, months) {
                // set day in result to the last day in the resulting month
                return DateUtils.addMonthsToDate(date, months, 'last');
            }
        },

        HOUR: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCHours(); }
        },

        ISLEAPYEAR: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.ISLEAPYEAR' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) {
                return DateUtils.isLeapYear(date.getUTCFullYear());
            }
        },

        ISOWEEKNUM: {
            // called WEEKNUM in the UI of AOO/LO
            category: 'datetime',
            name: { ooxml: '_xlfn.ISOWEEKNUM' },
            minParams: { ooxml: 1, odf: 2 },
            maxParams: { ooxml: 1, odf: 2 },
            type: 'val',
            signature: 'val:day val:int',
            resolve: function (date, mode) {
                // ODF: date mode 1 for Sunday, every other value for Monday; OOXML: always Monday
                return getWeekNum(date, (mode === 1) ? 0 : 1, true);
            }
        },

        MINUTE: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCMinutes(); }
        },

        MONTH: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) { return date.getUTCMonth() + 1; } // JS Date class returns zero-based month
        },

        MONTHS: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.MONTHS' },
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day val:int',
            resolve: function (startDate, endDate, whole) {
                return DateUtils.getMonthsDiff(startDate, endDate, whole !== 1);
            }
        },

        NETWORKDAYS: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day any',
            resolve: function (startDate, endDate, holidays) {
                return networkDaysIntl(this, startDate, endDate, 1, holidays);
            }
        },

        'NETWORKDAYS.INTL': {
            category: 'datetime',
            name: { odf: 'COM.MICROSOFT.NETWORKDAYS.INTL' },
            minParams: 2,
            maxParams: 4,
            type: 'val',
            signature: 'val:day val:day val any',
            resolve: function (startDate, endDate, weekend, holidays) {
                return networkDaysIntl(this, startDate, endDate, weekend, holidays);
            }
        },

        NOW: {
            category: 'datetime',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            format: 'datetime',
            recalc: 'always',
            resolve: DateUtils.makeUTCNow
        },

        SECOND: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCSeconds(); }
        },

        TIME: {
            category: 'datetime',
            minParams: 3,
            maxParams: 3,
            type: 'val',
            format: 'time',
            signature: 'val:int val:int val:int',
            resolve: function (hour, minute, second) {

                // the time value, as fraction of a day
                var time = (hour / 24 + minute / 1440 + second / 86400) % 1;

                // must not result in a negative value
                if (time < 0) { throw ErrorCode.NUM; }

                // convert to a date object, using the document's current null date
                return this.convertToDate(time);
            }
        },

        TIMEVALUE: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (timeText) {
                return parseDateTime(this, timeText) % 1;
            }
        },

        TODAY: {
            category: 'datetime',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            format: 'date',
            recalc: 'always',
            resolve: DateUtils.makeUTCToday
        },

        WEEKDAY: {
            category: 'datetime',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:day val:int',
            resolve: function (date, mode) {

                // weekday index, from 0 to 6, starting at Sunday
                var weekDay = date.getUTCDay();

                // missing parameter defaults to 1 (but: empty parameter defaults to 0)
                switch (_.isUndefined(mode) ? 1 : mode) {
                    case 1:
                        return weekDay + 1; // 1 to 7, starting at Sunday
                    case 2:
                        return (weekDay === 0) ? 7 : weekDay; // 1 to 7, starting at Monday
                    case 3:
                        return (weekDay + 6) % 7; // 0 to 6, starting at Monday
                }

                // other modes result in #NUM! error
                throw ErrorCode.NUM;
            }
        },

        WEEKNUM: {
            // called WEEKNUM_ADD in the UI of AOO/LO
            category: 'datetime',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:day val:int',
            resolve: (function () {

                // maps date modes to first day in week (zero-based, starting from Sunday)
                var FIRST_WEEKDAY = { 1: 0, 2: 1, 11: 1, 12: 2, 13: 3, 14: 4, 15: 5, 16: 6, 17: 0, 21: 1 };

                return function (date, mode) {

                    // first day in the week, according to passed date mode
                    var firstDayOfWeek = FIRST_WEEKDAY[_.isNumber(mode) ? mode : 1];

                    // bail out if passed date mode is invalid
                    if (!_.isNumber(firstDayOfWeek)) { throw ErrorCode.NUM; }

                    return getWeekNum(date, firstDayOfWeek, mode === 21);
                };
            }())
        },

        WEEKS: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.WEEKS' },
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day val:int',
            resolve: function (startDate, endDate, whole) {
                return DateUtils.getWeeksDiff(startDate, endDate, whole !== 1);
            }
        },

        WEEKSINYEAR: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.WEEKSINYEAR' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) {
                // the 28th of December has always maximum week number in ISO8601 mode
                return getWeekNum(DateUtils.makeUTCDate({ Y: date.getUTCFullYear(), M: 11, D: 28 }), 1, true);
            }
        },

        WORKDAY: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            format: 'date',
            signature: 'val:day val:int any',
            resolve: function (startDate, days, holidays) {
                return workDayIntl(this, startDate, days, undefined, holidays);
            }
        },

        'WORKDAY.INTL': {
            category: 'datetime',
            name: { odf: 'COM.MICROSOFT.WORKDAY.INTL' },
            minParams: 2,
            maxParams: 4,
            type: 'val',
            format: 'date',
            signature: 'val:day val:int val:int any',
            resolve: function (startDate, days, weekend, holidays) {
                return workDayIntl(this, startDate, days, weekend, holidays);
            }
        },

        YEAR: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:day',
            resolve: function (date) { return date.getUTCFullYear(); }
        },

        YEARFRAC: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day val:int',
            resolve: function (startDate, endDate, mode) {
                return DateUtils.getYearFrac(startDate, endDate, mode);
            }
        },

        YEARS: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.YEARS' },
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day val:int',
            resolve: function (startDate, endDate, whole) {
                return DateUtils.getYearsDiff(startDate, endDate, whole !== 1);
            }
        }
    };

});
