/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/formulautils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/utils/mathutils',
    'io.ox/office/spreadsheet/model/formula/utils/scalar',
    'io.ox/office/spreadsheet/model/formula/utils/complex',
    'io.ox/office/spreadsheet/model/formula/utils/matrix',
    'io.ox/office/spreadsheet/model/formula/utils/dimension',
    'io.ox/office/spreadsheet/model/formula/utils/formulaerror',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref'
], function (Utils, Logger, SheetUtils, MathUtils, Scalar, Complex, Matrix, Dimension, FormulaError, CellRef, SheetRef) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;

    // static private functions ===============================================

    /**
     * Creates a special error code for internal usage in the formula engine.
     */
    function makeInternalErrorCode(key, value) {
        var errorCode = new ErrorCode(key);
        errorCode.internal = true;
        errorCode.value = value;
        return errorCode;
    }

    // static class FormulaUtils ==============================================

    /**
     * The static class FormulaUtils provides low-level helper functions needed
     * to implement the formula engine of a spreadsheet application.
     * Additionally, the class is a console logger bound to the URL hash flag
     * 'spreadsheet:log-formulas', that logs detailed information about the
     * formula parser, compiler, and interpreter.
     */
    var FormulaUtils = {};

    // logger interface -------------------------------------------------------

    Logger.extend(FormulaUtils, { enable: 'spreadsheet:log-formulas', prefix: 'FMLA' });

    // helper classes ---------------------------------------------------------

    // export utility classes for convenience
    FormulaUtils.Math = MathUtils;
    FormulaUtils.Scalar = Scalar;
    FormulaUtils.Complex = Complex;
    FormulaUtils.Matrix = Matrix;
    FormulaUtils.Dimension = Dimension;
    FormulaUtils.FormulaError = FormulaError;
    FormulaUtils.CellRef = CellRef;
    FormulaUtils.SheetRef = SheetRef;

    // constants --------------------------------------------------------------

    /**
     * Maximum number of function parameters (CalcEngine limitation).
     *
     * @constant
     * @type {Number}
     */
    FormulaUtils.MAX_PARAM_COUNT = 254;

    /**
     * Maximum number of references in a reference list.
     *
     * @constant
     * @type {Number}
     */
    FormulaUtils.MAX_REF_LIST_SIZE = 1024;

    /**
     * Maximum time available for evaluating a single formula, in milliseconds.
     *
     * @constant
     * @type {Number}
     */
    FormulaUtils.MAX_EVAL_TIME = 2000;

    /**
     * Special error code literal to transport the 'unsupported' state.
     *
     * @constant
     * @type {ErrorCode}
     */
    FormulaUtils.UNSUPPORTED_ERROR = makeInternalErrorCode('unsupported', ErrorCode.NA);

    /**
     * Special error code literal to transport the 'circular reference' result.
     *
     * @constant
     * @type {ErrorCode}
     */
    FormulaUtils.CIRCULAR_ERROR = makeInternalErrorCode('circular', 0);

    /**
     * Special error code literal to transport the 'timeout' state.
     *
     * @constant
     * @type {ErrorCode}
     */
    FormulaUtils.TIMEOUT_ERROR = makeInternalErrorCode('timeout', ErrorCode.NA);

    /**
     * Special error code literal for other internal interpreter errors.
     *
     * @constant
     * @type {ErrorCode}
     */
    FormulaUtils.INTERNAL_ERROR = makeInternalErrorCode('internal', ErrorCode.NA);

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed value is an internal error code.
     *
     * @param {Any} value
     *  Any result value used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an internal error code.
     */
    FormulaUtils.isInternalError = function (value) {
        return (value instanceof ErrorCode) && !!value.internal;
    };

    /**
     * Throws the passed value, if it is an error code. Otherwise, the passed
     * value will be returned.
     *
     * @param {Any} value
     *  Any scalar value used in formulas.
     *
     * @returns {Any}
     *  The passed value, if it is not an error code (instance of ErrorCode).
     *
     * @throws {ErrorCode}
     *  The passed value, if it is an error code.
     */
    FormulaUtils.throwErrorCode = function (value) {
        if (value instanceof ErrorCode) { throw value; }
        return value;
    };

    /**
     * Prints the passed error message to the browser console, and throws the
     * internal error code FormulaUtils.INTERNAL_ERROR.
     *
     * @attention
     *  MUST ONLY BE USED to indicate an error in the internal implementation
     *  of the formula engine. NEVER throw this error code for wrong user
     *  input.
     *
     * @throws {ErrorCode}
     *  The FormulaUtils.INTERNAL_ERROR error code.
     */
    FormulaUtils.throwInternal = function (msg) {
        Utils.error(msg);
        throw FormulaUtils.INTERNAL_ERROR;
    };

    /**
     * Throws an UNSUPPORTED error code, if the passed matrix dimension is
     * outside the limits supported by the formula interpreter.
     *
     * @param {Dimension|Number} matrixDim
     *  The dimension of a matrix, or the number of rows in a matrix.
     *
     * @param {Number} [cols]
     *  The number of columns in a matrix. MUST be specified, if the first
     *  parameter is a number (row count). Ignored, if the first parameter is
     *  an instance of Dimension.
     *
     * @throws {ErrorCode}
     *  The UNSUPPORTED error code, if the passed matrix dimension is not
     *  inside the limits supported by the formula interpreter.
     */
    FormulaUtils.ensureMatrixDim = function (matrixDim, cols) {
        if (!Matrix.isValidDim(matrixDim, cols)) {
            throw FormulaUtils.UNSUPPORTED_ERROR;
        }
    };

    /**
     * Resolves the effective recalculation mode for the passed recalculation
     * modes.
     *
     * @param {String|Null} recalc1
     *  The first recalculation mode, either 'always', 'once', or null.
     *
     * @param {String|Null} recalc2
     *  The second recalculation mode, either 'always', 'once', or null.
     *
     * @returns {String|Null}
     *  The effective recalculation mode for the passed recalculation modes.
     */
    FormulaUtils.getRecalcMode = function (recalc1, recalc2) {
        if ((recalc1 === 'always') || (recalc2 === 'always')) { return 'always'; }
        if ((recalc1 === 'once') || (recalc2 === 'once')) { return 'once'; }
        return null;
    };

    // number formats ---------------------------------------------------------

    /**
     * Resolves the number format for adding two formatted numbers.
     *
     * @param {ParsedFormat|Null} parsedFormat1
     *  The first parsed number format, or null to indicate an invalid number
     *  format that cannot be combined with the other number format.
     *
     * @param {ParsedFormat|Null} parsedFormat2
     *  The second parsed number format, or null to indicate an invalid number
     *  format that cannot be combined with the other number format.
     *
     * @returns {ParsedFormat|Null}
     *  One of the passed format categories to be used for the sum of two
     *  formatted numbers; or the value null, if the two format categories
     *  cannot be combined.
     */
    FormulaUtils.combineParsedFormats = (function () {

        var NUMERIC_SET = Utils.makeSet(['number', 'scientific', 'currency', 'percent', 'fraction']);
        var DATETIME_SET = Utils.makeSet(['date', 'time', 'datetime']);

        return function (parsedFormat1, parsedFormat2) {

            // skip invalid categories (null values)
            if (!parsedFormat1 || !parsedFormat2) { return null; }

            // skip standard category
            if (parsedFormat1.isStandard()) { return parsedFormat2; }
            if (parsedFormat2.isStandard()) { return parsedFormat1; }

            // separate handling for numeric categories, and date/time categories
            var isNumeric1 = parsedFormat1.category in NUMERIC_SET;
            var isNumeric2 = parsedFormat2.category in NUMERIC_SET;
            var isDateTime1 = !isNumeric1 && (parsedFormat1.category in DATETIME_SET);
            var isDateTime2 = !isNumeric2 && (parsedFormat2.category in DATETIME_SET);

            // dates and times win over numeric categories
            if (isDateTime1 && isNumeric2) { return parsedFormat1; }
            if (isNumeric1 && isDateTime2) { return parsedFormat2; }

            // first of two numeric categories wins (TODO: more specific precedences?)
            if (isNumeric1 && isNumeric2) { return parsedFormat1; }

            // anything else cannot be combined (e.g. two dates/times)
            return null;
        };
    }());

    // debug ------------------------------------------------------------------

    /**
     * Writes all passed tokens to the browser console.
     *
     * @param {String} message
     *  A message string written before the formula tokens.
     *
     * @param {Array<Any>} tokens
     *  An array of formula tokens to be written to the browser console. The
     *  array elements can be of any type convertible to a string.
     */
    FormulaUtils.logTokens = FormulaUtils.isLoggingActive() ? function (message, tokens) {
        FormulaUtils.log(message + ': ' + tokens.join(' '));
    } : $.noop;

    // exports ================================================================

    return FormulaUtils;

});
