/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/timerhelper', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils'
], function (IteratorUtils, BaseObject, TimerMixin, DependencyUtils) {

    'use strict';

    // general options for background timers, for first-level loops (mapped by priority mode)
    var OUTER_TIMER_OPTIONS_MAP = {
        low:  { slice: DependencyUtils.TIMER_SLICE, interval: DependencyUtils.TIMER_SLICE_DELAY },
        high: { slice: DependencyUtils.PRIORITY_TIMER_SLICE, interval: DependencyUtils.PRIORITY_TIMER_SLICE_DELAY }
    };

    // general options for background timers, for emebdded loops (mapped by priority mode)
    var NESTED_TIMER_OPTIONS_MAP = {
        low:  _.extend({ delay: 'immediate' }, OUTER_TIMER_OPTIONS_MAP.low),
        high: _.extend({ delay: 'immediate' }, OUTER_TIMER_OPTIONS_MAP.high)
    };

    // class TimerHelper ======================================================

    /**
     * A small helper class to execute background loops with different timing
     * settings.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends TimerMixin
     */
    var TimerHelper = BaseObject.extend({ constructor: function (app) {

        // the number of nested background loops currently running
        var loopDepth = 0;

        // whether to run the background loops with higher priority
        var priority = 'low';

        var outerOptionsMap = _.mapObject(OUTER_TIMER_OPTIONS_MAP, function (options) {
            return _.extend({}, options, { infoString: 'DependencyManager.TimerHelper.iterate()', app: app });
        });

        var nestedOptionsMap = _.mapObject(NESTED_TIMER_OPTIONS_MAP, function (options) {
            return _.extend({}, options, { infoString: 'DependencyManager.TimerHelper.iterate()', app: app });
        });

        // base constructors --------------------------------------------------

        BaseObject.call(this);
        TimerMixin.call(this);

        // public methods -----------------------------------------------------

        /**
         * Changes the execution priority of the background loops.
         *
         * @param {String} newPriority
         *  The new priority mode. Must be one of 'low' or 'high'.
         *
         * @returns {TimerHelper}
         *  A reference to this instance.
         */
        this.setPriority = function (newPriority) {
            priority = newPriority;
            return this;
        };

        /**
         * Processes the passed iterator in a background loop.
         *
         * @param {Object} iterator
         *  The iterator to be processed.
         *
         * @param {Function} callback
         *  The callback function invoked for every iterator value.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise representing the background loop.
         */
        this.iterate = function (iterator, callback, context) {
            var options = ((loopDepth === 0) ? outerOptionsMap : nestedOptionsMap)[priority];
            loopDepth += 1;
            return this.iterateSliced(iterator, callback.bind(context), options).always(function () { loopDepth -= 1; });
        };

        /**
         * Invokes the passed callback function forever in a background loop.
         *
         * @param {Function} callback
         *  The callback function invoked repeatedly. Receives the zero-based
         *  index as first parameter. The return value Utils.BREAK will exit
         *  the loop.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise representing the background loop.
         */
        this.loop = function (callback, context) {
            var iterator = IteratorUtils.createIndexIterator(Number.POSITIVE_INFINITY);
            return this.iterate(iterator, callback, context);
        };

    } }); // class TimerHelper

    // exports ================================================================

    return TimerHelper;

});
