/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/dependencyutils', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/spreadsheet/utils/address'
], function (Config, Logger, Address) {

    'use strict';

    // small delays in Selenium tests and unit tests
    var FAST_MODE = Config.AUTOTEST || Config.UNITTEST;

    // static class DependencyUtils ===========================================

    var DependencyUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Duration for a single time slice of the background update task of the
     * dependency manager (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.TIMER_SLICE = FAST_MODE ? 300 : 150;

    /**
     * Duration between two time slices of the background update task of the
     * dependency manager (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.TIMER_SLICE_DELAY = FAST_MODE ? 10 : 100;

    /**
     * Duration for a single time slice of the background update task of the
     * dependency manager for high-priority mode (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.PRIORITY_TIMER_SLICE = FAST_MODE ? 300 : 200;

    /**
     * Duration between two time slices of the background update task of the
     * dependency manager for high-priority mode (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.PRIORITY_TIMER_SLICE_DELAY = FAST_MODE ? 10 : 50;

    /**
     * Delay for starting the background update task of the dependency manager
     * after receiving a document event (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.UPDATE_DEBOUNCE_DELAY = FAST_MODE ? 10 : 100;

    /**
     * Additional delay for the first background update task of the dependency
     * manager after import (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.IMPORT_UPDATE_DELAY = FAST_MODE ? 10 : 900;

    // logger interface -------------------------------------------------------

    // debug logger for the dependency manager
    Logger.extend(DependencyUtils, { enable: 'spreadsheet:log-deps', prefix: 'DEP' });

    // static methods ---------------------------------------------------------

    /**
     * Returns the cell key for the specified sheet UID and cell address.
     *
     * @param {String} sheetUid
     *  The UID of a sheet in a spreadsheet document.
     *
     * @param {Address} address
     *  The cell address to create a cell key for.
     *
     * @returns {String}
     *  A unique map key for the specified cell address.
     */
    DependencyUtils.getCellKey = function (sheetUid, address) {
        return sheetUid + '!' + address.key();
    };

    /**
     * Converts the passed formula key of a cell formula to a readable cell
     * address label for debug logging.
     *
     * @param {String} formulaKey
     *  A formula key of a any formula.
     *
     * @returns {String}
     * A readable cell address label for the passed key for debug logging, if
     * available; otherwise the passed formula key as-is.
     */
    DependencyUtils.getFormulaLabel = function (formulaKey) {
        var result = /(.+!)(\d+),(\d+)$/.exec(formulaKey);
        return result ? (result[1] + new Address(parseInt(result[2], 10), parseInt(result[3], 10))) : formulaKey;
    };

    /**
     * Returns a readable label for the passed defined name for debug logging.
     *
     * @param {NameModel} nameModel
     *  The model instance of a defined name.
     *
     * @returns {String}
     *  A readable label for the passed defined name for debug logging.
     */
    DependencyUtils.getNameLabel = function (nameModel) {
        var sheetModel = nameModel.getSheetModel();
        return 'name:' + (sheetModel ? (sheetModel.getName() + '!') : '') + nameModel.getLabel();
    };

    /**
     * Returns a readable label for the passed formatting rule for debug
     * logging.
     *
     * @param {RuleModel} ruleModel
     *  The model instance of a formatting rule.
     *
     * @returns {String}
     *  A readable label for the passed formatting rule for debug logging.
     */
    DependencyUtils.getRuleLabel = function (ruleModel) {
        return 'rule:' + ruleModel.getSheetModel().getName() + '!' + ruleModel.getUid();
    };

    /**
     * Prints all formula descriptors in the passed map to the browser console,
     * if dependency logging is active.
     *
     * @param {SimpleMap<FormulaDescriptor>} formulaMap
     *  A map with all formula descriptors to be logged to the browser console.
     */
    DependencyUtils.logFormulaMap = DependencyUtils.isLoggingActive() ? function (formulaMap) {
        formulaMap.forEach(function (formulaDesc, formulaKey) {
            DependencyUtils.log('key=' + DependencyUtils.getFormulaLabel(formulaKey) + ' references=' + formulaDesc.references + ' circular=' + formulaDesc.circular);
        });
    } : _.noop;

    // exports ================================================================

    return DependencyUtils;

});
