/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/manualcolorchart', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/chartstyleutil'
], function (Utils, ChartStyleUtil) {

    'use strict';

    var BACKGROUNDTRANSFORMATION = ChartStyleUtil.getBackgroundTransformation();

    var STYLESET = ChartStyleUtil.getStyleSet();

    var COLORSET = ChartStyleUtil.getColorSet();

    // class ManualColorHandler ===============================================

    function ManualColorHandler(chartModel, data) {

        var // self reference
            self = this,

            // the document model
            docModel = chartModel.getDocModel();

        // private methods ----------------------------------------------------

        /*  Apply tint/shade depending on the cycle index. The colors of leading
        814          series are darkened (color shade), the colors of trailing series are
        815          lightened (color tint). Shade/tint is applied in an exclusive range of
        816          -70% to 70%.
        817
        818          Example 1: 3 data series using single-color shading with accent color 1
        819          (e.g. automatic chart style #3). Shade/tint is applied per series.
        820          Shade/tint changes in steps of 140%/(<series_count+1) = 140%/4 = 35%,
        821          starting at -70%:
        822              Step 1: -70% -> Not used.
        823              Step 2: -35% -> Series 1 has 35% shade of accent color 1.
        824              Step 3:   0% -> Series 2 has pure accent color 1.
        825              Step 4:  35% -> Series 3 has 35% tint of accent color 1.
        826              Step 5:  70% -> Not used.
        827
        828          Example 2: 20 data series using accent color pattern (e.g. automatic
        829          chart style #2). Each color cycle has a size of 6 series (accent colors
        830          1 to 6). Shade/tint is applied per color cycle.
        831              Cycle #1: Series 1...6 are based on accent colors 1 to 6.
        832              Cycle #2: Series 7...12 are based on accent colors 1 to 6.
        833              Cycle #3: Series 13...18 are based on accent colors 1 to 6.
        834              Cycle #4: Series 19...20 are based on accent colors 1 to 2.
        835          Shade/tint changes in steps of 140%/(cycle_count+1) = 140%/5 = 28%,
        836          starting at -70%:
        837              Step 1: -70% -> Not used.
        838              Step 2: -42% -> Cycle #1 has 42% shade of accent colors 1...6
        839              step 3: -14% -> Cycle #2 has 14% shade of accent colors 1...6
        840              step 4:  14% -> Cycle #3 has 14% tint of accent colors 1...6
        841              step 5:  42% -> Cycle #4 has 42% tint of accent colors 1...6
        842              step 6:  70% -> Not used.
        843
        844      sal_Int32 nPhClr = maColorPattern[ static_cast< size_t >( nSeriesIdx % maColorPattern.size() ) ];
        845      size_t nCycleIdx = static_cast< size_t >( nSeriesIdx / maColorPattern.size() );
        846      size_t nMaxCycleIdx = static_cast< size_t >( mrData.mnMaxSeriesIdx / maColorPattern.size() );
        847      double fShadeTint = static_cast< double >( nCycleIdx + 1 ) / (nMaxCycleIdx + 2) * 1.4 - 0.7;
        848      if( fShadeTint != 0.0 )
        849      {
        850          Color aColor;
        851          aColor.setSrgbClr( nPhClr );
        852          aColor.addChartTintTransformation( fShadeTint );
        853          nPhClr = aColor.getColor( mrData.mrFilter.getGraphicHelper() );
        854      }
        855
        856      return nPhClr;
         */

        function makeShadeTintTrans(value) {
            if (value > 0) {
                return [{
                    type: 'tint',
                    value: (1 - value) * 100000
                }];
            } else if (value < 0) {
                return [{
                    type: 'shade',
                    value: (1 + value) * 100000
                }];
            }
        }

        function getColorOfPattern(meth, type, index, schemeColors, count) {

            var colorIndex = null;
            var relVariation = null;

            if (type === 'single') {
                relVariation = (index / count) - 0.5;
            } else if (type === 'group') {
                var group = Math.floor(index / schemeColors.length);
                var all = Math.floor(count / schemeColors.length);
                relVariation = group / all;

                if (count <= schemeColors.length * 2) {
                    //TODO: workaround for small charts, should be calculated correctly
                    relVariation *= 0.5;
                } else {
                    relVariation -= 0.5;
                }
            } else {
                Utils.warn('cant handle colorize type:' + type);
                return null;
            }

            if (meth === 'cycle') {
                colorIndex = index % schemeColors.length;
            } else {
                Utils.warn('cant handle color meth:' + meth);
                return null;
            }

            var color = schemeColors[colorIndex];

            var tmpColor = {};
            tmpColor.type = color.type;
            tmpColor.value = color.value;

            if (!relVariation) {
                relVariation = 0;
            }
            tmpColor.transformations = makeShadeTintTrans(relVariation * 1.4);
            return tmpColor;
        }

        function getColor(index, type, count) {
            var attributes = chartModel.getMergedAttributeSet(true);

            var colors = attributes.chart.chartColors;

            var result = null;

            if (colors) {
                result = getColorOfPattern(colors.meth, 'group', index, colors.schemeClr, count);
            } else if (attributes.chart.chartStyleId) {
                var styleId = attributes.chart.chartStyleId - 1;
                var colorSet = COLORSET[styleId % 8];

                var res = getColorOfPattern('cycle', colorSet.type, index, colorSet.colors, count);
                result = {
                    color: res,
                    type: type
                };
            }

            return result;
        }

        function getStyleSet(dataSeries, color) {
            var attributes = chartModel.getMergedAttributeSet(true);

            var result = color;
            var styleId = null;
            var styleSet = null;

            if (attributes.chart.chartStyleId) {
                styleId = attributes.chart.chartStyleId - 1;
                styleSet = STYLESET[Math.floor(styleId / 8)];
                dataSeries.bevelEnabled = styleSet.bevelEnabled;
            }

            if (dataSeries.seriesIndex === 0) {
                data.cssBackgroundColor = null;
                if (attributes.chart.chartStyleId && result) {

                    styleId = attributes.chart.chartStyleId - 1;
                    styleSet = STYLESET[Math.floor(styleId / 8)];

                    var givenStyle = styleSet.bg;
                    if (!givenStyle) {
                        givenStyle = { type: result.color.type, value: result.color.value, transformations: BACKGROUNDTRANSFORMATION };
                    }

                    data.cssBackgroundColor = docModel.getCssColor(givenStyle, 'fill');
                }
            }
        }

        function getVaryColor() {
            if (self.isVaryColorEnabled()) {
                var attributes = chartModel.getMergedAttributeSet(true);
                return attributes.chart.varyColors;
            }
            return false;
        }

        function applyColor(cjsPoint, fill, targetKey) {
            var cssColor;

            if (fill.type === 'none') {
                cssColor = 'transparent';
            } else {
                cssColor = docModel.getCssColor(fill.color, 'fill');
            }

            cjsPoint[targetKey] = cssColor;
        }

        // public functions ---------------------------------------------------

        this.isVaryColorEnabled = function () {
            if (data.series.length === 1) {
                return true;
            }
            var type = chartModel.getChartType().split(' ')[0];
            if (type === 'pie' || type === 'donut') {
                return true;
            }
            return false;
        };

        this.handleColor = function (shape, dataSeries, index, targetKey, count) {
            var vary = getVaryColor();
            var attributes = chartModel.getMergedAttributeSet(true);

            _.each(dataSeries.dataPoints, function (dataPoint) {
                dataPoint[targetKey] = null;
            });
            dataSeries[targetKey] = null;

            if (ChartStyleUtil.isAuto(shape.color)) {
                if (vary) {
                    dataSeries.color = null;
                    _(dataSeries.dataPoints).each(function (dataPoint, pi) {
                        var colorHolder = getColor(pi, shape.type, dataSeries.dataPoints.length);
                        if (pi === 0) {
                            getStyleSet(dataSeries, colorHolder);
                            if (colorHolder.type === 'none') {
                                //fallback for CanvasJS bug when series color is null and point color is set
                                dataSeries[targetKey] = 'transparent';
                            }
                        }
                        applyColor(dataPoint, colorHolder, targetKey);
                    });
                } else {
                    shape = getColor(index, shape.type, count);
                }
            }

            if (!ChartStyleUtil.isAuto(shape.color)) {
                applyColor(dataSeries, shape, targetKey);
            }

            if (!vary) {
                getStyleSet(dataSeries, shape);
            }
            if (index === 0) {
                var bg = attributes.fill.color;
                if (bg && !ChartStyleUtil.isAuto(bg)) {
                    data.cssBackgroundColor = docModel.getCssColor(bg, 'fill');
                }
            }
        };

        this.getBackgroundColor = function (chartStyleId) {
            var styleId = chartStyleId - 1;
            var index = Math.floor(styleId / 8);
            var styleSet = STYLESET[index];

            var givenStyle = styleSet && styleSet.bg;
            if (!givenStyle) {
                var colorSet = COLORSET[styleId % 8];
                if (colorSet) {
                    var color = getColorOfPattern('cycle', 'single', 1, colorSet.colors, 3);
                    givenStyle = { type: color.type, value: color.value, transformations: BACKGROUNDTRANSFORMATION };
                } else {
                    givenStyle = { type: 'auto' };
                }
            }
            if (!givenStyle.fallbackValue) {
                givenStyle.fallbackValue = docModel.getCssColor(givenStyle, 'fill');
            }
            return givenStyle;
        };

        // initialization -----------------------------------------------------

        data.colorSet = null;

    } // class ManualColorHandler

    // exports ================================================================

    return ManualColorHandler;

});
