/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/axismodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/drawing/titlemodel',
    'io.ox/office/spreadsheet/model/drawing/gridlinemodel'
], function (Utils, ChartStyleUtil, AttributedModel, Operations, TitleModel, GridlineModel) {

    'use strict';

    // class AxisModel ========================================================

    /**
     * Representation of a simgle axis in a chart drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     */
    var AxisModel = AttributedModel.extend({ constructor: function (chartModel, axisId) {

        var self = this;
        var titleModel = null;
        var gridModel = null;
        var format = null;

        var dataAxis = chartModel.getModelData()['axis' + axisId.toUpperCase()];
        if (dataAxis) {
            titleModel = new TitleModel(chartModel, axisId, dataAxis, 'title');
            gridModel = new GridlineModel(chartModel, axisId, {}, dataAxis);

            dataAxis.labelFormatter = function (info) { return info.label || formatNumber(info.value); };
        }

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chartModel.getDocModel(), null, { families: 'axis line character' });

        // private methods ----------------------------------------------------

        function formatNumber(value) {
            if (!chartModel) {
                Utils.warn('no chartmodel in AxisModel.formatNumber()');
                return;
            }
            var formatter = chartModel.getDocModel().getNumberFormatter();
            var realValue = value * Math.pow(10, format.exp);
            if (format && format.format) {
                return formatter.formatValue(format.format, realValue);
            } else {
                return formatter.formatStandardNumber(realValue, 8);
            }
        }

        function isHorizontal() {
            return chartModel.getAxisIdForDrawing(axisId) === 'x';
        }

        function refreshMinMax() {
            var data = chartModel.getModelData();
            if (!data.series.length) {
                return;
            }

            // fix for Bug 46910 & Bug 49574
            if (isHorizontal()) {
                delete dataAxis.labelAngle;
            } else {
                dataAxis.labelAngle = 0;
            }

            if (axisId === 'x') {
                refreshAxisX(data);
            } else {
                refreshAxisY(data);
            }
        }

        function makeMin(min, max) {
            if ((!format || !format.date) && (max === min || (min / max < 5 / 6))) {
                return 0;
            }
            return min - ((max - min) / 2);
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function distance(a, b) {
            return Math.abs(a - b);
        }

        function updateInterval(number) {
            var normalized = Utils.normalizeNumber(number);

            var distances = [];
            distances.push({ value: 1, dist: distance(normalized.mant, 1) });
            distances.push({ value: 2, dist: distance(normalized.mant, 2) });
            distances.push({ value: 5, dist: distance(normalized.mant, 5) });
            distances.push({ value: 10, dist: distance(normalized.mant, 10) });

            var newMant = null;
            var minDist = 999999;

            distances.forEach(function (d) {
                if (d.dist < minDist) {
                    minDist = d.dist;
                    newMant = d.value;
                }
            });

            return newMant * Math.pow(10, normalized.exp);
        }

        function updateMinMax(minValue, maxValue, precision, intervalCount) {
            if (minValue < maxValue) {
                var interval = (maxValue - minValue) / intervalCount;
                interval = round(interval, precision - 1);
                interval = updateInterval(interval);

                if (interval === 0) {
                    interval = null;
                    dataAxis.minimum = minValue;
                    dataAxis.maximum = maxValue;
                    dataAxis.interval = null;
                } else {
                    var intCount = 1 + ((maxValue - minValue) / interval);

                    if (Math.abs(maxValue) < Math.abs(minValue)) {
                        maxValue = Utils.roundUp(maxValue, interval);
                        minValue = maxValue - (interval * intCount);
                    } else {
                        minValue = Utils.roundDown(minValue, interval);
                        maxValue = minValue + (interval * intCount);
                    }
                }
                setMinMax(minValue, maxValue, interval);
            }
        }

        function setMinMax(min, max, interval) {
            if (!interval || !format || !isFinite(min) || !isFinite(max)) {
                delete dataAxis.minimum;
                delete dataAxis.maximum;
                delete dataAxis.interval;
                dataAxis.stripLines = [];
                return;
            }

            if (min < 0 && format && format.format && format.format.category === 'date') { min = 0; }

            dataAxis.minimum = min;
            dataAxis.maximum = max;
            dataAxis.interval = interval;

            var stripLines = dataAxis.stripLines;
            var i = 0;

            var index = 0;
            var steps = (dataAxis.maximum - dataAxis.minimum) / dataAxis.interval;
            var lastLine = null;
            var line = null;

            if (chartModel.isXYType() && isHorizontal()) {
                i = 1; // not enough space for first label
            }

            for (; i <= steps; i += 1) {
                var value = dataAxis.minimum + i * dataAxis.interval;

                line = stripLines[index];
                if (!line) {
                    line = {
                        value: 0,
                        color: 'transparent',
                        label: '',
                        labelBackgroundColor: 'transparent',
                        labelPlacement: 'outside'
                    };
                    stripLines[index] = line;
                }
                line.labelFontColor = 'transparent';
                line.label = '';
                line.value = value;
                line.labelPlacement = 'outside';
                line.label = formatNumber(value);

                if (!line.label || (lastLine && lastLine.label === line.label)) {
                    line.label = '';
                } else {
                    lastLine = line;
                }

                index++;
            }

            // workaround for Bug 49637
            if (!isHorizontal()) {
                dataAxis.minimum -= dataAxis.interval * 0.001;
                dataAxis.maximum += dataAxis.interval * 0.001;
            }

            for (; index < stripLines.length; index++) {
                line = stripLines[index];
                line.labelFontColor = 'transparent';
                line.label = '';
            }
        }

        function refreshAxisX(data) {
            var area = chartModel.getMergedAttributeSet(true).chart.type.indexOf('area') === 0;
            if (!data.linearX && (chartModel.isXYType() || area)) {
                var minValue = Number.POSITIVE_INFINITY;
                var maxValue = Number.NEGATIVE_INFINITY;

                _.each(data.series, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var value = dataPoint[axisId];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });

                if (chartModel.isXYType()) {
                    var intervalCount = Utils.minMax(data.series.length, 2, 8);
                    var diff = maxValue - minValue;
                    maxValue +=  diff / intervalCount;
                    minValue -=  diff / intervalCount;
                }

                if (!isFinite(minValue) || !isFinite(maxValue)) {
                    setMinMax();
                    return;
                }
                if (minValue === maxValue) {
                    maxValue = minValue + 0.5;
                    minValue -= 0.5;
                }

                updateMinMax(minValue, maxValue, 2, 4);
            } else {
                setMinMax(null, null, null);
            }
        }

        function round(value, precision) {
            if ((precision > 0) && (precision < 20)) {
                return parseFloat(value.toPrecision(precision));
            }
            return value;
        }

        function refreshAxisY(data) {
            var chartAttrs = chartModel.getMergedAttributeSet(true).chart;
            var stacking = chartAttrs.stacking;

            var givenMin = null;
            var givenMax = null;

            var axisInfo = self.getMergedAttributeSet(true);
            if (axisInfo) {
                if (axisInfo.min !== 'auto') {
                    givenMin = axisInfo.min;
                }
                if (axisInfo.max !== 'auto') {
                    givenMax = axisInfo.max;
                }
            }

            var minValue = Number.POSITIVE_INFINITY;
            var maxValue = Number.NEGATIVE_INFINITY;

            if (stacking === 'percentStacked' || (stacking === 'stacked' && data.series.length > 1)) {
                var first = data.series[0];
                _.times(first.dataPoints.length, function (j) {
                    var posValue = 0;
                    var negValue = 0;
                    _.each(data.series, function (dd) {
                        var jValue = dd.dataPoints[j];
                        if (!jValue) { return; }
                        var useValue = jValue[axisId];
                        if (useValue > 0) {
                            posValue += useValue;
                        } else if (useValue < 0) {
                            negValue += useValue;
                        }
                    });
                    minValue = Math.min(minValue, negValue);
                    maxValue = Math.max(maxValue, posValue);
                });
            } else {
                _.each(data.series, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var value = dataPoint[axisId];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });
            }

            if (chartModel.isXYType()) {
                var diff = maxValue - minValue;
                maxValue += diff / 8;
                minValue -= diff / 8;
            } else if (stacking === 'percentStacked') {
                format = {
                    exp: -2,
                    format: chartModel.getDocModel().getNumberFormatter().getParsedFormat('0%', { grammarId: 'ui' })
                };
                //Trial & error...
                if (maxValue >= 0 && minValue >= 0) {
                    // scenario 1
                    minValue = givenMin || 0;
                    maxValue = givenMax || 110;
                } else if (maxValue <= 0 && minValue <= 0) {
                    // scenario 2
                    maxValue = givenMax || 0;
                    minValue = givenMin || -110;
                } else {
                    // scenario 3
                    maxValue =  givenMax || 110;
                    if (-minValue > maxValue) {
                        minValue =  givenMin || -110;
                    } else {
                        minValue =  givenMin || (-100 * ((-minValue) / maxValue));
                    }
                }
            } else {
                if (maxValue >= 0 && minValue >= 0) {
                    // scenario 1
                    minValue = givenMin || makeMin(minValue, maxValue);
                    maxValue = givenMax || makeMax(minValue, maxValue);
                } else if (maxValue <= 0 && minValue <= 0) {
                    // scenario 2
                    maxValue = givenMax || -makeMin(-maxValue, -minValue);
                    minValue = givenMin || -makeMax(-maxValue, -minValue);
                } else {
                    // scenario 3
                    var ma = maxValue;
                    var mi = minValue;
                    maxValue =  givenMax || makeMax(mi, ma);
                    minValue =  givenMin || -makeMax(-ma, -mi);
                }
            }

            if (!isFinite(minValue) || !isFinite(maxValue)) {
                setMinMax();
                return;
            }

            if (minValue === maxValue) {
                maxValue = minValue + 0.5;
                minValue -= 0.5;
            }

            var precision = 2;
            if (format && format.date) {
                precision = 0;
            }

            if (Math.abs(maxValue - minValue) > 0.00000000001) {
                minValue = round(minValue, precision);
                maxValue = round(maxValue, precision);
            }

            updateMinMax(minValue, maxValue, precision, 8);
        }

        // public methods -----------------------------------------------------

        this.refreshInfo = function () {
            refreshMinMax();

            if (!dataAxis) { return; }

            var attrs = this.getMergedAttributeSet(true);

            ChartStyleUtil.handleLineProps(chartModel, attrs.line, dataAxis, 'line');
            ChartStyleUtil.handleLineProps(chartModel, attrs.line, dataAxis, 'tick');

            if (attrs.axis.label) {
                ChartStyleUtil.handleCharacterProps(chartModel, attrs.character, dataAxis, 'label');
            } else {
                dataAxis.labelFontColor = 'transparent';
                dataAxis.labelFontSize = 1;
            }
            dataAxis.labelPlacement = 'outside';

            var stripLines = dataAxis.stripLines;
            var overwriteLabels = stripLines.length > 0;

            if (overwriteLabels) {
                _.each(stripLines, function (stripLine) {
                    stripLine.labelFontColor    = dataAxis.labelFontColor;
                    stripLine.labelFontSize     = dataAxis.labelFontSize;
                    stripLine.labelFontFamily   = dataAxis.labelFontFamily;
                    stripLine.labelFontWeight   = dataAxis.labelFontWeight;
                    stripLine.labelFontStyle    = dataAxis.labelFontStyle;
                });
                dataAxis.labelFontColor = 'transparent';

            }

            gridModel.refreshInfo();
            titleModel.refreshInfo();
        };

        this.setFormat = function (newFormat) {
            format = newFormat;
        };

        this.getGrid = function () {
            return gridModel;
        };

        this.getTitle = function () {
            return titleModel;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this chart axis.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @returns {AxisModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {

            // restore this axis
            if (this.hasExplicitAttributes()) {
                var properties = { axis: axisId, attrs: this.getExplicitAttributeSet(false) };
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, properties, { undo: true });
            }

            // restore the axis title, and the grid line settings
            if (titleModel) { titleModel.generateRestoreOperations(generator, position); }
            if (gridModel) { gridModel.generateRestoreOperations(generator, position); }

            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source link of the axis title object.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {AxisModel}
         *  A reference to this instance.
         */
        this.generateUpdateFormulaOperations = function (generator, position, changeDesc) {
            if (titleModel) { titleModel.generateUpdateFormulaOperations(generator, position, changeDesc); }
            return this;
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            if (dataAxis) { delete dataAxis.labelFormatter; }
            if (gridModel) { gridModel.destroy(); }
            if (titleModel) { titleModel.destroy(); }
            dataAxis = chartModel = gridModel = titleModel = null;
        });

    } }); // class AxisModel

    // exports ================================================================

    return AxisModel;

});
