/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/sortmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/dialog/customsortdialog',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, SheetUtils, CustomSortDialog, gt) {

    'use strict';

    // mix-in class SortMixin =================================================

    /**
     * Implementations of all controller items for sorting selected ranges,
     * intended to be mixed into a document controller instance.
     *
     * @constructor
     */
    function SortMixin(app, docView) {

        // self reference (the document controller)
        var self = this;

        // the model and collections of the active sheet
        var sheetModel = null;
        var cellCollection = null;

        // address of the cell range to be sorted
        var sortRange = null;

        // should the selected range be checked
        var doCheck = true;

        // prevent resetting the sort options (in case of auto expand "change:selection")
        var preventReset = false;

        // state of sorting
        var state = null;

        // private methods ----------------------------------------------------

        /**
         * Initializes all sheet-dependent class members according to the
         * current active sheet.
         */
        function changeActiveSheetHandler() {
            sheetModel = docView.getSheetModel();
            cellCollection = sheetModel.getCellCollection();
        }

        /**
          * Resets all sort-options by sort-type
          *
          * @param {String} sorttype
          *  The type on which the reset should be executed
          */
        function resetSortOptions() {
            if (preventReset === true) {
                preventReset = false;
            } else {
                state = null;
                self.update();
            }
        }

        /**
         * Checks whether the cell selection is sortable, especially unlocked.
         *
         * @returns {jQuery.Promise}
         *  A resolved promise, if the selection is sortable; otherwise a
         *  promise that has been rejected with an appropriate error code.
         */
        function ensureSortableSelection() {

            // multi-selection not allowed
            var promise = (docView.getSelectedRanges().length > 1) ? SheetUtils.makeRejected('sort:multiselect') : $.when();

            // check that the selected range is not locked
            promise = promise.then(function () {
                return docView.ensureUnlockedSelection({ lockTables: 'header', lockMatrixes: 'full' });
            });

            // further checks
            promise = promise.then(function () {

                // the selected range to be sorted
                sortRange = docView.getActiveRange();

                // get out here, if selection is blank (nothing to sort)
                if (cellCollection.areRangesBlank(sortRange)) {
                    return SheetUtils.makeRejected('sort:blank');
                }

                // check the maximum supported sort size
                if ((sortRange.cols() > SheetUtils.MAX_SORT_LINES_COUNT) || (sortRange.rows() > SheetUtils.MAX_SORT_LINES_COUNT)) {
                    return SheetUtils.makeRejected('sort:overflow');
                }
            });

            // notify user about any problems
            return docView.yellOnFailure(promise);
        }

        function doExpand(newRange) {

            preventReset = true;

            // check that the new range does not have too many columns/rows
            var promise = null;
            if (newRange.cols() > SheetUtils.MAX_CHANGE_COLS_COUNT) {
                promise = SheetUtils.makeRejected('cols:overflow');
            } else if (newRange.rows() > SheetUtils.MAX_CHANGE_ROWS_COUNT) {
                promise = SheetUtils.makeRejected('rows:overflow');
            } else {
                // select the new range
                docView.selectRange(newRange, { active: docView.getActiveCell() });
                sortRange = newRange;
                promise = $.when();
            }

            return docView.yellOnFailure(promise);
        }

        /**
         * Check if the selection is expandable or actually must expand.
         */
        function expandSelection(options) {

            // exit when the expand-possibility should not be check anymore
            if (!doCheck) { return $.when(); }

            var newRange = cellCollection.findContentRange(sortRange);

            // the selection is only within one row/col. Ask the user whether expand selection or not.
            if ((sortRange.singleCol() !== sortRange.singleRow()) && newRange.differs(sortRange)) {

                // ask the user to expand the current selection
                var promise = docView.showQueryDialog(gt('Sorting'), gt('Expand the selection?'), { cancel: true });

                promise = promise.then(function (answer) {

                    // expand the selection automatically to current used rectangle
                    if (answer) { doExpand(newRange); }

                    // stop checking the selected range. The user has made a decision already for this selection.
                    doCheck = false;
                });

                if (Utils.getBooleanOption(options, 'custom', false)) {
                    promise = promise.then(function () {
                        resetSortOptions();
                        return new CustomSortDialog(docView).show();
                    });
                }

                return promise;
            }

            // only one cell is selected. Expand selection automatically (don't ask the user).
            if (sortRange.single() && newRange.differs(sortRange)) {
                // expand the selection automatically to current used rectangle
                return doExpand(newRange);
            }

            return $.when();
        }

        /**
         * Returns whether sorting is currently available according to the
         * selection in the active sheet.
         *
         * @returns {Boolean}
         *  Whether sorting is currently available.
         */
        function isSortAvailable() {
            // only one single range can be sorted
            return docView.getSelectedRanges().length <= 1;
        }

        /**
         * Returns the current sort state.
         *
         * @returns {Object}
         *  The current sort state.
         */
        function getSortState() {
            return { orderRules: [{ orderBy: (state === null) ? 'ascending' : state }] };
        }

        /**
         * Prepares and executes the sorting
         *
         * @param {String} sorttype
         *  The type on which currently be sorted ('standard', 'custom')
         *
         * @param {String} order
         *  The order-method ('ascending', 'descending', ...)
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the selected cell range has
         *  been sorted.
         */
        function sortCellRange(options) {

            // get out here, if there is something unexpected
            if (Utils.getBooleanOption(options, 'problem', false)) { return $.when(); }

            var pushToggleState = false;
            var orderBy = null;

            // check the selection, and initialize the variable 'sortRange'
            var promise = ensureSortableSelection();

            promise = promise.then(function () {

                var rules = Utils.getArrayOption(options, 'orderRules');

                // detect order-direction in case of 'toggle'
                if ((rules.length === 1) && (rules[0].index === undefined) && (rules[0].orderBy === 'toggle')) {
                    // get old order-direction, or use 'asc' by default
                    orderBy = state || 'ascending';

                    // prepare updating the toggle button state
                    pushToggleState = true;

                    options.orderRules[0].orderBy = orderBy;
                }

                // open the dialog to ask the user whether the range should expand or not
                return expandSelection();
            });

            promise = promise.then(function () {
                return sheetModel.createAndApplyOperations(function (generator) {
                    return cellCollection.generateSortOperations(generator, sortRange, docView.getActiveCell(), options);
                }, { storeSelection: true });
            });

            promise.done(function () {
                // set new/current sort state to toggle button
                state = pushToggleState ? (orderBy === 'ascending') ? 'descending' : 'ascending' : null;
            });

            return promise;
        }

        /**
         * Checks the sort environment. Decides, whether the selection should
         * expand, or in doubt, asks the user.
         */
        function showSortMenu() {
            // check the selection, and initialize the variable 'sortRange'
            var promise = ensureSortableSelection();

            promise = promise.then(function () {
                return expandSelection({ custom: true });
            });

            promise = promise.then(function () {
                resetSortOptions();
                return new CustomSortDialog(docView).show();
            });

            return promise.done(function (sortObj) {
                if (!sortObj.problem) {
                    app.getController().executeItem('cell/sort', sortObj);
                }
            });
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            'cell/sort': {
                parent: 'sheet/operation/unlocked/cell',
                enable: isSortAvailable,
                get: getSortState,
                set: sortCellRange
            },

            'cell/sort/menu': {
                parent: 'sheet/operation/unlocked/cell',
                set: showSortMenu,
                preserveFocus: true
            }
        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', changeActiveSheetHandler);

        // reset the state, if the selection changes (sort settings won't be saved)
        this.listenTo(docView, 'change:selection', function () {
            resetSortOptions();
            // reset also the expand decision
            doCheck = true;
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = self = app = null;
            sheetModel = cellCollection = null;
        });

    } // class SortMixin

    // exports ================================================================

    return SortMixin;

});
