/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/presentation/view/dialog/insertfooterdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/textframework/view/controls',
    'gettext!io.ox/office/presentation/main',
    'less!io.ox/office/presentation/view/dialog/presentationdialogs'
], function (Utils, DateUtils, Forms, Dialogs, LocaleData, Controls, gt) {

    'use strict';

    // Global constants and variables
    var insertFieldTitle = /*#. Insert footer fields from dialog menu*/ gt('Footer'),
        dialogWidth = Utils.SMALL_DEVICE ? 320 : 420,
        // container node for first column (category drop-down, fields type list)
        $controlArea = null,
        dateCheckbox = null,
        dateContainer = null,
        footerTxtContainer = null,
        slideNumCheckbox = null,
        footerTxtCheckbox = null,
        autoRadioBtn = null,
        fixedRadioBtn = null;

    // class InsertFooterDialog ===============================================

    /**
     * The insert date field modal dialog.
     * Provides localized date formats available for inserting into Presentation document.
     *
     *
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends Dialogs.ModalDialog
     *
     * @param {PresentationView} view
     *  The view instance containing this editor instance.
     * @param {Array} [dialogOptions]
     *  Options for filling the dialog
     * @param {Boolean} [dialogOptions.ftr]
     *  Footer placeholder exists on slide
     * @param {String} [dialogOptions.ftrText]
     *  Text inside footer placeholder
     * @param {Boolean} [dialogOptions.sldNum]
     *  Slide number placeholder exists on slide
     * @param {Boolean} [dialogOptions.dt]
     *  Datetime placeholder exists on slide
     * @param {String} [dialogOptions.dtType]
     *  Datetime placeholder contains date field and has this field type.
     * @param {String} [dialogOptions.dtText]
     *  Datetime placeholder contains fixed date as text.
     */
    function InsertFooterDialog(view, dialogOptions) {
        var
            self = this,
            model = view.getDocModel(),
            numberFormatter = model.getNumberFormatter(),
            // fetch locale date formats
            categoryCodesDate = numberFormatter.getCategoryCodes('date');

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, { title: insertFieldTitle, width: dialogWidth });

        // private methods ----------------------------------------------------

        /**
         * Create current date(time) value string with passed format type.
         *
         * @param {String} formatCode
         * @return {String}
         */
        function getDateTimeRepresentation(formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        }

        /**
         * Helper function for enabling or disabling the specified DOM elements.
         *
         * @param {jQuery} nodes
         *  The DOM elements to be manipulated. If this object is a jQuery
         *  collection, uses all nodes it contains.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the DOM elements.
         */
        function enableNodes(nodes, state) {
            nodes.toggleClass(Forms.DISABLED_CLASS, !state);
            if (state) {
                nodes.removeAttr('aria-disabled');
                nodes.prop('tabindex', 0);
            } else {
                nodes.attr('aria-disabled', true);
                nodes.prop('tabindex', -1);
            }
        }

        /**
         * Generate field values for selected value (date), and insert them into list.
         *
         * returns {String} generated html string for select box
         */
        function populateFieldFormats() {
            var localDateFormatList = [];

            if (categoryCodesDate.length) {
                _.each(categoryCodesDate, function (formatCode, index) {
                    localDateFormatList.push({ value: 'datetime' + (index + 1), label: getDateTimeRepresentation(formatCode.value) });
                });
            } else {
                localDateFormatList.push({ value: LocaleData.SHORT_DATE, label: getDateTimeRepresentation(LocaleData.SHORT_DATE) },
                        { value: 'datetime1', label: getDateTimeRepresentation(LocaleData.LONG_DATE) });
            }

            return Forms.createSelectBoxMarkup(localDateFormatList);
        }

        /**
         * Utility function for extracting object with given type from array of objects.
         *
         * @param {Array} optionsArray
         *  Array of objects.
         * @param {String} type
         *  Type of object which we are searching.
         * @returns {Object|undefined}
         */
        function extractByType(optionsArray, type) {
            return _.findWhere(optionsArray, { type: type });
        }

        /**
         * Initialize controls of the dialog.
         */
        function initControls() {
            var dateObject = extractByType(dialogOptions, 'dt');
            var isDate = !!dateObject;
            var isAutoDate = (dateObject && dateObject.automatic) || false;
            var slideNumObject = extractByType(dialogOptions, 'sldNum');
            var isSlideNum = !!slideNumObject;
            var footerObject = extractByType(dialogOptions, 'ftr');
            var isFooter = !!footerObject;
            var isAutoRadioEnabled = isDate ? isAutoDate : true;
            var isFixedRadioEnabled = isDate ? !isAutoDate : false;

            var dateTimeBtn = Forms.createButtonMarkup({ attributes: { class: 'date-time-btn', role: 'checkbox' }, label: gt('Date and time') });
            var slideNumberBtn = Forms.createButtonMarkup({ attributes: { class: 'slide-number-btn', role: 'checkbox' }, label: gt('Slide number') });
            var slideNumContainer = $('<div class="slidenum-container">').append(slideNumberBtn);
            var footerBtn = Forms.createButtonMarkup({ attributes: { class: 'footer-btn', role: 'checkbox' }, label: gt('Footer') });
            var fixedDateTextArea = Forms.createInputMarkup({ attributes: { disabled: true } });
            var autoRadio = Forms.createButtonMarkup({ attributes: { class: 'automatic-radio dradio', role: 'radio' }, label: gt('Update automatically') });
            var fixedRadio = Forms.createButtonMarkup({ attributes: { class: 'fixed-radio dradio', role: 'radio' }, label: gt('Fixed date') });

            dateContainer = $('<div class="date-container">').append(autoRadio, populateFieldFormats(), fixedRadio, fixedDateTextArea);
            // depending from auto or fixed date, disable select and input fields
            dateContainer.find('select').prop('disabled', !isDate || !isAutoDate).toggleClass('disabled', !isDate || !isAutoDate).val((dateObject && dateObject.dtType) || 'datetime1');
            dateContainer.find('input').prop('disabled', !isDate || isAutoDate).toggleClass('disabled', !isDate || isAutoDate);
            if (isDate && !isAutoDate) {
                dateContainer.find('input').val(dateObject && dateObject.dtText);
            } else {
                dateContainer.find('input').val(dateContainer.find('option:selected').text());
            }
            // if date chexbox is checked, enable auto and fixed radios
            enableNodes(dateContainer.find('.dradio'), isDate);

            var footerTxtTextArea = Forms.createInputMarkup({ attributes: !isFooter ? { disabled: true } : {} });
            footerTxtContainer = $('<div class="footertxt-container">').append(footerBtn, footerTxtTextArea);
            footerTxtContainer.find('input').val(footerObject && footerObject.ftrText);

            // append all containers to control area
            $controlArea.append(dateTimeBtn, dateContainer, footerTxtContainer, slideNumContainer);

            // date checkbox
            dateCheckbox = $controlArea.find('.date-time-btn');
            Forms.checkButtonNodes(dateCheckbox, isDate, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(dateCheckbox);
            // auto date radio
            autoRadioBtn = $controlArea.find('.automatic-radio');
            Forms.checkButtonNodes(autoRadioBtn, isAutoRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(autoRadioBtn);
            // fixed date radio
            fixedRadioBtn = $controlArea.find('.fixed-radio');
            Forms.checkButtonNodes(fixedRadioBtn, isFixedRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(fixedRadioBtn);

            // footer text checkbox
            footerTxtCheckbox = $controlArea.find('.footer-btn');
            Forms.checkButtonNodes(footerTxtCheckbox, isFooter, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(footerTxtCheckbox);
            // slide number checkbox
            slideNumCheckbox = $controlArea.find('.slide-number-btn');
            Forms.checkButtonNodes(slideNumCheckbox, isSlideNum, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(slideNumCheckbox);
        }

        // public methods -----------------------------------------------------

        /**
         * Executes the dialog and performs all necessary steps, if the
         * user ended the dialog by clicking the Ok button
         *
         * @returns {jQuery} $.promise
         *  The resolved/rejected promise of the executed dialog
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).done(function (action) {
                var selectedOption;
                var formatIndex;
                var obj = null;
                var checkedOptions = [];
                var diffContent;
                var diffState;

                if (Forms.isCheckedButtonNode(dateCheckbox)) {
                    if (Forms.isCheckedButtonNode(autoRadioBtn)) {
                        selectedOption = $controlArea.find('option:selected');
                        formatIndex = selectedOption.length ? (selectedOption.index() + 1) : null;
                        obj = { type: 'dt', automatic: true, formatIndex: formatIndex, representation: selectedOption[0].text };
                    } else {
                        obj = { type: 'dt', automatic: false, representation: dateContainer.find('input').val() };
                    }
                    checkedOptions.push(obj);
                }
                if (Forms.isCheckedButtonNode(slideNumCheckbox)) {
                    checkedOptions.push({ type: 'sldNum' });
                }
                if (Forms.isCheckedButtonNode(footerTxtCheckbox)) {
                    checkedOptions.push({ type: 'ftr', ftrText: footerTxtContainer.find('input').val() });
                }

                switch (action) {
                    case 'ok':
                        diffContent = model.getFieldManager().diffInputContent(dialogOptions, checkedOptions);
                        diffState = model.getFieldManager().diffStateCheckboxes(dialogOptions, checkedOptions);
                        model.getFieldManager().insertFooter(diffContent, diffState);
                        break;
                    case 'applyall':
                        model.getFieldManager().insertAllFooters(checkedOptions);
                }
            }).always(function () {
                // blocking keyboard input during applying of operations
                model.setBlockKeyboardEvent(false);

                self = model = view = numberFormatter = null;
            });
        });

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        view.closeDialogOnReadOnlyMode(this);

        $controlArea = $('<div class="control-area">');

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-presentation-dialog-insert-footer')
            .toggleClass('mobile', !!Utils.SMALL_DEVICE)
            .append($controlArea);

        // create the layout of the dialog
        initControls();

        // set initial control values and initial focus
        this.on('show', function () {
            // blocking keyboard input during applying of operations
            model.setBlockKeyboardEvent(true);

            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus should not be in an input element
            Utils.setFocus(self.getOkButton());
        });

        // add additional button
        this.addButton('applyall', gt('Apply to all'));
        var applyallBtn = this.getButton('applyall');
        applyallBtn.detach();
        applyallBtn.addClass('btn-apply-all btn-primary');
        this.getFooter().append(applyallBtn);
        this.getOkButton().children().html(gt('Apply'));

        // ensure proper button order on small devices, because base dialog makes reorder
        if (Utils.SMALL_DEVICE) {
            this.on('beforeshow', function () {
                applyallBtn.parent().detach();
                this.getFooter().children('.row').append(applyallBtn.parent());
            });
        }

        // listen on checkbox to change icon
        dateCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
            // enable/disable date subgroup
            enableNodes(dateContainer.find('.dradio'), Forms.isCheckedButtonNode(this));
            dateContainer.find('select, input').prop('disabled', !Forms.isCheckedButtonNode(this)).toggleClass('disabled', !Forms.isCheckedButtonNode(this));
            if (Forms.isCheckedButtonNode(autoRadioBtn)) {
                dateContainer.find('select').prop('disabled', !Forms.isCheckedButtonNode(this)).toggleClass('disabled', !Forms.isCheckedButtonNode(this));
                dateContainer.find('input').prop('disabled', true).addClass('disabled');
            } else {
                dateContainer.find('input').prop('disabled', !Forms.isCheckedButtonNode(this)).toggleClass('disabled', !Forms.isCheckedButtonNode(this));
                dateContainer.find('select').prop('disabled', true).addClass('disabled');
            }
        });
        slideNumCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
        });
        footerTxtCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
            // enable/disable footer textarea
            footerTxtContainer.find('input').prop('disabled', !Forms.isCheckedButtonNode(this));

        });

        autoRadioBtn.on('click', function () {
            Forms.checkButtonNodes(fixedRadioBtn, false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            dateContainer.find('select').prop('disabled', false).removeClass('disabled');
            dateContainer.find('input').prop('disabled', true).addClass('disabled');
        });

        fixedRadioBtn.on('click', function () {
            Forms.checkButtonNodes(autoRadioBtn, false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            dateContainer.find('input').prop('disabled', false).removeClass('disabled');
            dateContainer.find('select').prop('disabled', true).addClass('disabled');
        });

        // clean up after closing
        this.on('close', function () {
            $controlArea = dateCheckbox = dateContainer = footerTxtContainer = slideNumCheckbox = footerTxtCheckbox = autoRadioBtn = fixedRadioBtn = null;
            self = model = numberFormatter = categoryCodesDate = null;
        });

    } // class InsertFooterDialog

    // exports ================================================================

    // derive this class from class Dialogs.ModalDialog
    return Dialogs.ModalDialog.extend({ constructor: InsertFooterDialog });

});
