/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/updatedocumentmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom'
], function (DrawingFrame, Config, Utils, DOM) {

    'use strict';

    // mix-in class UpdateDocumentMixin ======================================

    /**
     * A mix-in class for the 'updateDocumentFormatting' function.
     *
     * @param app
     *  The application object.
     *
     * @constructor
     */
    function UpdateDocumentMixin(app) {

        var // self reference for local functions
            self = this,
            // the page node
            pagediv = null;

        // private methods ----------------------------------------------------

        /**
         * Dumps profiling information to browser console.
         */
        function dumpProfilingInformation() {

            var tableCount,
                cellCount,
                paragraphCount,
                spanCount,
                drawingCount;

            function dumpUpdateFormattingCount(styleSheets, elementCount, elementName) {
                var updateCount = styleSheets.DBG_COUNT || 0,
                    perElementCount = (elementCount === 0) ? 0 : Utils.round(updateCount / elementCount, 0.01);
                if ((elementCount > 0) || (updateCount > 0)) {
                    Utils.info('Editor.updateDocumentFormatting(): ' + elementCount + ' ' + elementName + ' updated ' + updateCount + ' times (' + perElementCount + ' per element)');
                }
            }

            function dumpParagraphCount(methodName, debugCount) {
                var updateCount = debugCount || 0,
                    perElementCount = (paragraphCount === 0) ? 0 : Utils.round(updateCount / paragraphCount, 0.01);
                Utils.info('Editor.' + methodName + '(): called ' + updateCount + ' times (' + perElementCount + ' per paragraph)');
            }

            if (!Config.DEBUG) { return $.noop; }

            tableCount = pagediv.find(DOM.TABLE_NODE_SELECTOR).length;
            cellCount = pagediv.find('td').length;
            paragraphCount = pagediv.find(DOM.PARAGRAPH_NODE_SELECTOR).length;
            spanCount = pagediv.find('span').filter(function () { return DOM.isPortionSpan(this) || DOM.isTextComponentNode(this.parentNode); }).length;
            drawingCount = pagediv.find(DrawingFrame.NODE_SELECTOR).length;

            dumpUpdateFormattingCount(self.getCharacterStyles(), spanCount, 'text spans');
            dumpUpdateFormattingCount(self.getParagraphStyles(), paragraphCount, 'paragraphs');
            dumpUpdateFormattingCount(self.getTableCellStyles(), cellCount, 'table cells');
            dumpUpdateFormattingCount(self.getTableStyles(), tableCount, 'tables');
            dumpUpdateFormattingCount(self.getDrawingStyles(), drawingCount, 'drawing objects');
            dumpParagraphCount('validateParagraphNode', self.getValidateParagraphNode().DBG_COUNT);
        }

        // public methods -----------------------------------------------------

        /**
         * Updates the formatting of all elements, after the import operations
         * of the document have been applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formatting has been
         *  updated successfully, or rejected when an error has occurred.
         */
        this.updateDocumentFormatting = function () {

            var // an object containing all slide IDs and slides
                allSlides = self.getAllSlides(),
                // the id of the active slide (TODO: This is not always the first slide in standard view)
                // -> this can also be undefined, if there is not slide in the view
                activeSlide = self.getIdOfFirstSlideOfActiveView(),
                // the format manager
                slideFormatManager = self.getSlideFormatManager(),
                // the page styles object
                pageStyles = self.getPageStyles(),
                // the object containing the page attributes
                pageAttributes = pageStyles.getElementAttributes(self.getNode());

            // a helper function that collects all functionality that needs to be done, after the
            // active slide (if it exists) is set.
            function postActiveSlideSettings() {

                // making the (empty) slide pane visible, so that the slide is positioned correctly
                app.getView().getSlidePane().setSidePaneWidthDuringLoad(self.getSlidePaneWidthInDocAttributes());

                // checking the visibility of the page (not visible for empty views)
                self.checkPageVisibility();

                // making the slides visible
                app.leaveBusyDuringImport();

                // NO autotest (means: no selenium)
                if (!Config.AUTOTEST) {
                    // Setting zoom level to 'Fit to screen width'
                    app.getView().setZoomType('slide');
                }

                // Setting contenteditable 'false' to pagecontent node in IE (46752)
                if (_.browser.IE) { DOM.getPageContentNode(pagediv).attr('contenteditable', false); }

                // writing some load logging information into the console (only in debug mode)
                dumpProfilingInformation();

                // trigger event to initialize the SlidePane, before the controller updates the GUI (looks better than vice versa)
                // -> making slidepane immediately visible, so that user sees previews of available slides
                self.trigger('slideModel:init', { isMasterView: self.isMasterView(), activeSlideId: self.getActiveSlideId(), width: self.getSlidePaneWidth(), slideRatio: self.getSlideRatio() });
            }

            // setting globals
            // drawingStyles = self.getDrawingStyles();
            // slideStyles = self.getSlideStyles();
            pagediv = self.getNode();

            // setting tasks at the slide format manager
            slideFormatManager.initializeManager(self.getIdTypeConnectionModel(), self.getStandardSlideOrder(true), self.getMasterSlideOrder(true));

            // hiding all slides
            _.invoke(allSlides, 'addClass', 'invisibleslide');

            // resetting an existing selection, after all operations are applied
            self.getSelection().resetSelection();

            // saving page attributes at model for convenience reasons
            self.setPageAttributes(pageAttributes);

            // update the root page node
            pageStyles.updateElementFormatting(pagediv);

            if (activeSlide) {

                // special handling for the first slide -> force formatting of active slide
                return slideFormatManager.forceSlideFormatting(activeSlide).done(function () {
                    // activating the first slide after formatting it and its parents completely
                    self.setActiveSlideId(activeSlide, { forceUpdate: true });
                    postActiveSlideSettings();
                });

            } else {
                postActiveSlideSettings();
                return $.when();
            }

        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class UpdateDocumentMixin

    // exports ================================================================

    return UpdateDocumentMixin;

});
