/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/slidevisibilitymanager', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/textframework/utils/dom'
], function (TriggerObject, TimerMixin, DOM) {

    'use strict';

    // class SlideVisibilityManager ===========================================

    /**
     * An instance of this class represents the slide visisibility manager. This manager
     * keeps track of slides that do not need to be inserted into the DOM for performance
     * reasons.
     * This visibility manager handles only document slides. No master or layout slides.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {PresentationApplication} app
     *  The application instance.
     */
    function SlideVisibilityManager(app) {

        var // self reference
            self = this,
            // a reference to the model
            model = null,
            // the promise of the repeated function that checks if slides can be removed from the DOM
            slideCheckerPromise = null,
            // the period of time until that a check for slide can be done (in ms).
            updateDateDelay = 10000,
           // a container with all empty slide nodes
            emptySlideNodes = {},
            // a container that keeps track of the slides that are removed from the DOM
            tempSlideContainer = {},
            // a collector with the IDs of all modified slides
            operationSlideIds = {};

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Repeated function that checks, if slides can be removed from the DOM.
         * The following slides will NOT be removed from the DOM:
         *   - the first slide (caused by selection problems).
         *   - the currently active slide.
         *   - the slide is currently formatted by the slide format manager.
         *   - the slide is already removed from the DOM.
         *   - it is register in the collector 'operationSlideIds'. Then it will be removed in the next run.
         *     Registering a slide in this collector guarantees, that it stays in the DOM for at least the
         *     delay of this repeated function (10 seconds).
         */
        function checkSlidesInDom() {

            // the slide format manager
            var formatManager = null;
            // the order of the document slides
            var slideOrder = null;

            if (!model || !model.getSlideFormatManager) { return; }

            formatManager = model.getSlideFormatManager();
            slideOrder = model.getStandardSlideOrder();

            if (!formatManager) { return; }

            if (app.getView().isBusy()) { return; } // not removing slides from the DOM in long running processes (48796)

            // remove all slides from the DOM, except:
            // - the first slide (caused by selection problems).
            // - the currently active slide.
            // - the slide is currently formatted by the slide format manager.
            // - the slide is already removed from the DOM.
            // - it is register in the collector 'operationSlideIds'. Then it will be removed in the next run.
            _.each(slideOrder, function (slideId) {
                if (!emptySlideNodes[slideId] && !model.isActiveSlideId(slideId) && !formatManager.isRunningTask(slideId) && slideId !== slideOrder[0] && !operationSlideIds[slideId]) {
                    removeSlideFromDom(slideId);
                }
            });

            // clearing the registered operation IDs, so that they can be deleted in the next run
            operationSlideIds = {};
        }

        /**
         * Registering the handler to find slides that can be removed from the DOM.
         *
         * @returns {undefined}
         */
        function startSlideChecker() {
            if (!slideCheckerPromise) {
                slideCheckerPromise = self.repeatDelayed(checkSlidesInDom, { repeatDelay: updateDateDelay, delay: updateDateDelay, infoString: 'Presentation: SlideVisibilityManager', priority: 'background-proc', app: app });
            }
        }

        /**
         * Deregistering the handler to find slides that can be removed from the DOM.
         */
        function deregisterSlideChecker() {
            if (slideCheckerPromise) { slideCheckerPromise.abort(); }
            slideCheckerPromise = null;
        }

        /**
         * Removing one slide specified by its ID from the DOM.
         *
         * @param {String} slideId
         *  The ID of the slide that shall be removed from the DOM.
         */
        function removeSlideFromDom(slideId) {

            // the slide node that will be replaced with an empty slide
            var slideNode = null;
            // the empty slide node, that will replace the specified slide in the DOM
            var emptySlide = null;

            if (!tempSlideContainer[slideId]) { // check, if the slide is already cached

                slideNode = model.getSlideById(slideId);
                emptySlide = DOM.createSlideNode();

                emptySlide.addClass('replacementslide'); // marker for empty slides
                DOM.setTargetContainerId(emptySlide, slideId); // saving ID at new empty slide

                emptySlide.insertAfter(slideNode);
                slideNode.detach();
                // slideNode.replaceWith(emptySlide);

                tempSlideContainer[slideId] = slideNode; // saving the slide node in the container
                emptySlideNodes[slideId] = emptySlide; // saving a reference to the empty slide
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Putting all slides from the cache container into the DOM.
         */
        this.appendAllSlidesToDom = function () {

            // the order of the document slides
            var slideOrder = null;

            if (!model || !model.getSlideFormatManager) { return; }

            slideOrder = model.getStandardSlideOrder();

            _.each(slideOrder, function (slideId) {
                self.appendSlideToDom(slideId);
            });
        };

        /**
         * Putting one slide from the cache container into the DOM.
         *
         * @param {String} slideId
         *  The ID of the slide that inserted into the DOM.
         */
        this.appendSlideToDom = function (slideId) {

            // the empty slide node, that is currently in the DOM
            var emptySlide = null;

            if (emptySlideNodes[slideId]) { // check, if the slide is cached

                emptySlide = emptySlideNodes[slideId];
                tempSlideContainer[slideId].insertAfter(emptySlide);
                emptySlide.remove();
                // emptySlide.replaceWith(tempSlideContainer[slideId]);

                delete tempSlideContainer[slideId];
                delete emptySlideNodes[slideId];

                emptySlide = null;
            }

            // register the slide, so that it is not removed immediately again
            // in the next run of the function 'checkSlidesInDom'
            // -> also register the slide, if it is already in the DOM, so that
            //    it is also save for at least 10 seconds.
            self.registerOperationSlideId(slideId);
        };

        /**
         * Check, whether a slide specified by its ID is attached to the DOM.
         *
         * @param {String} slideId
         *  The ID of the slide that is checked.
         *
         * @returns {Boolean}
         *  Whether the slide with the specified ID is attached to the DOM.
         */
        this.isSlideInDom = function (slideId) {
            return !emptySlideNodes[slideId];
        };

        /**
         * Registering the slide IDs for all those slides, that receive an operation.
         * These slides shall not be formatted in the next run of the function
         * 'checkSlidesInDom'.
         *
         * @param {String} slideId
         *  The ID of the slide that is registered because of an operation.
         */
        this.registerOperationSlideId = function (slideId) {
            operationSlideIds[slideId] = 1;
        };

        /**
         * Setting the initial tasks for the slide visibility manager.
         */
        this.initializeManager = function () {
            // starting the process to remove formatted slides from the DOM
            startSlideChecker();
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            deregisterSlideChecker();
            model = emptySlideNodes = tempSlideContainer = operationSlideIds = null;
        });

    } // class SlideVisibilityManager

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SlideVisibilityManager });
});
