/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/objectoperationmixin', [
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/deferredutils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/presentation/utils/operations',
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil',
    'gettext!io.ox/office/presentation/main'
], function (KeyCodes, Utils, DeferredUtils, AttributeUtils, Border, Table, DOM, Position, Snapshot, Operations, PresentationUtils, DrawingUtils, DrawingLabels, DrawingFrame, Image, gt) {

    'use strict';

    /**
     * TODO - <peter.seliger@open-xchange.com> needs to provide documentation.
     *
     * @param {}
     *
     *
     * @returns {}
     *
     */
    function recalculateDrawingAttributes(drawingAttrs, ratioDeltas) {
        var
            round   = Math.round,

            width   = drawingAttrs.width  || 0, // does dafault to ZERO for easch value if (explicit) attribute value could not be retrieved.
            height  = drawingAttrs.height || 0, //
            left    = drawingAttrs.left   || 0, //
            top     = drawingAttrs.top    || 0, //

            ratio   = (width / height),

            deltaRatioX = ratioDeltas.horizontal,
            deltaRatioY = ratioDeltas.vertical,

            attrs   = {};

        if (('noChangeAspect' in drawingAttrs) && (drawingAttrs.noChangeAspect === true))  {

            ratio = ((_.isFinite(ratio) || (ratio !== 0)) && ratio) || 1; // does fall back to 1 when running into edge cases.

            if ((deltaRatioX !== 1) && (deltaRatioX < deltaRatioY)) {

                attrs.width   = round(width * deltaRatioX);
                attrs.height  = round(attrs.width * ratio);

            } else if ((deltaRatioY !== 1) && (deltaRatioY < deltaRatioX)) {

                attrs.height  = round(height * deltaRatioY);
                attrs.width   = round(attrs.height * ratio);
            }
            if ('width' in attrs) {

                attrs.left  = round(left * deltaRatioX);
                attrs.top   = round(top  * deltaRatioY);
            }
        } else {
            attrs.width   = round(width  * deltaRatioX);
            attrs.height  = round(height * deltaRatioY);
            attrs.left    = round(left   * deltaRatioX);
            attrs.top     = round(top    * deltaRatioY);
        }
        return attrs;
    }

    /**
     * TODO - <peter.seliger@open-xchange.com> needs to provide documentation.
     *
     * @param {}
     *
     *
     * @returns {}
     *
     */
    function reformatLayoutMasterSlide(data) {
        var
            model       = data.model,
            generator   = data.generator,
            $slide      = data.$slide,
            ratioDeltas = data.ratioDeltas,
            slideId     = (_.isString(data.slideId) && data.slideId) || model.getSlideId($slide); // `slideId` is optional.

        $slide.children('.drawing').toArray().forEach(function (drawingNode) {
            var
                rootNode      = model.getRootNode(slideId),
                drawingAttrs  = AttributeUtils.getExplicitAttributes(drawingNode, { family: 'drawing' });

          //Utils.info('+++ reformatLayoutMasterSlide +++ forEach drawingNode :: drawingAttrs : ', drawingAttrs);

            if (_.isNumber(drawingAttrs.left) || _.isNumber(drawingAttrs.width) || _.isNumber(drawingAttrs.height) || _.isNumber(drawingAttrs.top)) {

                drawingAttrs  = recalculateDrawingAttributes(drawingAttrs, ratioDeltas);

                generator.generateOperation(Operations.SET_ATTRIBUTES, {

                    start:  Position.getOxoPosition(rootNode, drawingNode, 0),
                    attrs:  { drawing: drawingAttrs },
                    target: slideId
                });
            }
        });
    }

    // mix-in class ObjectOperationMixin ======================================

    /**
     * A mix-in class for the document model class PresentationModel providing
     * the style sheet containers for all attribute families used in a
     * presentation document.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @constructor
     */
    function ObjectOperationMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Calculates the size of the image defined by the given url,
         * limiting the size to the paragraph size of the current selection.
         *
         * @param {String} url
         *  The image url or base64 data url
         *
         * @returns {jQuery.Promise}
         *  A promise where the resolve result contains the size object
         *  including width and height in Hmm {width: 123, height: 456}
         */
        function getImageSize(url) {

            var // the result deferred
                def = DeferredUtils.createDeferred(app, 'OperationMixin: getImageSize'),
                // the image for size rendering
                image = $('<img>'),
                // the image url
                absUrl,
                // the clipboard holding the image
                clipboard = null;

            if (!url) { return def.reject(); }

            absUrl = Image.getFileUrl(app, url);

            // append the clipboard div to the body and place the image into it
            clipboard = app.getView().createClipboardNode();
            clipboard.append(image);

            // if you set the load handler BEFORE you set the .src property on a new image, you will reliably get the load event.
            image.one('load', function () {

                // Workaround for a strange Chrome behavior, even if we use .one() Chrome fires the 'load' event twice.
                // One time for the image node rendered and the other time for a not rendered image node.
                // We check for the rendered image node
                if (Utils.containsNode(clipboard, image)) {

                    // maybe the slide is not so big
                    // -> reducing the image size to slide size, if required
                    def.resolve({ width: Utils.convertLengthToHmm(image.width(), 'px'), height: Utils.convertLengthToHmm(image.height(), 'px') });
                }
            })
            .error(function () {
                Utils.warn('Editor.getImageSize(): image load error');
                def.reject();
            })
            .attr('src', absUrl);

            // always remove the clipboard again
            def.always(function () {
                clipboard.remove();
            });

            return def.promise();
        }

        /**
         * Inserts a drawing component into the document DOM.
         *
         * @param {String} type
         *  The type of the drawing. Supported values are 'shape', 'group', 'table',
         *  'image', 'diagram', 'chart', 'ole', 'horizontal_line', 'undefined'
         *
         * @param {Number[]} start
         *  The logical start position for the new drawing.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new drawing component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {String} [target]
         *  ID of root container node.
         *
         * @returns {Boolean}
         *  Whether the drawing has been inserted successfully.
         */
        function implInsertDrawing(type, start, attrs, target) {

            var // text span that will precede the field
                span = null,
                // deep copy of attributes, because they are modified in webkit browsers
                attributes = _.copy(attrs, true),
                // new drawing node
                drawingNode = null,
                // root node container
                rootNode = self.getRootNode(target),
                // image aspect ratio
                currentElement = Position.getContentNodeElement(rootNode, start.slice(0, -1), { allowDrawingGroup: true }),
                // whether the drawing is a group drawing node
                isGroupNode = currentElement && DrawingFrame.isGroupDrawingFrame(currentElement),
                // whether the drawing is inserted into a drawing group
                insertIntoDrawingGroup = false,
                // the function used to insert the new drawing frame
                insertFunction = 'insertAfter',
                // a table node for drawings of type 'table'
                table = null,
                // the number of children in the drawing group
                childrenCount = 0;

            // helper function to insert a drawing frame into an existing drawing group
            function addDrawingFrameIntoDrawingGroup() {

                childrenCount = DrawingFrame.getGroupDrawingCount(currentElement);

                if (_.isNumber(childrenCount)) {
                    if (childrenCount === 0) {
                        if (_.last(start) === 0) {
                            span = $(currentElement).children().first(); // using 'span' for the content element in the drawing group
                            insertFunction = 'appendTo';
                        }
                    } else {
                        if (_.last(start) === 0) {
                            // inserting before the first element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, 0);
                            insertFunction = 'insertBefore';
                        } else if (_.last(start) <= childrenCount) {
                            // inserting after the span element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, _.last(start) - 1);
                        }
                    }
                    insertIntoDrawingGroup = true;
                }
            }

            if (isGroupNode) {
                addDrawingFrameIntoDrawingGroup(); // adding drawing into the group, without using a text span
            } else {
                try {
                    span = self.prepareTextSpanForInsertion(start, {}, target);
                } catch (ex) {
                    // do nothing, try to repair missing text spans
                }
            }

            // insert the drawing with default settings between the two text nodes (store original URL for later use)
            drawingNode = DrawingFrame.createDrawingFrame(type)[insertFunction](span);

            // apply the passed drawing attributes
            if (_.isObject(attributes)) { self.getDrawingStyles().setElementAttributes(drawingNode, attributes); }

            // saving list style attributes, that are assigned directly to non-place holder drawings (45279)
            if (_.isObject(attributes)) { self.getDrawingStyles().setDrawingListStyleAttributes(drawingNode, attributes); }

            // applying all attributes (TODO: Also handling attributes from other layers)
            if (self.isImportFinished() && !insertIntoDrawingGroup) { self.getDrawingStyles().updateElementFormatting(drawingNode); } // TODO (is this necessary?)

            // inserting the table node into a drawing of type 'table'
            if (type === 'table') {

                table = $('<table>').attr('role', 'grid').addClass(DrawingFrame.TABLE_NODE_IN_TABLE_DRAWING_CLASS).append($('<colgroup>'));
                self.insertContentNode(Position.appendNewIndex(start), table, (target ? target : undefined));

                // apply the passed table attributes
                if (attrs && attrs.table) { self.getTableStyles().setElementAttributes(table, attrs); }
            }

            // saving the drawing attributes in drawing styles (ignoring the return value, no event is fired)
            if (target) { self.getDrawingStyles().savePlaceHolderAttributes(target, attrs, drawingNode); }

            self.setLastOperationEnd(Position.increaseLastIndex(start));

            return true;
        }

        // simplified version of implInsertDrawing, that allows only drawings inside slides

//        function implInsertDrawing(type, start, attrs, target) {
//
//            var // deep copy of attributes, because they are modified in webkit browsers
//                attributes = _.copy(attrs, true),
//                // new drawing node
//                drawingNode = null,
//                // determining the parent of the new drawing (this can only be a group or a slide, not a paragraph)
//                drawingParentPoint = Position.getDOMPosition(self.getRootNode(target), _.initial(start), true),
//                // the parent node
//                drawingParent = drawingParentPoint ? drawingParentPoint.node : null,
//                // the children of the drawing container
//                children = drawingParent ? $(drawingParent).children(DrawingFrame.NODE_SELECTOR) : null,
//                // the number of children in the drawing group
//                childrenCount = children ? children.length : 0,
//                // the function used to insert the new drawing frame
//                insertFunction = 'insertAfter',
//                // the element, relative to that the new drawing is inserted
//                insertElement = drawingParent;
//
//            // checking, if the drawing parent is a slide or a group container
//            if (!(DOM.isSlideNode(drawingParent) || DrawingFrame.isGroupDrawingFrame(drawingParent))) { return false; }
//
//            // checking, if the insert position is valid -> it is not possible to insert at position 2, if there is only
//            // one drawing inside slide or group container
//            if (_.last(start) > childrenCount) { return false; }
//
//            // check, if 'appendTo' must be used instead of 'insertAfter'
//            if (_.last(start) === 0) {
//                // inserting the new drawing as first child of the parent
//                insertFunction = 'prependTo';
//            } else {
//                // finding that drawing, after which the new drawing will be inserted
//                insertElement = children[_.last(start) - 1];
//            }
//
//            // insert the drawing with default settings between the two text nodes (store original URL for later use)
//            drawingNode = DrawingFrame.createDrawingFrame(type)[insertFunction](insertElement);
//
//            // apply the passed drawing attributes
//            if (_.isObject(attributes)) { drawingStyles.setElementAttributes(drawingNode, attributes); }
//
//            // saving the attributes in drawing styles (ignoring the return value, no event is fired)
//            if (target) { self.getDrawingStyles().savePlaceHolderAttributes(target, attrs); }
//
//            return true;
//        }

        /**
         * Calculating a default size for an inserted shape or text frame. This is necessary, if the element
         * is inserted via keyboard and not with the mouse.
         *
         * @returns {Object}
         *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
         *  The values are given in 1/100 mm relative to the slide.
         */
        function getDefaultRactangleForInsert() {

            var scrollNode = app.getView().getContentRootNode();
            var pos = { left: scrollNode.scrollLeft(), top: scrollNode.scrollTop(), width: scrollNode.width(), height: scrollNode.height() };

            pos.left += pos.width * 0.1;
            pos.top += pos.height * 0.1;
            pos.width *= 0.8;
            pos.height *= 0.8;

            return PresentationUtils.convertAppContentBoxToOperationBox(app, pos);
        }

        /**
         * Inserts a table component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new table.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new table component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *
         * @returns {Boolean}
         *  Whether the table has been inserted successfully.
         */
//        function implInsertTable(start, attrs, target) {
//
//            var // the new table node
//                table = $('<table>').attr('role', 'grid').append($('<colgroup>')),
//                // whether the table was inserted into the DOM
//                inserted = self.insertContentNode(_.clone(start), table, (target ? target : undefined));
//
//            // insertContentNode() writes warning to console
//            if (!inserted) { return false; }
//
//            // apply the passed table attributes
//            if (attrs && attrs.table) { self.getTableStyles().setElementAttributes(table, attrs); }
//
//            return true;
//        }

        /**
         * Calculating new value for position, size and rotate of a drawing node. The new values depend
         * on the old values, the event properties and some addtional properties.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {jQuery|Node} drawingNode
         *  The drawing node that will get new values assigned.
         *
         * @param {Object[]} allProperties
         *  A collector with objects for the properties that need to be modified. It is necessary
         *  that each property is specified by its 'name'. Additional optional values can be
         *  specified to influence the calculation. These properties are 'type', 'delta' or
         *  'ctrlDoubles'.
         *
         * @returns {Object|Null}
         *  The new drawing object for a setAttributes operation. If no valid new value can be
         *  calculated, null is returned.
         */
        function getDrawingPropertiesObject(event, drawingNode, allProperties) {

            var // the explicit attributes at the specified element
                elemAttrs = AttributeUtils.getExplicitAttributes(drawingNode),
                // the merged attributes of the specified element (required for positions of place holder drawings)
                mergedAttrs = null,
                // the previously calculated value
                previousDelta = null,
                // whether the return value must be 0
                invalidAttributes = false,
                // the old value of the specified property
                attrs = {};

            _.each(allProperties, function (property) {

                var // the old and the new value for the property
                    oldValue = null, newValue = null,
                    // whether the new value must be increased relative to the current position
                    increase = Utils.getBooleanOption(property, 'increase', false);

                if (elemAttrs && elemAttrs.drawing && _.isNumber(elemAttrs.drawing[property.name])) {
                    oldValue = elemAttrs.drawing[property.name];
                } else {
                    mergedAttrs = mergedAttrs || self.getDrawingStyles().getElementAttributes(drawingNode);
                    oldValue = mergedAttrs.drawing[property.name];
                }

                if (property.name === 'rotation') {
                    newValue = oldValue + property.delta;
                    // normalize negative and angles greater than 360
                    if (newValue < 0) {
                        newValue += 360;
                    }
                    if (newValue > 360) {
                        newValue -= 360;
                    }
                } else {
                    if (property.delta === 'halfOfPrevious') { property.delta = Utils.round(0.5 * previousDelta, 1); }

                    if (property.type === 'percentage') { property.delta = Utils.round((property.delta * oldValue) / 100, 1); }

                    if (property.ctrlDoubles && event.ctrlKey) { property.delta *= 2; }

                    newValue = increase ? oldValue + property.delta : oldValue - property.delta;

                    // avoiding operations without change, because the percentage is not changing anything (very small drawings)
                    if (property.type === 'percentage' && !increase && (oldValue - newValue < 2)) { invalidAttributes = true; }
                }

                attrs[property.name] = newValue;

                previousDelta = property.delta; // saving for the following property
            });

            if (invalidAttributes) { attrs = null; }

            return attrs;
        }

        /**
         * Generating one operation for a drawing specified by its drawing selection object. This can be defined
         * by a single drawing or a multiple drawing selection.
         *
         * @param {Object} generator
         *  The operation generator object that generates and collects all operations.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} drawingSelection
         *  A drawing selection object as it is created by a multiple drawing selection. This object contains
         *  information about the selected drawing and its logical position.
         *
         * @param {Boolean} leftRightCursor
         *  Whether this event was triggered by pressing left or right cursor key.
         */
        function generateDrawingAttributeOperation(generator, event, drawingSelection, leftRightCursor) {

            var // the key for setting the top left corner position
                positionProperty = leftRightCursor ? 'left' : 'top',
                // the property name for changing width or height
                sizeProperty = leftRightCursor ? 'width' : 'height',
                // whether a new operation needs to be created
                createOperation = true,
                // the properties for the new setAttributes operation
                operationProperties = { start: drawingSelection.startPosition, attrs: { drawing: {} } },
                // the delta value for the moving of a drawing (in 1/100 mm)
                positionDelta = 100,
                // the delta value for the width and height of a drawing (percentage)
                sizeDeltaPercentage = 5,
                // drawing node from multiselection
                drawingNode = self.getSelection().getDrawingNodeFromMultiSelection(drawingSelection);

            if (event.shiftKey) {
                // generating a resize operation for all selected drawings
                operationProperties.attrs.drawing = getDrawingPropertiesObject(event, drawingNode, [{ name: sizeProperty, delta: sizeDeltaPercentage, type: 'percentage', increase: (event.keyCode === KeyCodes.RIGHT_ARROW || event.keyCode === KeyCodes.UP_ARROW) }, { name: positionProperty, delta: 'halfOfPrevious', increase: (event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.DOWN_ARROW) }]);
                if (!operationProperties.attrs.drawing) { createOperation = false; } // avoiding OPs without change or negative width/height
            } else if (event.altKey && leftRightCursor) {
                if (DrawingFrame.isTableDrawingFrame(drawingNode)) {
                    createOperation = false; // no rotation of tables
                } else {
                    operationProperties.attrs.drawing = getDrawingPropertiesObject(event, drawingNode, [{ name: 'rotation', delta: (event.keyCode === KeyCodes.RIGHT_ARROW ? 15 : -15) }]);
                }
            } else {
                // generating a move operation for all selected drawings (different velocity, if Ctrl is pressed)
                operationProperties.attrs.drawing = getDrawingPropertiesObject(event, drawingNode, [{ name: positionProperty, delta: positionDelta, ctrlDoubles: true, increase: (event.keyCode === KeyCodes.RIGHT_ARROW || event.keyCode === KeyCodes.DOWN_ARROW) }]);
            }

            if (createOperation) {
                PresentationUtils.extendPlaceholderProp(app, drawingNode, operationProperties);
                self.extendPropertiesWithTarget(operationProperties, self.getActiveTarget());
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
            }

        }

        /**
         * Handler function for moving one drawing from the specified start position to the
         * specified end position.
         * Info: In Presentation app only drawings can be moved.
         *
         * @param {Number[]} start
         *  The logical start position of the drawing.
         *
         * @param {Number[]} to
         *  The logical end position of the drawing.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which the logical positions are related.
         *
         * @returns {Boolean}
         *  Whether the move of the drawing has been successful.
         */
        function implMove(start, to, target) {

            var // the active root node
                activeRootNode = self.getRootNode(target),
                // the source point (node and offset) at the logical start position
                sourcePoint = Position.getDOMPosition(activeRootNode, start, true),
                // the source point (node and offset) at the logical end position
                destPoint = Position.getDOMPosition(activeRootNode, to, true),
                // the drawing node at the logical start position
                sourceNode = null,
                // the drawing node at the logical end position
                destNode = null,
                // whether the drawing is moved backwards (to the foreground) or forwards (to the background)
                backwards = _.last(to) > _.last(start),
                // whether the moved drawing is inserted before the drawing node that is currently at the
                // specified destination position
                insertBefore = !backwards;

            if (sourcePoint && destPoint && sourcePoint.node && destPoint.node) {

                sourceNode = $(sourcePoint.node);
                destNode = $(destPoint.node);

                // moving the drawing (and handle text spans)
                if (insertBefore) {
                    sourceNode.next().remove();
                    destNode = DOM.splitTextSpan(destNode.prev(), 0, { append: true });
                    sourceNode.insertBefore(destNode);
                } else {
                    sourceNode.prev().remove();
                    destNode = DOM.splitTextSpan(destNode.next(), 0);
                    $(sourceNode).insertAfter(destNode);
                }

            }

            return true;
        }

        /**
         * Return the drawing attributes of the given drawingNode.
         * @param {HTMLElement|jQuery} drawingNode
         * @returns {Object} the drawing attributes.
         */
        function getDrawingAttrsByNode(drawingNode) {
            var expAttrs = AttributeUtils.getExplicitAttributes(drawingNode),
                drawing = expAttrs.drawing;

            if (PresentationUtils.isPlaceHolderAttributeSet(expAttrs)) {
                drawing = self.getDrawingStyles().getElementAttributes(drawingNode).drawing;
            }
            if (DrawingFrame.isTableDrawingFrame(drawingNode)) {
                drawing.height = Utils.convertLengthToHmm(drawingNode.height(), 'px');
            }
            return drawing;
        }

        /**
         * Class to change the alignment of the selected drawings.
         */
        function ChangeAlignment() {

            var self2 = this,
                baseObject = {},
                moveAttrName = null,
                rightBottomPosition = null;

            // private methods ----------------------------------------------------

            /**
             * Get the drawing from a drawing selection.
             */
            function getRotatedDrawingByDrawingSelection(drawingSelection) {
                var drawingNode = self.getSelection().getDrawingNodeFromMultiSelection(drawingSelection),
                    drawing = getDrawingAttrsByNode(drawingNode),
                    angle = DrawingFrame.getDrawingRotationAngle(app.getModel(), drawingNode);
                return DrawingUtils.getRotatedDrawingPoints(drawing, angle);
            }
            /**
             * Set the drawing which is on the lowest left or top position, to move the selected drawings.
             */
            function setLeftTopDrawing(drawingSelection) {
                var drawing = getRotatedDrawingByDrawingSelection(drawingSelection);
                if (baseObject[moveAttrName] === null || drawing[moveAttrName] < baseObject[moveAttrName]) {
                    baseObject.object = drawingSelection;
                    baseObject[moveAttrName] = drawing[moveAttrName];
                }
            }

            /**
             * Return the operation properties to move the drawingNode.
             * @param {HTMLElement|jQuery} drawingNode
             * @param {Number} position the new position of the drawing.
             * @param {Number[]} startPosition
             *  The logical start position of this drawing.
             * @returns {Object|null} the new operations to move the drawing or null if the drawing position is not changed,
             *  this happens if the drawing is the base object.
             */
            function getOperationsProperties(drawingNode, position, startPosition) {
                if (position !== null && _.isNumber(position) && !_.isNaN(position)) {
                    var operationProperties = { start: startPosition, attrs: { drawing: { } } };
                    operationProperties.attrs.drawing[moveAttrName] = position;
                    PresentationUtils.extendPlaceholderProp(app, drawingNode, operationProperties);
                    return operationProperties;
                }
                return null;
            }

            /**
             * Returns a helper object wich contains some objects which are used in different
             * function. This helper function is created to avoid code doubling.
             * @param {Object} drawingSelection the drawing to calculate the new Postition.
             *
             * @returns {Object}
             *  @param {HTMLElement|jQuery} drawingNode
             *  @param {Object} drawing the drawing attributes
             *  @param {Number} angle the rotation angle of the drawing
             *  @param {Object} rotPos
             *      the result of the getRotatedDrawingPoints(drawing, angle) function
             */
            function getDrawingHelperObject(drawingSelection) {
                var drawingNode = self.getSelection().getDrawingNodeFromMultiSelection(drawingSelection),
                    drawing = getDrawingAttrsByNode(drawingNode),
                    angle = DrawingFrame.getDrawingRotationAngle(app.getModel(), drawingNode),
                    rotPos = DrawingUtils.getRotatedDrawingPoints(drawing, angle);

                return {
                    drawingNode: drawingNode,
                    drawing: drawing,
                    angle: angle,
                    rotPos: rotPos
                };
            }

            /**
             * Returns the new left or top position for the given drawing.
             */
            function getLeftTopPosition(drawingSelection, startPosition) {
                if (drawingSelection !== baseObject.object) {
                    var obj = getDrawingHelperObject(drawingSelection),
                        moveLeft = baseObject.left - obj.rotPos.left,
                        moveTop = baseObject.top - obj.rotPos.top,
                        centerLeft = obj.drawing.left + obj.drawing.width / 2,
                        centerTop = obj.drawing.top  + obj.drawing.height / 2,
                        position = Math.round(DrawingUtils.rotatePointWithAngle(centerLeft + moveLeft, centerTop + moveTop, obj.rotPos.leftTopPoint.left + moveLeft, obj.rotPos.leftTopPoint.top + moveTop, -obj.angle)[moveAttrName]);

                    return getOperationsProperties(obj.drawingNode, position, startPosition);
                }
                return null;
            }

            /**
             * Set the drawings which are on the lowest left and  most left position to calculate the center of
             * this two drawings.
             */
            function setCenterPosition(drawingSelection) {
                setLeftTopDrawing(drawingSelection);
                setRightBottomDrawing(drawingSelection);
            }

            /**
             * Set the drawings which are on the lowest top and  most top position to calculate the middle of
             * this two drawings.
             */
            function setMiddlePosition(drawingSelection) {
                setLeftTopDrawing(drawingSelection);
                setRightBottomDrawing(drawingSelection);
            }

            /**
             * Returns the new center or middle position of the given drawing. If it returns center or middle
             * will be set in the setAlignmentDirection function.
             */
            function getCenterMiddlePosition(drawingSelection, startPosition) {
                if (baseObject.centerMiddle === null) {
                    baseObject.centerMiddle = Math.round((baseObject[rightBottomPosition] - baseObject[moveAttrName]) / 2 + baseObject[moveAttrName]);
                }
                var obj = getDrawingHelperObject(drawingSelection),
                    moveLeft = baseObject.centerMiddle - obj.rotPos.width / 2 - obj.rotPos.left,
                    moveTop = baseObject.centerMiddle - obj.rotPos.height / 2 - obj.rotPos.top,
                    centerLeft = obj.drawing.left + obj.drawing.width / 2,
                    centerTop = obj.drawing.top  + obj.drawing.height / 2,
                    position = Math.round(DrawingUtils.rotatePointWithAngle(centerLeft + moveLeft, centerTop + moveTop, obj.rotPos.leftTopPoint.left + moveLeft, obj.rotPos.leftTopPoint.top + moveTop, -obj.angle)[moveAttrName]);

                return getOperationsProperties(obj.drawingNode, position, startPosition);
            }

            /**
             * Set the drawing which is on the most left or top position, to move the selected drawings.
             */
            function setRightBottomDrawing(drawingSelection) {
                var drawing = getRotatedDrawingByDrawingSelection(drawingSelection),
                    pos = drawing[rightBottomPosition];

                if (baseObject[rightBottomPosition] === null || pos > baseObject[rightBottomPosition]) {
                    baseObject.object = drawingSelection;
                    baseObject[rightBottomPosition] = pos;
                }
            }

            /**
             * Returns the new left or top position for the given drawing.
             */
            function getRightBottomPosition(drawingSelection, startPosition) {
                if (drawingSelection !== baseObject.object) {
                    var obj = getDrawingHelperObject(drawingSelection),
                        moveLeft = baseObject.right - obj.rotPos.width - obj.rotPos.left,
                        moveTop = baseObject.bottom - obj.rotPos.height - obj.rotPos.top,
                        centerLeft = obj.drawing.left + obj.drawing.width / 2,
                        centerTop = obj.drawing.top  + obj.drawing.height / 2,
                        position = Math.round(DrawingUtils.rotatePointWithAngle(centerLeft + moveLeft, centerTop + moveTop, obj.rotPos.leftTopPoint.left + moveLeft, obj.rotPos.leftTopPoint.top + moveTop, -obj.angle)[moveAttrName]);
                    return getOperationsProperties(obj.drawingNode, position, startPosition);
                }
                return null;
            }

            /**
             * Set the direction of the alignment for the selected drawings.
             *
             * @param {String} direction of the alignment.
             */
            this.setAlignmentDirection = function (direction) {
                baseObject = {
                    left: null,
                    right: null,
                    top: null,
                    bottom: null,
                    centerMiddle: null,
                    object: null
                };
                moveAttrName = null;
                self2.setBaseDrawing = $.noop();
                self2.getOperationPropertiesForMultiSelection = $.noop();
                rightBottomPosition = null;

                switch (direction) {
                    case 'left':
                        moveAttrName = 'left';
                        self2.setBaseDrawing = setLeftTopDrawing;
                        self2.getOperationPropertiesForMultiSelection = getLeftTopPosition;
                        break;
                    case 'center':
                        moveAttrName = 'left';
                        rightBottomPosition = 'right';
                        self2.setBaseDrawing = setCenterPosition;
                        self2.getOperationPropertiesForMultiSelection = getCenterMiddlePosition;
                        break;
                    case 'right':
                        moveAttrName = 'left';
                        rightBottomPosition = 'right';
                        self2.setBaseDrawing = setRightBottomDrawing;
                        self2.getOperationPropertiesForMultiSelection = getRightBottomPosition;
                        break;
                    case 'top':
                        moveAttrName = 'top';
                        self2.setBaseDrawing = setLeftTopDrawing;
                        self2.getOperationPropertiesForMultiSelection = getLeftTopPosition;
                        break;
                    case 'middle':
                        moveAttrName = 'top';
                        rightBottomPosition = 'bottom';
                        self2.setBaseDrawing = setMiddlePosition;
                        self2.getOperationPropertiesForMultiSelection = getCenterMiddlePosition;
                        break;
                    case 'bottom':
                        moveAttrName = 'top';
                        rightBottomPosition = 'bottom';
                        self2.setBaseDrawing = setRightBottomDrawing;
                        self2.getOperationPropertiesForMultiSelection = getRightBottomPosition;
                        break;
                    default:
                        return false;
                }
            };

            // public methods -----------------------------------------------------

            /**
             * Get the name for the move operation attribute.
             * It's left or top.
             *
             * @returns {String} left or top
             */
            this.getMoveAttrName = function () {
                return moveAttrName;
            };

            /**
             * Set the drawing to check if the base object e.g. if it is the drawing with the most left position.
             * The function will be set on the init function.
             *
             * @param {Object} drawingSelect the drawing to get the base object.
             * @param {Object} drawing the drawing attributes of the given drawing object.
             */
            this.setBaseDrawing = $.noop();

            /**
             * Get the operation properties for the new position of the drawing, which
             * is a part of a multiselection.
             * The function will be set on the init function.
             *
             * @param {Object} drawingSelection the drawing to calculate the new Postition.
             *
             * @param {Number[]} startPosition
             *  The logical start position of this drawing.
             *
             * @returns {Object|null} the new operations to move the drawing or null if the drawing position is not changed,
             * this happens if the drawing is the base object.
             */
            this.getOperationPropertiesForMultiSelection = $.noop();

            /**
             * Get the operation properties for the new position of a single selected drawing.
             * The drawing will be alligned on the page.
             * The function will be set on the init function.
             *
             * @param {Object} drawingSelection the drawing to calculate the new Postition.
             *
             * @param {Number[]} startPosition
             *  The logical start position of this drawing.
             *
             * @returns {Object|null} the new operations to move the drawing or null if the drawing position is not changed,
             * this happens if the drawing is the base object.
             */
            this.getOperationPropertiesForSingleSelection = function (drawingSelection, direction) {
                var selection = self.getSelection(),
                    drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection),
                    pageStyles = self.getStyleCollection('page'),
                    page = pageStyles.getElementAttributes(self.getRootNode(self.getActiveSlideId())).page,
                    startPosition;

                if (selection.isAdditionalTextframeSelection()) {
                    startPosition = Position.getOxoPosition(self.getCurrentRootNode(), drawingNode, 0);
                } else {
                    startPosition = drawingSelection.startPosition;
                }

                self2.setAlignmentDirection(direction);

                baseObject = {
                    left: 0,
                    right: page.width,
                    top: 0,
                    bottom: page.height,
                    centerMiddle: null,
                    object: { left: 0, top: 0, width: page.width, height: page.height }
                };

                return self2.getOperationPropertiesForMultiSelection(drawingSelection, startPosition);
            };
        }

        /** Single alignment object to calculate the new position of the selected drawings. */
        var changeAlignment = new ChangeAlignment();

        // public methods -----------------------------------------------------

        /**
         * TODO - <peter.seliger@open-xchange.com> needs to provide documentation.
         *
         * @param {}
         *
         *
         * @returns {}
         *
         */
        this.reformatDocument = function reformatDocument(newPageAttrs) {
          //Utils.info('+++ reformatDocument +++ [newPageAttrs] : ', newPageAttrs);
            var
                operationsDeferred,
                operationList,

                snapshot,

                generator     = self.createOperationsGenerator(),
                recentAttrs   = self.getPageAttributes(),

                recentWidth   = recentAttrs.page.width,
                recentHeight  = recentAttrs.page.height,

              //orientation   = newPageAttrs.orientation,

                pageWidth     = _.isNumber(newPageAttrs.width) ? newPageAttrs.width : recentWidth, // 48537
                pageHeight    = _.isNumber(newPageAttrs.height) ? newPageAttrs.height : recentHeight,

                ratioDeltas   = {
                    vertical:   (pageHeight / recentHeight),
                    horizontal: (pageWidth / recentWidth)
                },
              //isPageOperationBeforeSlideOperations = (pageHeight < recentHeight),

                activeSlideId         = self.getActiveSlideId(),
                slideIdSet            = self.getSlideIdSet(activeSlideId),

              //activeStandardId      = slideIdSet.slideId,
                activeLayoutId        = slideIdSet.layoutId,
                activeMasterId        = slideIdSet.masterId,

              //$activeStandardSlide  = self.getSlideById(activeStandardId),
                $activeLayoutSlide    = self.getSlideById(activeLayoutId),
                $activeMasterSlide    = self.getSlideById(activeMasterId);

          //Utils.info('+++ reformatDocument +++ [recentWidth, recentHeight, recentAttrs] : ', recentWidth, recentHeight, recentAttrs);
          //Utils.info('+++ reformatDocument +++ [pageWidth, pageHeight, orientation] : ', pageWidth, pageHeight, orientation);
          //Utils.info('+++ reformatDocument +++ [isPageOperationBeforeSlideOperations] : ', isPageOperationBeforeSlideOperations);

            generator.generateOperation(Operations.SET_DOCUMENT_ATTRIBUTES, { attrs: { page: newPageAttrs } });

            if ($activeMasterSlide) {
                reformatLayoutMasterSlide({ model: self, generator: generator, $slide: $activeMasterSlide, ratioDeltas: ratioDeltas, slideId: activeMasterId/*optional*/ });
            }

            if ($activeLayoutSlide) {
                reformatLayoutMasterSlide({ model: self, generator: generator, $slide: $activeLayoutSlide, ratioDeltas: ratioDeltas, slideId: activeLayoutId/*optional*/ });
            }/*
            if ($activeStandardSlide) {
                reformatLayoutMasterSlide({ model: self, generator: generator, $slide: $activeStandardSlide, ratioDeltas: ratioDeltas, slideId: activeStandardId/ *optional* / });
            }*/

            // `true` flag for a deep clone of this slide-id-list.
            self.getMasterSlideOrder(true).forEach(function (slideId/*, idx, list*/) {

                var $slide = null;

                if (slideId !== activeLayoutId && slideId !== activeMasterId) { // not iterating twice over active layout or master slide
                    $slide = self.getSlideById(slideId);
                    //Utils.info('+++ reformatDocument +++ forEach slideId :: $slide : ', $slide);
                    reformatLayoutMasterSlide({ model: self, generator: generator, $slide: $slide, ratioDeltas: ratioDeltas, slideId: slideId/*optional*/ });
                }
            });

            self.setBlockKeyboardEvent(true);

            snapshot            = new Snapshot(app);

            operationList       = generator.getOperations(); // operationList.reverse();
            operationsDeferred  = self.applyOperations(operationList, { async: true });

            app.getView().enterBusy({

                cancelHandler: function () {
                    if (operationsDeferred && operationsDeferred.abort) {

                        snapshot.apply();  // restoring the old state
                        app.enterBlockOperationsMode(function () { operationsDeferred.abort(); }); // block sending of operations
                    }
                },
                immediate: true,

                warningLabel: /*#. shown while all recorded changes will be applied to the document */ gt('Applying changes, please wait...')
            });

            operationsDeferred.progress(function (progress) { // handle the result of change track operations.

                // update the progress bar according to progress of the operations promise.
                app.getView().updateBusyProgress(progress);

            }).always(function () {

                app.getView().leaveBusy();
                self.setBlockKeyboardEvent(false);

              //app.getModel().trigger('reformat:document:after', { slideRatio: (pageWidth / pageHeight) });
            });
        };

        /**
         * Inserting an image into a slide.
         *
         * @param {Object} imageHolder
         *  The object containing the image attributes. This are 'url' or 'substring'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.rectangle=null]
         *   A user defined rectangle containing the properties 'top', 'left', 'width'
         *   and 'height' in 1/100 mm.
         *  @param {Number[]} [options.position=null]
         *   The logical position at that the text frame drawing shall be inserted.
         *  @param {Number[]} [options.removeExisting=false]
         *   Whether an existing drawing at the specified position shall be removed. This can
         *   only be used, if a position was specified as option.
         *  @param {Object} [options.attributes=null]
         *   The explicit attributes of the drawing to be replaced. This will only be
         *   evaluated, if the option 'removeExisting' is set to true.
         */
        this.insertImageURL = function (imageHolder, options) {

            var // the attributes for the image
                attrs = { drawing: self.getInsertDrawingAttibutes(), image: {}, line: { type: 'none' } };

            if (imageHolder.url) {
                attrs.image.imageUrl = imageHolder.url;
                attrs.drawing.name = imageHolder.name;
            } else if (imageHolder.substring(0, 10) === 'data:image') {
                attrs.image.imageData = imageHolder;
            } else {
                attrs.image.imageUrl = imageHolder;
            }

            return getImageSize(attrs.image.imageUrl || attrs.image.imageData).then(function (size) {

                var // the deferred for inserting the image
                    def = DeferredUtils.createDeferred(app, 'OperationMixin: insertImageURL'),
                    // whether the image was imported successfully
                    result = false,
                    // created operation
                    newOperation = null,
                    // target for operation - if exists
                    target = self.getActiveTarget(),
                    // the operations generator
                    generator = self.createOperationsGenerator(),
                    // whether an existing drawing at the specified position shall be removed
                    removeExisting = Utils.getBooleanOption(options, 'removeExisting', false),
                    // the explicit attributes of a place holder drawing that will be replaced
                    expAttrs = Utils.getObjectOption(options, 'attributes', null),
                    // whether an existing drawing at the specified position shall be removed
                    position = Utils.getArrayOption(options, 'position', null),
                    // the position width and height of the text frame
                    rectangle = Utils.getObjectOption(options, 'rectangle', null),
                    // whether a place holder of type 'pic' shall be removed
                    replacePicPlaceHoder = false,
                    // the logical position of the drawing
                    start = position || self.getNextAvailablePositionInActiveSlide(),
                    // a cropping object for inserting images into drawing place holders of type 'pic'
                    cropAttrs = null,
                    // a modified image size, if it necessary to reduce the size (content place holders)
                    newSize = null,
                    // a drawing object that describes a place holder drawing that will be replaced
                    drawingObj = null;

                // exit silently if we lost the edit rights
                if (!self.getEditMode()) {
                    return def.resolve();
                }

                self.getUndoManager().enterUndoGroup(function () {

                    // finding a valid position for the table if not specified (but not in master view)
                    if (!self.isMasterView() && !position) {
                        drawingObj = self.getValidPlaceHolderPositionForInsertDrawing(PresentationUtils.IMAGE_PLACEHOLDER_TYPE);
                        if (drawingObj) {
                            start = drawingObj.position;
                            rectangle = drawingObj.rectangle;
                            expAttrs = drawingObj.attributes;
                            removeExisting = true; // removing an existing place holder drawing
                        }
                    }

                    // removing an existing drawing, if specified
                    if (removeExisting) {
                        newOperation = { start: start };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);

                        if (expAttrs) {

                            replacePicPlaceHoder = (PresentationUtils.getPlaceHolderDrawingType(null, expAttrs) === 'pic');

                            // expanding the drawing attributes with the explicit attributes of the existing place holder drawing
                            _.extend(attrs, expAttrs); // simply overwriting existing drawing attributes 'name', 'left' and 'top'
                            // ... but not sending phSize to the server
                            if (attrs.presentation && attrs.presentation.phSize) { delete attrs.presentation.phSize; }

                            // inserting an image into the image place holder requires a cropping of the drawing
                            if (replacePicPlaceHoder) {

                                cropAttrs = self.calculateCroppingValue(rectangle, size);

                                if (cropAttrs && cropAttrs.value) {
                                    if (cropAttrs.isHorizontal) {
                                        attrs.image.cropLeft = cropAttrs.value;
                                        attrs.image.cropRight = cropAttrs.value;
                                    } else {
                                        attrs.image.cropTop = cropAttrs.value;
                                        attrs.image.cropBottom = cropAttrs.value;
                                    }
                                }
                            } else {
                                // in content body place holder the inserted image must not be larger than the drawing
                                newSize = self.checkImageSizeInDrawing(rectangle, size);
                                if (newSize) { size = newSize; }
                            }
                        }

                    }

                    // expanding the drawing attributes with the default position and size of the image
                    // -> but not for drawing place holders of type 'pic'
                    if (!replacePicPlaceHoder) {

                        _.extend(attrs.drawing, size); // adding width and height to the drawing attributes

                        if (rectangle) {
                            attrs.drawing.top = rectangle.top + (0.5 * (rectangle.height - attrs.drawing.height));
                            attrs.drawing.left = rectangle.left + (0.5 * (rectangle.width - attrs.drawing.width));
                        }
                    }

                    // no change of aspect ratio
                    attrs.drawing.noChangeAspect = true;

                    // if a place holder drawing is replaced, the new drawing needs to get the noGroup attributes
                    // if (removeExisting) { attrs.drawing.noGroup = true; }

                    // creating the operation
                    newOperation = { start: start, type: 'image', attrs: attrs };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DRAWING_INSERT, newOperation);

                    result = self.applyOperations(generator);

                }); // enterUndoGroup()

                // selecting the new inserted drawing
                self.getSelection().setTextSelection(start, Position.increaseLastIndex(_.clone(start)));

                return result ? def.resolve() : def.reject();
            })
            .fail(function () {
                app.rejectEditAttempt('image');
            });
        };

        /**
         * Calculation the cropping that is required, so that an image fits into a specified drawing
         * without changing the ratio of width to height. If the drawing does not have the same ratio
         * of width-to-height as the image, the image need to be cropped on the left/right side or
         * the top/bottom side. Both sides shall be cropped by the same value in percent.
         *
         * @param {Object} drawingSize
         *  An object containing at least the properties 'width' and 'height' of the drawing, in that
         *  the image shall be inserted.
         *
         * @param {Object} imageSize
         *  An object containing at least the properties 'width' and 'height' of the image, that
         *  shall be inserted into a specified drawing.
         *
         * @returns {Object|Null}
         *  An object that contains a boolean value 'isHorizontal' that is set to true, if the left
         *  and right side of the image shall be cropped. If it is false, the top and bottom side
         *  of the image shall be cropped. Additionally a value in percent is specified. If drawing
         *  and image have the same height-to-width ratio, this value is 0 and no croppping is
         *  required.
         *  If the parameter are not sufficient for a calculation, null is returned.
         */
        this.calculateCroppingValue = function (drawingSize, imageSize) {

            if (!drawingSize || !imageSize || !_.isNumber(drawingSize.width) || !_.isNumber(drawingSize.height) || !_.isNumber(imageSize.width) || !_.isNumber(imageSize.height)) {
                return null;
            }

            var drawingRatio = drawingSize.height / drawingSize.width;
            var imageRatio = imageSize.height / imageSize.width;

            var horizontalCrop = drawingRatio > imageRatio;

            // calculating the percentage for left/right or top/bottom cropping
            // -> using 0.5 of 100%, because cropping is done on two sides (maximum is 50% on each side)
            var value = horizontalCrop ? Utils.round((1 - (imageRatio / drawingRatio)) * 50, 1) : Utils.round((1 - (drawingRatio / imageRatio)) * 50, 1);

            return { isHorizontal: horizontalCrop, value: value };
        };

        /**
         * If an image is inserted into a place holder drawing this is not of type 'pic', its size
         * must be reduced to the size of the drawing (it must be a content place holder drawing).
         * If the image is inserted into a 'pic' place holder, it is cropped.
         * The function calculates the reduced size of the drawing, if it does not fit into the
         * content place holder drawing.
         *
         * @param {Object} drawingSize
         *  An object containing at least the properties 'width' and 'height' of the drawing, in that
         *  the image shall be inserted.
         *
         * @param {Object} imageSize
         *  An object containing at least the properties 'width' and 'height' of the image, that
         *  shall be inserted into a specified drawing.
         *
         * @returns {Object|Null}
         *  An object that contains a boolean values 'width' and 'height' for the image size, if the
         *  specified image size must be reduced. If it is not necessary to reduce the image size,
         *  null is returned.
         */
        this.checkImageSizeInDrawing = function (drawingSize, imageSize) {

            var newImageSize = null;

            if (!drawingSize || !imageSize || !_.isNumber(drawingSize.width) || !_.isNumber(drawingSize.height) || !_.isNumber(imageSize.width) || !_.isNumber(imageSize.height)) {
                return null;
            }

            if ((imageSize.width > drawingSize.width) || (imageSize.height > drawingSize.height)) {

                var horzFactor = imageSize.width / drawingSize.width;
                var vertFactor = imageSize.height / drawingSize.height;
                var factor = Math.max(horzFactor, vertFactor);

                newImageSize = { width: Utils.round(imageSize.width / factor, 1), height: Utils.round(imageSize.height / factor, 1) };
            }

            return newImageSize;
        };

        /**
         * Preparations for inserting a text frame. A selection box can be opened
         * so that the user can determine the position of the text frame.
         */
        this.insertTextFramePreparation = function () {

            var // the active selection box
                selectionBox = self.getSelectionBox(),
                // the name of the text frame mode
                textFrameMode = 'textframe',
                // the stop handler for the text frame mode
                stopHandler = function (box) {
                    var options = {
                        // converting the pixel relative to app content root node in 1/100 mm relative to slide
                        rectangle: PresentationUtils.convertAppContentBoxToOperationBox(app, box)
                    };
                    // fix for Bug 48882
                    if (options.rectangle.width < 300) { options.wordWrap = false; }

                    // inserting the new text frame
                    self.insertTextFrame(options);
                    // reset to the previous selection box mode
                    selectionBox.setPreviousMode();
                };

            if (self.isInsertTextFrameActive()) {
                selectionBox.cancelSelectionBox();
                return false;
            }

            // registering the mode for text frame insertion (if not already done)
            if (!selectionBox.isModeRegistered(textFrameMode)) {
                selectionBox.registerSelectionBoxMode(textFrameMode, { contentRootClass: 'textframemode', preventScrolling: true, leaveModeOnCancel: true, stopHandler: stopHandler });
            }

            // activating the mode for text frame insertion
            selectionBox.setActiveMode(textFrameMode);

            return true;
        };

        /**
         * Whether the 'textframe' mode is the active mode.
         *
         * @returns {Boolean}
         *  Whether the 'textframe' mode is the active mode.
         */
        this.isInsertTextFrameActive = function () {
            return self.getSelectionBox() && self.getSelectionBox().getActiveMode() === 'textframe';
        };

        /**
         * Whether the 'shape' mode is the active mode.
         *
         * @returns {Boolean}
         *  Whether the 'shape' mode is the active mode.
         */
        this.isInsertShapeActive = function () {
            return self.getSelectionBox() && self.getSelectionBox().getActiveMode() === 'shape';
        };

        /**
         * Inserting a text frame element with a default size. This is useful for
         * keyboard handling.
         *
         * @param {Object} [options]
         *  For Optional parameters see insertTextFrame.
         *
         * @returns {jQuery.Promise}
         *  A promise that is already resolved.
         */
        this.insertTextFrameWithDefaultBox = function (options) {
            return self.insertTextFrame(Utils.extendOptions(options, { rectangle: getDefaultRactangleForInsert() }));
        };

        /**
         * Inserting a shape with a default size. This is useful for keyboard handling.
         *
         * @param {String} presetShapeId
         *  The identifier for the shape.
         *
         * @param {Object} [options]
         *  For Optional parameters see insertShape.
         *
         * @returns {undefined}
         *  The return value of 'self.insertShape'.
         */
        this.insertShapeWithDefaultBox = function (presetShapeId, options) {
            var preset = DrawingLabels.getPresetShape(presetShapeId);
            var lineAttrs = { type: 'solid', color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 50000 }] }, width: 53 };

            return self.insertShape(Utils.extendOptions(options, { rectangle: getDefaultRactangleForInsert(), presetShape: preset.id, lineAttrs: _.extend(lineAttrs, preset.lineAttrs) }));
        };

        /**
         * Inserting a text frame drawing node of type 'shape' or inserting a table drawing.
         * This insertion can be done via a selection box (text frame), via the table size
         * dialog or via one of the buttons in a place holder drawing.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.insertTable=false]
         *   Whether a text frame containing a table shall be inserted.
         *  @param {Object} [options.size=null]
         *   The size of the table to be inserted. This is only evaluated, if the option
         *   'insertTable' is true. The size object must contain the two properties
         *   'width' and 'height'.
         *  @param {Object} [options.rectangle=null]
         *   A user defined rectangle containing the properties 'top', 'left', 'width'
         *   and 'height' in 1/100 mm.
         *  @param {Object} [options.attributes=null]
         *   The explicit attributes of the drawing to be replaced. This will only be
         *   evaluated, if the option 'removeExisting' is set to true and if 'insertTable'
         *   is set to true.
         *  @param {Number[]} [options.position=null]
         *   The logical position at that the text frame drawing shall be inserted.
         *  @param {Number[]} [options.removeExisting=false]
         *   Whether an existing drawing at the specified position shall be removed. This can
         *   only be used, if a position was specified as option.
         *  @param {Boolean} [options.selectDrawingOnly=false]
         *   If the inserted drawing is selected or if the selection is a text
         *   selection inside the drawing.
         *
         * @returns {jQuery.Promise}
         *  A promise of the call back function. This is already resolved.
         */
        this.insertTextFrame = function (options) {

            // the undo manager returns the return value of the callback function
            return self.getUndoManager().enterUndoGroup(function () {

                // helper function, that generates the operations
                function doInsertTextFrame() {

                    var // the operations generator
                        generator = self.createOperationsGenerator(),
                        // the default width of the text frame in 1/100 mm
                        defaultWidth = 8000,
                        // whether a paragraph or a table shall be included into the text frame
                        insertTable = Utils.getBooleanOption(options, 'insertTable', false),
                        // the table size
                        size = Utils.getObjectOption(options, 'size', null),
                        // the position width and height of the text frame
                        userBox = Utils.getObjectOption(options, 'rectangle', null),
                        // the logical position at which the drawing shall be inserted
                        startPos = Utils.getArrayOption(options, 'position', null),
                        // the explicit attributes of a place holder drawing that will be replaced
                        expAttrs = Utils.getObjectOption(options, 'attributes', null),
                        // the current cursor position
                        start = startPos || self.getNextAvailablePositionInActiveSlide(),
                        // whether an existing drawing at the specified position shall be removed (position must be specified)
                        removeExisting = startPos && Utils.getBooleanOption(options, 'removeExisting', false),
                        // the default drawing attributes
                        drawingAttrs = Utils.extendOptions({ width: defaultWidth }, self.getInsertDrawingAttibutes()),
                        // the default border attributes
                        lineAttrs = { color: { type: 'rgb', value: '000000' }, style: 'single', type: 'none', width: Border.getWidthForPreset('thin') },
                        // the default fill color attributes (pptx filter requires no background, the default in PP)
                        fillAttrs = { type: 'none' },
                        // the default geometry attributes
                        geometryAttrs = { presetShape: 'rect', avList: {} },
                        // wordWrap for shape attributes
                        wordWrap = Utils.getBooleanOption(options, 'wordWrap', true),
                        // the default attributes
                        attrs = { drawing: drawingAttrs, shape: { autoResizeHeight: true, noAutoResize: false, wordWrap: wordWrap }, line: lineAttrs, fill: fillAttrs, geometry: geometryAttrs },
                        // the position of the first paragraph inside the text frame
                        paraPos = _.clone(start),
                        // a required style ID for style handling
                        parentStyleId = null,
                        // the attributes of the inserted table
                        tableAttributes = null,
                        // default table style
                        tableStyleId = null,
                        // the text position
                        textPos = null,
                        // target for operation - if exists
                        target = self.getActiveTarget(),
                        // the drawing styles
                        drawingStyles = self.getDrawingStyles(),
                        // a drawing object that describes a place holder drawing that will be replaced
                        drawingObj = null,
                        // created operation
                        newOperation = null,
                        // if the drawing is selected or if the selection is inside the drawing after insert
                        selectDrawingOnly = Utils.getBooleanOption(options, 'selectDrawingOnly', false);

                    // finding a valid position for the table if not specified (but not in master view)
                    if (insertTable && !self.isMasterView()) {
                        if (!userBox && !startPos) {
                            // finding a valid position for the table if not specified (but not in master view)
                            drawingObj = self.getValidPlaceHolderPositionForInsertDrawing(PresentationUtils.TABLE_PLACEHOLDER_TYPE);
                            if (drawingObj) {
                                start = drawingObj.position;
                                userBox = drawingObj.rectangle;
                                expAttrs = drawingObj.attributes;
                                paraPos = _.clone(start);
                                removeExisting = true; // removing an existing place holder drawing
                            }
                        }

                        if (removeExisting && expAttrs) {
                            // the drawing attributes must be set to the table, rotation must be removed
                            expAttrs.drawing = expAttrs.drawing || {};
                            expAttrs.drawing.rotation = 0; // no rotation for tables
                            // expanding the drawing attributes with the explicit attributes of the existing place holder drawing
                            _.extend(attrs, expAttrs);
                            // ... but not sending phSize to the server
                            if (attrs.presentation && attrs.presentation.phSize) { delete attrs.presentation.phSize; }
                        }
                    }

                    // overwriting default attribute values for left, top, width and height with user defined values in option 'box'.
                    if (userBox) { _.extend(attrs.drawing, userBox); }

                    // removing an existing drawing, if specified
                    if (removeExisting) {
                        newOperation = { start: start };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);
                    }

                    // Adding styleId (only for ODT)
                    if (app.isODF()) { attrs.styleId = 'Frame'; }

                    // handling the required styles for the text frame
                    if (_.isString(attrs.styleId) && drawingStyles.isDirty(attrs.styleId)) {

                        // checking the parent style of the specified style
                        parentStyleId = drawingStyles.getParentId(attrs.styleId);

                        if (_.isString(parentStyleId) && drawingStyles.isDirty(parentStyleId)) {
                            // inserting parent of text frame style to document
                            self.generateInsertStyleOp(generator, 'drawing', parentStyleId, true);
                            drawingStyles.setDirty(parentStyleId, false);
                        }

                        // insert text frame style to document
                        self.generateInsertStyleOp(generator, 'drawing', attrs.styleId);
                    }

                    // position of the paragraph/row inside the text frame
                    paraPos.push(0);

                    if (insertTable) {

                        tableAttributes = { tableGrid: [], width: 'auto' };
                        tableStyleId = self.getDefaultUITableStylesheet();

                        // prepare table column widths (values are relative to each other)
                        _(size.width).times(function () { tableAttributes.tableGrid.push(1000); });

                        // set default table style
                        if (_.isString(tableStyleId)) {

                            // insert a pending table style if needed
                            Table.checkForLateralTableStyle(generator, self, tableStyleId);

                            // add table style name to attributes
                            attrs.styleId = tableStyleId;
                            // default: tables do not have first column, last column, last row and vertical bands
                            tableAttributes.exclude = ['bandsVert', 'firstCol', 'lastCol', 'lastRow'];
                        }

                        attrs.table = tableAttributes;
                        attrs.drawing.noGroup = true; // table cannot be grouped

                        // text frames are drawings of type shape
                        newOperation = { attrs: attrs, start: start, type: 'table' };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DRAWING_INSERT, newOperation);  // this includes insertion of a table node

                        // inserting the rows without inserting a table via operation

                        newOperation = { start: _.clone(paraPos), count: size.height, insertDefaultCells: true, attrs: { row: { height: 1500 } } }; // TODO: 1500 fix?
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.ROWS_INSERT, newOperation);

                        textPos = _.clone(paraPos).concat([0, 0, 0]); // adding position of cell, paragraph and text

                    } else {

                        // text frames are drawings of type shape
                        newOperation = { attrs: attrs, start: start, type: 'shape' };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DRAWING_INSERT, newOperation);

                        // add a paragraph into the shape, so that the cursor can be set into the text frame
                        // -> an operation is required for this step, so that remote clients are also prepared (-> no implicit paragraph)
                        newOperation = { start: paraPos };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.PARA_INSERT, newOperation);

                        // reducing distance of paragraphs inside the text frame
                        newOperation = { start: paraPos, attrs: { paragraph: { marginBottom: 0 } } };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                        textPos = Position.appendNewIndex(_.clone(paraPos), 0);
                    }

                    // apply all collected operations
                    self.applyOperations(generator);

                    if (selectDrawingOnly) {
                        // select the drawing only
                        self.getSelection().setTextSelection(start, Position.increaseLastIndex(start));
                    } else {
                        // setting cursor into text frame
                        self.getSelection().setTextSelection(textPos);
                    }
                }

                doInsertTextFrame();
                return $.when();

            }, this);  // enterUndoGroup()
        };

        /**
         * Preparations for inserting a shape. A selection box can be opened
         * so that the user can determine the position of the shape.
         *
         * @param {String} presetShapeId
         *  The identifier for the shape.
         *
         * @returns {Boolean}
         *  Whether a selection box could be activated.
         */
        this.insertShapePreparation = function (presetShapeId) {

            var preset = null,
                selectionBox = self.getSelectionBox(),
                shapeMode = 'shape',
                canvasDiv = $('<div style="position: absolute; z-index: 99" class="hoshi"><canvas class="' + DrawingFrame.CANVAS_CLASS + '"></div>'),
                fillAttrs = { type: 'solid', color: { type: 'scheme', value: 'accent1' } },
                lineAttrs = { type: 'solid', color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 50000 }] }, width: 53 },
                moveHandler = function (box) {
                    if (!preset) { return; }
                    canvasDiv.css({ left: box.left, top: box.top });
                    var rect = { left: 0, top: 0, width: box.width, height: box.height };
                    var canvas = DrawingFrame.initializeCanvas(canvasDiv, rect, 1);
                    canvas.getNode().css('opacity', 0.6);
                    canvas.render(function (context, width, height) {
                        context.clearRect(-100, -100, width + 100, height + 100);
                        DrawingFrame.drawShape(app.getModel(), context, rect, preset.shape, fillAttrs, _.extend(lineAttrs, preset.lineAttrs), {});
                    });
                },
                startHandler = function (box) {
                    preset = DrawingLabels.getPresetShape(presetShapeId);

                    app.getView().getContentRootNode().append(canvasDiv);
                    moveHandler(box);
                },
                stopHandler = function (box) {
                    // converting the pixel relative to app content root node in 1/100 mm relative to slide
                    box = PresentationUtils.convertAppContentBoxToOperationBox(app, box);
                    // inserting the new text frame
                    self.insertShape({ rectangle: box, presetShape: preset.id, lineAttrs: lineAttrs });
                },
                finalizeHandler = function () {
                    self.executeDelayed(function () {
                        DrawingFrame.initializeCanvas(canvasDiv, { left: 0, top: 0, width: 20, height: 20 }, 1).destroy();
                        canvasDiv.remove();
                    }, { delay: 50, infoString: 'ObjectOperationMixin: insertShapePreparation', app: app });
                    // reset to the previous selection box mode
                    selectionBox.setPreviousMode();
                };

            if (self.isInsertShapeActive()) {
                selectionBox.cancelSelectionBox();
                return false;
            }

            // always registering the mode for text shape insertion
            selectionBox.registerSelectionBoxMode(shapeMode, {
                contentRootClass: 'textframemode',
                preventScrolling: true,
                leaveModeOnCancel: true,
                stopHandler: stopHandler,
                startHandler: startHandler,
                moveHandler: moveHandler,
                finalizeHandler: finalizeHandler
            });

            // activating the mode for text frame insertion
            selectionBox.setActiveMode(shapeMode);

            return true;
        };

        /**
         * Inserting a shape element at the end of the slide (logical position).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.presetShape='actionButtonHome']
         *   A string specifying a default shape object.
         *  @param {Object} [options.rectangle=null]
         *   A user defined rectangle containing the properties 'top', 'left', 'width'
         *   and 'height' in 1/100 mm.
         *  @param {Boolean} [options.selectDrawingOnly=false]
         *   If the inserted drawing is selected or if the selection is a text
         *   selection inside the drawing.
         */
        this.insertShape = function (options) {
            var presetShape = Utils.getStringOption(options, 'presetShape', 'actionButtonHome');
            var rectangle = Utils.getOption(options, 'rectangle');
            var target = self.getActiveTarget();

            var start = self.getNextAvailablePositionInActiveSlide();
            var paraPos = _.clone(start);
            paraPos.push(0);

            var generator = self.createOperationsGenerator();
            var operationOptions = {};
            var fill = { type: 'solid', color: { type: 'scheme', value: 'accent1' } };
            var line = { type: 'solid', color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 50000 }] }, width: 53 };
            _.extend(line, Utils.getObjectOption(options, 'lineAttrs', {}));
            var shape = { anchor: 'centered', anchorCentered: false, paddingLeft: 100, paddingRight: 100, paddingTop: 100, paddingBottom: 100, wordWrap: true, horzOverflow: 'overflow', vertOverflow: 'overflow' };
            var geometry = { presetShape: presetShape, avList: {} };
            var drawing = { name: presetShape, width: 1500, height: 1500 };
            var charAttrs = { color: { type: 'scheme', value: 'light1' } };
            var selectDrawingOnly = Utils.getBooleanOption(options, 'selectDrawingOnly', false);

            //var padding = 100;

            if (rectangle) {
                _.extend(drawing, rectangle);
                shape.paddingLeft = Math.round(rectangle.width / 10);
                shape.paddingRight = shape.paddingLeft;
                shape.paddingTop = Math.round(rectangle.height / 10);
                shape.paddingBottom = shape.paddingTop;
            }

            operationOptions.attrs = { fill: fill, line: line, shape: shape, geometry: geometry, drawing: drawing, character: charAttrs };
            operationOptions.start = start;
            operationOptions.type = 'shape';

            self.extendPropertiesWithTarget(operationOptions, target);
            generator.generateOperation(Operations.DRAWING_INSERT, operationOptions);

            var setAttrs = { attrs: { shape: { anchor: 'centered' } }, start: start };
            self.extendPropertiesWithTarget(setAttrs, target);
            generator.generateOperation(Operations.SET_ATTRIBUTES, setAttrs);

            var paraInsert = { start: paraPos };
            self.extendPropertiesWithTarget(paraInsert, target);
            generator.generateOperation(Operations.PARA_INSERT, paraInsert);

            this.applyOperations(generator);

            if (selectDrawingOnly) {
                // select the drawing only
                self.getSelection().setTextSelection(start, Position.increaseLastIndex(start));
            } else {
                // setting cursor into text frame
                self.getSelection().setTextSelection(Position.appendNewIndex(_.clone(paraPos), 0));
            }
        };

        /**
         * Generating the operations for moving drawing(s) in z-order. Drawings
         * can be moved one level upwards or backwards or they can be moved to
         * the background or the foreground.
         *
         * @param {String} value
         *  A string describing the change of z-order of the selected drawing(s).
         *  Supported values are: 'front', 'back', 'forward', 'backward'.
         *  'forward' and 'backward' mean, that the selected drawing(s) shall be moved one
         *  step to the front ('forward') or to the back ('backward').
         *  'front' and 'back' that the selected drawing(s) shall be moved completely into
         *  the foreground ('front') or the background ('back').
         *
         * @returns {Number}
         *  The number of generated operations.
         */
        this.changeDrawingOrder = function (value) {

            var // whether the drawing shall be moved upwards or downwards
                upwards = (value === 'front' || value === 'forward'),
                // whether the drawing shall be moved to foreground or background
                full = (value === 'front' || value === 'back'),
                // the selection object
                selection = self.getSelection(),
                // the logical start position of the selected drawing
                startPosition = selection.getStartPosition(),
                // the logical new destination position of the selected drawing
                destPosition = null,
                // the operations generator
                generator = null,
                // the options for the operation
                operationOptions = null,
                // the number of drawings on the current level (only needed, if upwards is true)
                drawingCount = 0,
                // an array with all logical positions in a multi drawing selection
                allPositions = null,
                // an array with all indices (the last numbers of the logical positions) in a multi drawing selection
                allSelectedIndices = null,
                // the logical position of the parent of the drawing
                rootPosition = null,
                // a container for all destination indices
                destIndices = null,
                // a container for all indices for the move operation(s)
                moveIndices = null,
                // a container for all selected drawing nodes
                allSelectedDrawings = null,
                // the number of generated operations
                operationCounter = 0;

            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {

                // handle multi drawing selection
                allPositions = selection.getArrayOfLogicalPositions();
                allSelectedIndices = _.map(allPositions, function (pos) { return _.last(pos); });
                rootPosition = _.initial(allPositions[0]);
                destIndices = [];
                moveIndices = [];

                if (upwards) {
                    allSelectedIndices.reverse();  // reverting the order of the selected drawings when moving upwards
                    drawingCount = selection.getFirstSelectedDrawingNode().parent().children(DrawingFrame.NODE_SELECTOR).length; // only needed, if upwards is true
                }

                if (full) {

                    if (upwards) {

                        _.each(allSelectedIndices, function (index) {

                            var newIndex = drawingCount - 1;

                            // finding the smallest number (greater or equal 0) smaller than index, that is not part of the selection
                            while (newIndex > index && _.contains(allSelectedIndices, newIndex)) { newIndex--; }

                            if (newIndex > index) {
                                while (_.contains(destIndices, newIndex)) { newIndex--; }
                                destIndices.push(newIndex);
                                moveIndices.push({ from: index, to: newIndex });
                            }
                        });

                    } else {

                        _.each(allSelectedIndices, function (index) {

                            var newIndex = 0;

                            // finding the smallest number (greater or equal 0) smaller than index, that is not part of the selection
                            while (newIndex < index && _.contains(allSelectedIndices, newIndex)) { newIndex++; }

                            if (newIndex < index) {
                                while (_.contains(destIndices, newIndex)) { newIndex++; }
                                destIndices.push(newIndex);
                                moveIndices.push({ from: index, to: newIndex });
                            }
                        });
                    }

                } else {

                    if (upwards) {

                        _.each(allSelectedIndices, function (index) {

                            // Not upwards: every index need to be increased to a position of a not selected drawing
                            var newIndex = index + 1;

                            // starting at the last index, trying to find the previous position that is not in the selection
                            while (newIndex < drawingCount && _.contains(allSelectedIndices, newIndex)) { newIndex++; }

                            if (newIndex < drawingCount && newIndex !== index) {
                                while (_.contains(destIndices, newIndex)) { newIndex--; }
                                destIndices.push(newIndex);
                                moveIndices.push({ from: index, to: newIndex });
                            }
                        });

                    } else {

                        _.each(allSelectedIndices, function (index) {

                            // Not upwards: every index need to be reduced to a position of a not selected drawing
                            var newIndex = index - 1;

                            // starting at the last index, trying to find the previous position that is not in the selection
                            while (newIndex >= 0 && _.contains(allSelectedIndices, newIndex)) { newIndex--; }

                            if (newIndex >= 0 && newIndex !== index) {
                                while (_.contains(destIndices, newIndex)) { newIndex++; }
                                destIndices.push(newIndex);
                                moveIndices.push({ from: index, to: newIndex });
                            }
                        });
                    }
                }

                if (moveIndices.length > 0) {

                    allSelectedDrawings = selection.getArrayOfSelectedDrawingNodes();

                    generator = self.createOperationsGenerator();

                    _.each(moveIndices, function (index) {
                        var dest = Position.appendNewIndex(rootPosition, index.to);
                        operationOptions = { start: Position.appendNewIndex(rootPosition, index.from), to: dest };
                        generator.generateOperation(Operations.MOVE, operationOptions);
                    });

                    operationCounter = generator.getOperationCount();

                    self.applyOperations(generator);

                    // setting the multi selection using the selected drawing nodes
                    selection.setMultiDrawingSelection(allSelectedDrawings);
                }

            } else {

                // single drawing selection

                drawingCount = upwards ? selection.getAnyDrawingSelection().parent().children(DrawingFrame.NODE_SELECTOR).length : 0; // only needed, if upwards is true

                // also handling the case, if the cursor is positioned inside a shape
                if (selection.isAdditionalTextframeSelection()) {
                    startPosition = Position.getOxoPosition(self.getRootNode(self.getActiveSlideId()), selection.getSelectedTextFrameDrawing(), 0);
                }

                if ((!upwards && _.last(startPosition) > 0) || (upwards && _.last(startPosition) < (drawingCount - 1))) {

                    if (full) {
                        destPosition = upwards ? Position.appendNewIndex(_.initial(startPosition), drawingCount - 1) : Position.appendNewIndex(_.initial(startPosition), 0);
                    } else {
                        destPosition = Position.increaseLastIndex(startPosition, upwards ? 1 : -1);
                    }

                    // generating the new operation
                    generator = self.createOperationsGenerator();
                    operationOptions = { start: startPosition, to: destPosition };
                    generator.generateOperation(Operations.MOVE, operationOptions);

                    operationCounter = generator.getOperationCount();

                    // self.extendPropertiesWithTarget(newOperation, self.getActiveTarget()); // TODO: Is it necessary to add target to operation?
                    self.applyOperations(generator);

                    // setting the selection to the new drawing position
                    selection.setTextSelection(destPosition, Position.increaseLastIndex(destPosition));
                }
            }

            return operationCounter;
        };

        /**
         * Rotate or flip selected drawing or group of drawings, depending on the selected value from the toolbar.
         *
         * @param {String} value
         *  A string describing the type of drawing modification:
         *      'cw90' - rotate right for 90 degrees
         *      'ccw90' - rotate left for 90 degrees
         *      'flipH' - flip horizontal
         *      'flipV' - flip vertical
         */
        this.rotateFlipDrawing = function (value) {
            // the operations generator
            var generator = self.createOperationsGenerator();
            // the selection object
            var selection = self.getSelection();
            // collection of all currently selected drawings
            var allDrawingSelections;
            // an array of selected drawing(s) after operations are applied
            var selectedDrawings;
            // if drawings need to be rotated
            var isRotate = (_.isString(value) && value.indexOf('cw90')) > -1;
            // rotation angle
            var rotateAngle = (value === 'ccw90') ? -90 : 90;
            // if drawings need to be fliped
            var isFlip = (_.isString(value) && value.indexOf('flip')) > -1;

            if (!value || !_.isString(value)) {
                Utils.error('rotateFlipDrawing(): missing function argument!');
                return;
            }

            if (selection.isMultiSelection()) {
                allDrawingSelections = selection.getMultiSelection();
            } else {
                if (selection.isAdditionalTextframeSelection()) {
                    var startPos = Position.getOxoPosition(self.getRootNode(self.getActiveSlideId()), selection.getSelectedTextFrameDrawing(), 0);
                    var endPos = Position.increaseLastIndex(startPos);
                    allDrawingSelections = [{ drawing: selection.getSelectedTextFrameDrawing(), startPosition: startPos, endPosition: endPos }];
                } else {
                    allDrawingSelections = [{ drawing: selection.getSelectedDrawing(), startPosition: selection.getStartPosition(), endPosition: selection.getEndPosition() }];
                }
            }

            _.each(allDrawingSelections, function (oneDrawingSelection) {
                // the properties for the new setAttributes operation
                var operationProperties = { start: oneDrawingSelection.startPosition, attrs: { drawing: {} } };
                // drawing node from multiselection
                var drawingNode = self.getSelection().getDrawingNodeFromMultiSelection(oneDrawingSelection);
                // the explicit attributes at the specified element
                var elemAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
                // old rotation angle value
                var oldRotateValue;
                // old flip value
                var oldFlipValue;
                // new rotation angle value
                var newRotationAngle;

                if (isRotate && !DrawingFrame.isTableDrawingFrame(drawingNode)) { // no rotation of tables
                    oldRotateValue = elemAttrs && elemAttrs.drawing && _.isNumber(elemAttrs.drawing.rotation) ? elemAttrs.drawing.rotation : 0;
                    newRotationAngle = oldRotateValue + rotateAngle;
                    // normalize negative and angles greater than 360
                    if (newRotationAngle < 0) {
                        newRotationAngle += 360;
                    }
                    if (newRotationAngle > 360) {
                        newRotationAngle -= 360;
                    }
                    operationProperties.attrs.drawing.rotation = newRotationAngle;
                } else if (isFlip) {
                    oldFlipValue = (elemAttrs && elemAttrs.drawing && elemAttrs.drawing[value]) || false;
                    operationProperties.attrs.drawing[value] = !oldFlipValue;
                }

                PresentationUtils.extendPlaceholderProp(app, drawingNode, operationProperties);
                self.extendPropertiesWithTarget(operationProperties, self.getActiveTarget());
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

            });

            // Applying the operations (undo group is created automatically)
            self.applyOperations(generator);

            // after operations are applied, update resize mouse poiters
            selectedDrawings = selection.getAllDrawingsInSelection();
            if (selectedDrawings) {
                _.each(selectedDrawings, function (drawingNode) {
                    var angle = DrawingFrame.getDrawingRotationAngle(self, drawingNode) || 0;
                    DrawingFrame.updateResizersMousePointers(drawingNode, angle);
                });
            }
        };

        /**
         * This function generates setAttributes operation for selected drawings that are moved or
         * resized with the cursor keys via keyboard.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @returns {Boolean}
         *  Whether this keyboard event was handled within this function. This is the case, if this
         *  is a right/left/up/down cursor key together with any shift or ctrl and if at least one
         *  drawing is selected.
         */
        this.handleDrawingOperations = function (event) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // whether event is a Left/Right cursor key
                leftRightCursor = (event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW),
                // whether event is a Left/Right cursor key
                upDownCursor = (event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW),
                // a container for the selected drawings
                allDrawingSelections = null,
                // the selection object
                selection = self.getSelection();

            if (!leftRightCursor && !upDownCursor) { return false; }

            allDrawingSelections = selection.isMultiSelection() ? selection.getMultiSelection() : selection.generateMultiSelectionObjectFromDrawingSelection();

            _.each(allDrawingSelections, function (oneDrawingSelection) {
                generateDrawingAttributeOperation(generator, event, oneDrawingSelection, leftRightCursor);
            });

            // Applying the operations (undo group is created automatically)
            self.applyOperations(generator);

            // no further handling of event required
            return true;
        };

        /**
         * Handling the insertion of a table or image by clicking on the template image
         * inside an empty place holder drawing.
         *
         * @param {jQuery} node
         *  The jQueryfied parent node of the node that is the event target.
         */
        this.handleTemplateImageEvent = function (node) {

            // the drawing node
            var drawing = null;
            // the logical position of the drawing
            var pos = null;
            // the drawing attributes of the affected drawing node
            var drawingAttrs = null;
            // the explicit attributes of the affected drawing node
            var explicitAttributes = null;
            // the position of the new inserted drawing (left, top, width and height)
            var rectangle = null;

            // helper function to get the affected drawing node and its logical position
            function getDrawingPosition() {
                // getting the drawing node
                drawing = node.closest('.drawing');
                pos = Position.getOxoPosition(self.getRootNode(self.getActiveSlideId()), drawing, 0);
                drawingAttrs = self.getDrawingStyles().getElementAttributes(drawing).drawing;
                explicitAttributes = AttributeUtils.getExplicitAttributes(drawing);
                rectangle = { left: drawingAttrs.left, top: drawingAttrs.top, width: drawingAttrs.width, height: drawingAttrs.height };
            }

            if (DOM.isImagePlaceholderTemplateButton(node)) {

                getDrawingPosition();

                // showing the dialog to select an image
                self.showInsertImageDialog('local', { position: _.clone(pos), attributes: _.copy(explicitAttributes, true), rectangle: rectangle, removeExisting: true });

            } else if (DOM.isTablePlaceholderTemplateButton(node)) {

                getDrawingPosition();

                // inserting a table drawing at the specified position with specified size (2 rows, 5 columns) and remove an existing drawing
                self.insertTextFrame({ insertTable: true, position: _.clone(pos), size: { width: 5, height: 2 }, attributes: _.copy(explicitAttributes, true), rectangle: rectangle, removeExisting: true });
            }
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the insertDrawing operation.
         */
        this.insertDrawingHandler = function (operation) {

            var // the undo operation for the insertSlide operation
                undoOperation = null;

            if (!implInsertDrawing(operation.type, operation.start, operation.attrs, operation.target)) {
                return false;
            }

            if (self.getUndoManager().isUndoEnabled()) {
                // not registering for undo if this is a drawing inside a drawing group (36150)
                undoOperation = Position.isPositionInsideDrawingGroup(self.getNode(), operation.start) ? null : { name: Operations.DELETE, start: operation.start };
                if (undoOperation) { self.extendPropertiesWithTarget(undoOperation, operation.target); }
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        /**
         * The handler for the insertTable operation.
         */
//        this.insertTableHandler = function (operation) {
//
//            var // the undo operation for the insertSlide operation
//                undoOperation = null;
//
//            if (!implInsertTable(operation.start, operation.attrs, operation.target)) {
//                return false;
//            }
//
//            // generate undo/redo operations
//            if (self.getUndoManager().isUndoEnabled()) {
//                undoOperation = { name: Operations.DELETE, start: operation.start };
//                self.extendPropertiesWithTarget(undoOperation, operation.target);
//                self.getUndoManager().addUndo(undoOperation, operation);
//            }
//
//            return true;
//        };

        /**
         * The handler for the move operation.
         */
        this.moveHandler = function (operation) {

            var // the undo operation for the insertSlide operation
                undoOperation = null;

            if (!implMove(operation.start, operation.to, operation.target)) {
                return false;
            }

            // generate undo/redo operations
            if (self.getUndoManager().isUndoEnabled()) {
                undoOperation = { name: Operations.MOVE, start: operation.to, to: operation.start };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        /**
         * The handler for the change alginment of the selected drawings operations.
         */
        this.changeAlignment = function (direction) {

            var selection = self.getSelection(),
                allDrawingSelections = selection.isMultiSelection() ? selection.getMultiSelection() : selection.generateMultiSelectionObjectFromDrawingSelection({ anyDrawing: true }),
                generator = self.createOperationsGenerator(),
                operationProperties;
            // change the alignment of multi selected drawings

            // Only one drawing is selected, the alignment of the drawing will be oriented on the page.
            if (allDrawingSelections.length === 1) {
                operationProperties = changeAlignment.getOperationPropertiesForSingleSelection(allDrawingSelections[0], direction);
                if (operationProperties !== null) {
                    self.extendPropertiesWithTarget(operationProperties, self.getActiveTarget());
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                }
            } else {

                changeAlignment.setAlignmentDirection(direction);

                // derermine the drawing/s which is e.g. on the lower left position, to the change the alignment
                _.each(allDrawingSelections, function (drawingSelection) {
                    changeAlignment.setBaseDrawing(drawingSelection);
                });

                // generate the operations to move the drawings to the new position
                _.each(allDrawingSelections, function (drawingSelection) {
                    operationProperties = changeAlignment.getOperationPropertiesForMultiSelection(drawingSelection, drawingSelection.startPosition);

                    if (operationProperties !== null) {
                        self.extendPropertiesWithTarget(operationProperties, self.getActiveTarget());
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                    }
                });
            }
            self.applyOperations(generator);

            return true;
        };

        /**
         * The handler for distribution of drawings horizontally or vertically,
         * on slide or among selection of drawings.
         *
         * @param {String} mode
         *  Direction: horizontal or vertical, and on slide or among selection of drawings.
         */
        this.distributeDrawings = function (mode) {
            var selection = self.getSelection();
            var allDrawingSelections = selection.isMultiSelection() ? selection.getMultiSelection() : selection.generateMultiSelectionObjectFromDrawingSelection({ anyDrawing: true });
            var allDrawingsCopy = [];
            var allDrawingSelLength = allDrawingSelections.length;
            var activeSlideId = self.getActiveSlideId();
            var generator = self.createOperationsGenerator();
            var pageStyles = self.getStyleCollection('page');
            var pageAttrs = pageStyles.getElementAttributes(self.getRootNode(activeSlideId));
            var page = pageAttrs && pageAttrs.page;
            var operationProperties;
            var oneDirection, numSplits;
            var minBoundary, maxBoundary, distRange;
            var usedSpace = 0, startPoint = 0, distributionSpace, calculatedPosition = 0;
            var isHorizontal = mode.indexOf('horz') > -1;
            var isSlideDist = mode.indexOf('Slide') > -1;
            var ltPosProp = isHorizontal ? 'left' : 'top';
            var whProperty = isHorizontal ? 'width' : 'height';

            if (allDrawingSelLength === 1) { // for only one drawing in selection, use align to slide
                oneDirection = isHorizontal ? 'center' : 'middle';
                operationProperties = changeAlignment.getOperationPropertiesForSingleSelection(allDrawingSelections[0], oneDirection);
                if (operationProperties !== null) {
                    self.extendPropertiesWithTarget(operationProperties, self.getActiveTarget());
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                }
            } else {
                _.each(allDrawingSelections, function (drawingSelection) {
                    var drawingAttrs = getDrawingAttrsByNode(drawingSelection.drawing);
                    var rotationAngle = DrawingFrame.getDrawingRotationAngle(self, drawingSelection.drawing);
                    var rotatedAttrs = DrawingUtils.getRotatedDrawingPoints(drawingAttrs, rotationAngle);
                    var finalAttrs = (Math.abs(drawingAttrs.width - drawingAttrs.height) < 300) ? drawingAttrs : rotatedAttrs; // ignore rotation for (almost) squared shapes

                    allDrawingsCopy.push({
                        posAttrs: finalAttrs,
                        centerPos: rotatedAttrs[ltPosProp] + rotatedAttrs[whProperty] / 2,
                        leftDiff: finalAttrs.left - drawingAttrs.left,
                        topDiff: finalAttrs.top - drawingAttrs.top,
                        startPosition: _.clone(drawingSelection.startPosition)
                    });
                    usedSpace += isHorizontal ? finalAttrs.width : finalAttrs.height;
                });
                allDrawingsCopy.sort(function (a, b) { // sort by center of drawing
                    if (a.centerPos < b.centerPos) { return -1; }
                    if (a.centerPos > b.centerPos) { return 1; }
                    return 0;
                });

                if (isSlideDist || allDrawingSelLength === 2) { // if only 2 drawings, behavior is like slide distribution
                    minBoundary = 0;
                    maxBoundary = isHorizontal ? page.width : page.height;
                    distRange = maxBoundary - minBoundary;
                    numSplits = allDrawingSelLength + (distRange >= usedSpace ?  1 : -1);
                    distributionSpace = (distRange - usedSpace) / numSplits;
                    startPoint = Math.max(0, (distRange > usedSpace ? distributionSpace : 0));
                } else {
                    minBoundary = _.first(allDrawingsCopy).posAttrs[ltPosProp];
                    maxBoundary = _.last(allDrawingsCopy).posAttrs[ltPosProp] + _.last(allDrawingsCopy).posAttrs[whProperty];
                    distRange = maxBoundary - minBoundary;
                    numSplits = allDrawingSelLength - 1;
                    distributionSpace = (distRange - usedSpace) / numSplits;
                    startPoint = minBoundary;
                }
                calculatedPosition = startPoint;
                _.each(allDrawingsCopy, function (drawingSelection, index) {
                    var drawingSpace = isHorizontal ? drawingSelection.posAttrs.width : drawingSelection.posAttrs.height;
                    var rotationDiff = isHorizontal ? drawingSelection.leftDiff : drawingSelection.topDiff;
                    if (isSlideDist || (index !== 0 && index !== allDrawingSelLength - 1)) { // if is not slide distribution, don't send ops for first and last drawing in range
                        operationProperties = { start: drawingSelection.startPosition, attrs: { drawing: {} } };
                        operationProperties.attrs.drawing[ltPosProp] = Utils.round(calculatedPosition - rotationDiff, 1);
                        if (operationProperties !== null) {
                            self.extendPropertiesWithTarget(operationProperties, self.getActiveTarget());
                            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                        }
                    }
                    calculatedPosition += drawingSpace + distributionSpace;
                });
            }

            self.applyOperations(generator);
            return true;
        };

        /**
         * Finding a valid logical position for inserting a table or an image.
         * First a selected empty text frame is used, then the first not selected
         * empty text frame.
         * Order of inserting a table or image: First a 'specialized' template
         * place holder of type 'tbl' is used. Then a 'global' body place holder,
         * at which no type is explicitely set. This is determined by
         * 'PresentationUtils.isContentBodyPlaceHolderAttributeSet'.
         * In this case a selected content body is not preferred to a specialized table
         * place holder drawing.
         *
         * @param {String} type
         *  The type of the drawing that shall be inserted. Currently the types
         *  'pic' and 'tbl' are supported.
         *
         * @returns {Object|Null}
         *  An object containing the logical position of an empty place holder drawing,
         *  its rectangle position and its explicit attributes. The property names are
         *  'position', 'rectangle' and 'attributes'.
         *  Or null, if no valid placeholder can be found.
         */
        this.getValidPlaceHolderPositionForInsertDrawing = function (type) {
            return _.first(this.getAllValidPlaceHolderPositionsForInsertDrawing(type, { firstMatch: true })) || null;
        };

        /**
         * Finding all valid logical positions for inserting a table or an image.
         * First a selected empty text frame is used, then the first not selected
         * empty text frame.
         * Order of inserting a table or image: First a 'specialized' template
         * place holder of type 'tbl' is used. Then a 'global' body place holder,
         * at which no type is explicitely set. This is determined by
         * 'PresentationUtils.isContentBodyPlaceHolderAttributeSet'.
         * In this case a selected content body is not preferred to a specialized table
         * place holder drawing.
         *
         * @param {String} type
         *  The type of the drawing that shall be inserted. Currently the types
         *  'pic' and 'tbl' are supported.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.firstMatch=false]
         *      If true returns the first and best match only.
         *
         * @returns {Array}
         *  An array of objects containing the logical position of an empty place holder drawing,
         *  its rectangle position and its explicit attributes. The property names are
         *  'position', 'rectangle' and 'attributes'.
         */
        this.getAllValidPlaceHolderPositionsForInsertDrawing = function (type, options) {

            // whether to look for all or the first valid place holder
            var firstMatch = Utils.getBooleanOption(options, 'firstMatch', false);
            // the active slide
            var slide = self.getSlideById(self.getActiveSlideId());
            // an optionally selected drawing
            var selectedDrawing = self.getSelection().getAnyDrawingSelection();
            // the drawings on the slide
            var allDrawings = $(slide).children('div.drawing');
            // the resulting array of place holder positions
            var result = [];

            // helper function to find a valid empty place holder drawing of specified type or of type 'body'
            function isValidDrawing(drawing, localType, allowContentPlaceHolder) {

                var isValid = false;
                var attrs = AttributeUtils.getExplicitAttributes(drawing);
                var drawingType = null;

                if (PresentationUtils.isPlaceHolderAttributeSet(attrs)) {

                    drawingType = PresentationUtils.getPlaceHolderDrawingType(drawing, attrs);

                    if (allowContentPlaceHolder) {
                        isValid = (!drawingType || drawingType === localType) && PresentationUtils.isEmptyPlaceHolderDrawing(drawing);
                    } else {
                        isValid = (drawingType === localType) && PresentationUtils.isEmptyPlaceHolderDrawing(drawing);
                    }
                }

                return isValid;
            }

            // helper function to get place holder data from the drawing node
            function getPlaceHolderData(drawingNode) {

                // the logical position of a valid empty place holder drawing
                var pos = Position.getOxoPosition(self.getCurrentRootNode(), drawingNode, 0);
                // the merged drawing attributes
                var drawingAttrs = self.getDrawingStyles().getElementAttributes(drawingNode).drawing;
                // the explicit drawing attributes
                var explicitAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
                // the position of the drawing
                var rectangle = { left: drawingAttrs.left, top: drawingAttrs.top, width: drawingAttrs.width, height: drawingAttrs.height };

                return { position: pos, rectangle: rectangle, attributes: explicitAttrs };
            }

            if (selectedDrawing && isValidDrawing(selectedDrawing, type, true)) {
                // using the already selected drawing. This shall always be used first. In PP it seems, that
                // the selected drawing is used as first drawing for images, but not for tables.
                result.push(getPlaceHolderData(selectedDrawing));
            }

            if (!firstMatch || _.isEmpty(result)) {
                // iterating over all drawings on slide that have the correct type
                Utils.iterateArray(allDrawings, function (oneDrawing) {
                    if (isValidDrawing(oneDrawing, type, false)) { // not allowing content body place holders -> only those with the correct type
                        result.push(getPlaceHolderData(oneDrawing));
                        if (firstMatch) {
                            return Utils.BREAK; // leaving the loop
                        }
                    }
                });
            }

            if (!firstMatch || _.isEmpty(result)) {
                // iterating over all drawings on slide that can also be content body place holders
                Utils.iterateArray(allDrawings, function (oneDrawing) {
                    if (isValidDrawing(oneDrawing, type, true)) {
                        result.push(getPlaceHolderData(oneDrawing));
                        if (firstMatch) {
                            return Utils.BREAK; // leaving the loop
                        }
                    }
                });
            }

            if (result.length > 1) {
                // remove duplicates
                result = _.uniq(result, false, function (item) { return item.position.join(','); });
            }

            return result;
        };

        /**
         * Special handling for non empty place holder drawings that are deleted. In this case the content
         * of the place holder drawing is removed, not the place holder drawing itself. So after the delete
         * operation for the drawing, an insertDrawing operation is generated, to insert an empty place
         * holder drawing.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node that will be removed.
         *
         * @param {Number[]} pos
         *  The logical position at which the new drawing will be inserted. This is not necessarily
         *  the logical position of the removed drawing, because in a multi drawing selection, there
         *  might be additional drawings removed before, that are not replaced.
         *
         * @param {Object} [globalGenerator]
         *  An optional global operation generator. If it is specified, it is used. Otherwise a new
         *  operation generator is created.
         *
         * @returns {Object|Null}
         *  The operations generator object, if created or specified. Otherwise null is returned.
         */
        this.handleNonEmptyPlaceHolderDrawings = function (drawing, pos, globalGenerator) {

            // the operations generator
            var generator = globalGenerator || null;
            // the explicit drawing attributes
            var expAttrs = null;
            // the drawing attributes for the new empty placeholder drawing
            var attrs = null;

            if (PresentationUtils.isPlaceHolderDrawing(drawing) && !PresentationUtils.isEmptyPlaceHolderDrawing(drawing)) {

                // using the specified operation generator or create a new one
                generator = generator || self.createOperationsGenerator();

                // the explicit drawing attributes
                expAttrs = AttributeUtils.getExplicitAttributes(drawing);

                attrs = {};
                attrs.presentation = _.clone(expAttrs.presentation);
                if (attrs.presentation.phSize) { delete attrs.presentation.phSize; }

                attrs.drawing = {};
                attrs.drawing.noGroup = true;
                if (expAttrs.drawing && expAttrs.drawing.name) { attrs.drawing.name = expAttrs.drawing.name; }

                // generating operation for inserting empty place holder drawing (families 'presentation' and 'drawing')
                generator.generateOperation(Operations.DRAWING_INSERT, { start: _.clone(pos), type: 'shape', attrs: attrs });

                // inserting an empty paragraph in all body place holders, but also into 'title', 'ctrTitle', ...
                if (!PresentationUtils.isPlaceHolderWithoutTextAttributeSet(attrs)) {
                    generator.generateOperation(Operations.PARA_INSERT, { start: Position.appendNewIndex(pos) });
                }
            }

            return generator;
        };

        /**
         * Special handling for place holder drawings on layout slides. If these slides are deleted,
         * the attributes must be transferred to the dependent drawings on the document slides (47435).
         *
         * @param {jQuery|Node} drawing
         *  The drawing node on the layout slide that will be removed.
         *
         * @param {String} target
         *  The target of the affected layout slide.
         *
         * @param {Object} [globalGenerator]
         *  An optional global operation generator. If it is specified, it is used. Otherwise a new
         *  operation generator is created.
         *
         * @returns {Object|Null}
         *  The operations generator object, if created or specified. Otherwise null is returned.
         */
        this.handlePlaceHolderAttributeTransfer = function (drawing, target, globalGenerator) {

            // the operations generator
            var generator = globalGenerator || null;
            // the explicit attributes of the place holder drawing on the layout slide
            var expAttrs = null;
            // the IDs of all affected document slides
            var allSlideIDs = null;
            // the drawing styles object
            var drawingStyles = self.getDrawingStyles();
            // the type of place holder that was modified
            var type = null;
            // the type of place holder that was modified
            var index = 0;
            // whether the target is a target of a master slide
            var isMasterSlide = false;

            if (self.isLayoutOrMasterId(target) && PresentationUtils.isPlaceHolderDrawing(drawing)) {

                isMasterSlide = self.isMasterSlideId(target);

                // a container with the IDs of all slides affected by the modification of the master/layout slide
                allSlideIDs = self.getAllSlideIDsWithSpecificTargetParent(target);

                expAttrs = AttributeUtils.getExplicitAttributes(drawing);
                if (expAttrs.listStyle) { delete expAttrs.listStyle; }

                type = PresentationUtils.getPlaceHolderDrawingType(drawing, expAttrs) || 'body';
                index = PresentationUtils.getPlaceHolderDrawingIndex(drawing, expAttrs);

                _.each(allSlideIDs, function (id) {

                    // updating all drawings, that have the specified place holder type and index
                    _.each(drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(id, type, index), function (drawingNode) {

                        // the existing explicit attributes
                        var expDrawingAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
                        // whether the drawing is a generic place holder (type is not defined)
                        var isGenericContentBody = PresentationUtils.isContentBodyPlaceHolderAttributeSet(expDrawingAttrs);
                        // the new attributes for the drawing
                        var newAttrs = Utils.extendOptions(expAttrs, expDrawingAttrs);
                        // the attributes of the place holder drawing on the master slide
                        var masterAttrs = null;

                        // using attributes from master slide, if required (no width defined at the layout slide)
                        if (!isMasterSlide && (!(newAttrs && newAttrs.drawing && _.isNumber(newAttrs.drawing.width)))) {
                            masterAttrs = drawingStyles.getPlaceHolderAttributes(self.getMasterSlideId(target), type, index);
                            if (masterAttrs && masterAttrs.drawing && _.isNumber(masterAttrs.drawing.width)) {
                                newAttrs.drawing.left = masterAttrs.drawing.left;
                                newAttrs.drawing.top = masterAttrs.drawing.top;
                                newAttrs.drawing.width = masterAttrs.drawing.width;
                                newAttrs.drawing.height = masterAttrs.drawing.height;
                            }
                        }

                        // avoiding to set type 'body' at content body place holders
                        if (isGenericContentBody && newAttrs && newAttrs.presentation && newAttrs.presentation.phType) { delete newAttrs.presentation.phType; }

                        // setting default value for place holder index (only on document slides)
                        if (!isMasterSlide) { newAttrs.presentation.phIndex = drawingStyles.getMsDefaultPlaceHolderIndex(); }

                        // using the specified operation generator or create a new one
                        generator = generator || self.createOperationsGenerator();

                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            start: Position.getOxoPosition(self.getRootNode(isMasterSlide ? id : ''), drawingNode, 0),
                            attrs: newAttrs,
                            target: isMasterSlide ? id : self.getOperationTargetBlocker()
                        });
                    });
                });
            }

            return generator;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class ObjectOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return ObjectOperationMixin;

});
