/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/tablecellstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/editframework/utils/attributeutils'
], function (Utils, Color, Border, StyleCollection, DOM, AttributeUtils) {

    'use strict';

    var // definitions for table cell attributes
        DEFINITIONS = {

            /**
             * The number of grid columns spanned by the table cell.
             */
            gridSpan: {
                def: 1,
                scope: 'element',
                format: function (element, gridSpan) {
                    element.attr('colspan', gridSpan);
                }
            },

            /**
             * Fill color of the table cell.
             */
            fillColor: {
                def: Color.AUTO
            },

            /**
             * Style, width and color of the left table cell border.
             */
            borderLeft: {
                def: Border.NONE
            },

            /**
             * Style, width and color of the right table cell border.
             */
            borderRight: {
                def: Border.NONE
            },

            /**
             * Style, width and color of the top table cell border.
             */
            borderTop: {
                def: Border.NONE
            },

            /**
             * Style, width and color of the bottom table cell border.
             */
            borderBottom: {
                def: Border.NONE
            },

            /**
             * Inner horizontal table cell borders, used in table style sheets
             * to format inner borders of specific table areas (first/last
             * column, inner vertical bands, ...).
             */
            borderInsideHor: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Inner vertical table cell borders, used in table style sheets to
             * format inner borders of specific table areas (first/last row,
             * inner horizontal bands, ...).
             */
            borderInsideVert: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Top padding of the table cell
             */
            paddingTop: {
                def: 125,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-top', value);
                }
            },

            /**
             * Bottom padding of the table cell
             */
            paddingBottom: {
                def: 125,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px,  required for cell resizing
                    element.css('padding-bottom', value);
                }
            },

            /**
             * Left padding of the table cell
             */
            paddingLeft: {
                def: 254,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-left', value);
                }
            },

            /**
             * Right padding of the table cell
             */
            paddingRight: {
                def: 254,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-right', value);
                }
            }

        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            table: function (cell) { return cell.closest(DOM.TABLE_NODE_SELECTOR); }
        };

    // class TableCellStyles ==================================================

    /**
     * Contains the style sheets for table cell formatting attributes. The CSS
     * formatting will be read from and written to <th> and <td> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {PresentationModel} docModel
     *  The presentation document model containing instance.
     */
    function TableCellStyles(docModel) {

        var self = this;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'cell', {
            styleSheetSupport: false,
            families: 'paragraph character changes',
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateTableCellFormatting,
            previewHandler: previewTableCellFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every table cell element whose attributes have
         * been changed. Reformats the table cell according to the passed
         * attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         */
        function updateTableCellFormatting(cell, mergedAttributes) {

            var // the paragraph style sheet container
                paragraphStyles = docModel.getStyleCollection('paragraph');

            updateCSSTableCellFormatting(cell, mergedAttributes, { clearNone: true });

            // update all paragraphs in the table cell
            Utils.iterateSelectedDescendantNodes(DOM.getCellContentNode(cell), DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                paragraphStyles.updateElementFormatting(paragraph, { baseAttributes: mergedAttributes });
            }, undefined, { children: true });

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(cell, mergedAttributes);
        }

        /**
         * Will be called for every table cell for table preview generation.
         * Formats the table cell according to the passed attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes need to be formatted, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         */
        function previewTableCellFormatting(cell, mergedAttributes) {
            updateCSSTableCellFormatting(cell, mergedAttributes, { preview: true });
        }

        /**
         * Will be called for every table cell for table preview generation.
         * Formats the table cell according to the passed attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes need to be formatted, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         *
         * @param {Object} borderOptions
         *  optional options used when calling this.getCssBorder()
         */
        function updateCSSTableCellFormatting(cell, mergedAttributes, borderOptions) {

            var // the target chain, that is required to resolve the correct theme
                target = null,
                // the color object for the cell fill color
                cellColor = mergedAttributes.cell.fillColor,
                // the different borders
                leftBorder = mergedAttributes.cell.borderLeft,
                rightBorder = mergedAttributes.cell.borderRight,
                topBorder = mergedAttributes.cell.borderTop,
                bottomBorder = mergedAttributes.cell.borderBottom;

            // the target chain for the slide must be set, if at least on color type is 'scheme'
            if (AttributeUtils.isColorThemed(cellColor) ||
                AttributeUtils.isLineThemed(leftBorder) ||
                AttributeUtils.isLineThemed(rightBorder) ||
                AttributeUtils.isLineThemed(topBorder) ||
                AttributeUtils.isLineThemed(bottomBorder)) {
                target = docModel.getTargetChainForNode(cell);
            }

            function getCssBorderColor(attrs) {
                return self.getCssBorder(attrs, borderOptions, (AttributeUtils.isLineThemed(attrs) ? target : null));
            }

            cell.css('background-color', self.getCssColor(cellColor, 'fill', (AttributeUtils.isColorThemed(cellColor) ? target : null)));

            cell.css('border-left', getCssBorderColor(mergedAttributes.cell.borderLeft));
            cell.css('border-right', getCssBorderColor(mergedAttributes.cell.borderRight));
            cell.css('border-top', getCssBorderColor(mergedAttributes.cell.borderTop));
            cell.css('border-bottom', getCssBorderColor(mergedAttributes.cell.borderBottom));
        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('cell', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class TableCellStyles

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: TableCellStyles });

});
