/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/paragraphstyles', [
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/drawinglayer/view/drawingframe'
], function (PresentationUtils, Utils, AttributeUtils, Color, Border, LineHeight, StyleCollection, DOM, DrawingFrame) {

    'use strict';

    var // definitions for paragraph attributes
        DEFINITIONS = {

            alignment: {
                def: 'left',
                format: function (element, value) {
                    var drawingNode = element.closest(DrawingFrame.NODE_SELECTOR);
                    var alignSelfVal = '';

                    // #49446
                    if (DrawingFrame.isNoWordWrapDrawingFrame(drawingNode) && !DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                        switch (value) {
                            case 'center':
                                alignSelfVal = value;
                                break;
                            case 'left':
                                alignSelfVal = 'flex-start';
                                break;
                            case 'right':
                                alignSelfVal = 'flex-end';
                                break;
                            case 'justify':
                                alignSelfVal = 'stretch';
                                break;
                        }
                    }
                    element.css({ textAlign: value, alignSelf: alignSelfVal });
                },
                preview: true // 'true' calls the own 'format' method
            },

            fillColor: {
                def: Color.AUTO, // auto for paragraph fill resolves to 'transparent'
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            /**
             * Presentation specific level attribute
             */
            level: { def: 0 },

            /**
             * Line height relative to font settings. The CSS attribute
             * 'line-height' must be set separately at every descendant text
             * span because a relative line height (e.g. 200%) would not be
             * derived from the paragraph relatively to the spans, but
             * absolutely according to the paragraph's font size. Example: The
             * paragraph has a font size of 12pt and a line-height of 200%,
             * resulting in 24pt. This value will be derived absolutely to a
             * span with a font size of 6pt, resulting in a relative line
             * height of 24pt/6pt = 400% instead of the expected 200%.
             */
            lineHeight: { def: LineHeight._115 },
            // workaround for Bug 47960 & Bug 45951, lineHeight in powerpoint is based on "baseline" instead of "fontsize"

            listLabelHidden: { def: false },

            listStartValue: { def: -1 },

            outlineLevel: { def: 9 },

            tabStops: {
                def: [],
                merge: function (tabStops1, tabStops2) {
                    // Merge tabStops2 into array tabstop1
                    // Support to clear tab stops defined in tabStops1
                    var clearedTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value === 'clear';
                        }),
                        additionalTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value !== 'clear';
                        }),
                        newTabstops = tabStops1 ? (additionalTabstops ? tabStops1.concat(additionalTabstops) : tabStops1) : (additionalTabstops ? additionalTabstops : []);

                    // Make entries in newTabstops unique
                    newTabstops = _.unique(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });

                    // Filter out cleared tabStops
                    if (clearedTabstops.length > 0) {
                        newTabstops = _.filter(newTabstops, function (tabstop) {
                            // return only tab stops which are not part of the clearedTabstops array
                            return (_.find(clearedTabstops, function (cleared) {
                                return cleared.pos === tabstop.pos;
                            }) === undefined);
                        });
                    }

                    // Sort tabStops by position
                    return _.sortBy(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });
                }
            },

            borderLeft: { def: Border.NONE },

            borderRight: { def: Border.NONE },

            borderTop: { def: Border.NONE },

            borderBottom: { def: Border.NONE },

            borderInside: { def: Border.NONE },

            indentFirstLine: { def: 0 },

            indentLeft: { def: 0 },

            indentRight: { def: 0 },

            marginTop: { def: 0 },

            marginBottom: { def: 0 },

            contextualSpacing: { def: false },

            bullet: { def: {} },

            bulletFont: { def: { followText: true } },

            bulletColor: { def: { followText: true } },

            bulletSize: { def: { type: 'followText', size: 0 } },

            defaultTabSize: { def: 2540 }, // default in presentation app: one inch (47964)

            spacingBefore: { def: 0 },

            spacingAfter: { def: 0 }
        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            cell: function (paragraph) { return DOM.isCellContentNode(paragraph.parent()) ? paragraph.closest('td') : null; }
        },

        // maps fill character attribute values to fill characters
        TAB_FILL_CHARS = { dot: '.', hyphen: '-', underscore: '_' },

        NBSP = '\xa0';

    // private static functions ===============================================

    function isMergeBorders(attributes1, attributes2) {
        return (Border.isVisibleBorder(attributes1.borderLeft) ||
            Border.isVisibleBorder(attributes1.borderRight) ||
            Border.isVisibleBorder(attributes1.borderTop) ||
            Border.isVisibleBorder(attributes1.borderBottom) ||
            Border.isVisibleBorder(attributes1.borderInside)) &&
            Utils.hasEqualProperties(attributes1, attributes2, [
                'borderLeft', 'borderRight', 'borderTop', 'borderBottom', 'borderInside',
                'listStyleId', 'listLevel', 'indentLeft', 'indentRight', 'indentFirstLine'
            ]);
    }

    /**
     * Fills the passed text span with a sufficient number of the specified
     * fill character.
     *
     * @param {jQuery} spanNode
     *  The span node to be filled. The current CSS formatting of this node is
     *  used to calculate the number of fill characters needed.
     *
     * @param {String} fillChar
     *  The fill character.
     *
     * @param {Number} width
     *  The target width of the tabulator, in 1/100 of millimeters.
     */
    function insertTabFillChar(spanNode, fillChar, width) {

        var // multiplier for a better average calculation
            multiplier = 15,
            // multiplier fill characters, used to calculate average character width
            checkString = Utils.repeatString(fillChar, multiplier),
            // average character width, in 1/100 mm
            charWidth = Utils.convertLengthToHmm(spanNode.contents().remove().end().text(checkString).width(), 'px') / multiplier,
            // number of characters needed to fill the specified width
            charCount = Math.floor(width / charWidth),
            // the fill character, repeated by the calculated number
            fillString = (charCount > 0) ? Utils.repeatString(fillChar, charCount) : NBSP,
            // a shortened string, if element is too wide
            shortFillString = null;

        // insert the fill string into the element
        spanNode.contents().remove().end().text(fillString);
        if (!fillString) { DOM.ensureExistingTextNode(spanNode); }

        // shorten fill string by one character, if element is too wide (e.g. due to rounding errors)
        if ((fillString.length > 1) && (Utils.convertLengthToHmm(spanNode.width(), 'px') >= width)) {
            shortFillString = fillString.slice(0, -1);
            spanNode.contents().remove().end().text(shortFillString);
            if (!shortFillString) { DOM.ensureExistingTextNode(spanNode); }
        }

        // workaround for iOS & Safari mobile - needs a text node directly within
        // the tab div to enable an external keyboard to jump through a tab chains
        // with empty spans
        if (_.device('ios') && _.device('safari')) {
            if (!spanNode[0].nextSibling) {
                spanNode.after(document.createTextNode(NBSP));
            }
        }
    }

    // class ParagraphStyles ==================================================

    /**
     * Contains the style sheets for paragraph formatting attributes. The CSS
     * formatting will be read from and written to paragraph <p> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function ParagraphStyles(docModel) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'paragraph', {
            families: 'character changes',
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateParagraphFormatting,
            layoutHandler: addLayoutAttributes
        });

        // private methods ----------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @attention
         *  Currently, only left&right aligned tabs are supported!
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateTabStops(paragraph, mergedAttributes) {

            var // default tab size width from paragraph settings
                defaultTabStop = mergedAttributes.paragraph.defaultTabSize,
                // paragraph tab stop definitions
                paraTabStops = mergedAttributes.paragraph.tabStops,
                // effective left margin of the paragraph
                marginLeft = Utils.convertCssLengthToHmm(paragraph.css('margin-left'), 'px'),
                // end position of previous tabulator
                prevTabEndPos = null,
                // first node of paragraph
                firstNode = null,
                // top position of first node
                topFirst = 0,
                // height of first node
                heightFirst = 0,
                // leftmost position used a base to calculate tab raster
                zeroLeft = 0,
                // first line  indent
                indentFirstLine = mergedAttributes.paragraph.indentFirstLine,
                // intersect function
                intersect = function (xs1, xe1, xs2, xe2) {
                    return ((xs1 <= xs2) && (xe1 >= xs2)) || ((xs2 <= xs1) && (xe2 >= xs1));
                },
                // active paragraph tab stops
                activeParaTabStops = null,
                // if paraTabStops has left and right oriented tabs, mixedTabs is set by the right-tab. if it is set it overwrites behavior of text-aligment and para-margin
                mixedTab = getMixedTab(paraTabStops);

            if (docModel.getApp().isInQuit()) {
                return;
            }

           // zoom factor in floating point notation
            var zoomFactor = docModel.getApp().getView().getZoomFactor() / 100;

            if (paragraph.length) {
                // Calculate top position and height of first paragraph child. Our paragraph
                // have always at least one child element.
                firstNode = $(paragraph.get(0).firstChild);
                topFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.position().top, 'px') : 0;
                heightFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.height(), 'px') : 0;

                //reset all tab sizes, for a much cleaner new calculating
                //(for example changing text-alignment from justify to left)
                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {
                    $(this).css('width', 0);
                });

                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {

                    var // the current tab node of the paragraph
                        tabNode = $(this),
                        // the span within the tab node containing the fill characters
                        tabSpan = $(this.firstChild),
                        // the position of the tab node
                        pos = tabNode.position(),
                        // the left bound position of the tab node in 1/100th mm
                        leftHMM = 0,
                        // the top bound position of the tab node in 1/100th mm
                        topHMM = Utils.convertLengthToHmm(pos.top, 'px'),
                        // the top bound position of the first child node of the paragraph in 1/100th mm
                        topSpanHMM = 0,
                        // the height of the first child node of the paragraph in 1/100th mm
                        heightSpanHMM = 0,
                        // the calculated width of the tab stop
                        width = 0,
                        // the fill character of the tab stop
                        fillChar = null,
                        // the tab stop values
                        tabStop,
                        // ignore the tab position and continue with the next one
                        ignore = false,
                        // tab stop type
                        tabStopType = 'left',
                        // indent first line tab stop position
                        indentFirstLineTabStopValue = 0,
                        // first line tab stop
                        indentFirstLineTabStop = null,
                        // insert index for virtual first line indent tab stop
                        indentFirstLineTabStopIndex = 0;

                    var calcLeftTab = function () {
                        return Math.max(0, tabStop.pos - (leftHMM % tabStop.pos));
                    };
                    var calcRightTab = function () {
                        return calcRightAlignTabstop(tabStop, tabNode, leftHMM);
                    };
                    var calcMiddleTab = function () {
                        return 0.5 * (calcLeftTab() + calcRightTab());
                    };

                    if (mergedAttributes.paragraph.alignment === 'center') {
                        // for a centered paragraph tab stops are calculated where the first character is treated as leftmost (0) position
                        zeroLeft = firstNode.length ? -(marginLeft + Utils.convertLengthToHmm(firstNode.position().left, 'px')) : -marginLeft;
                    } else {
                        zeroLeft = marginLeft;
                    }

                    // calculate left bound position of the tab node in 1/100th mm, including zoom factor for browsers using transf.scale
                    leftHMM = zeroLeft + Utils.convertLengthToHmm(pos.left / zoomFactor, 'px');

                    if (prevTabEndPos) {
                        // Check to see if we are in a chain of tabs. Force to use the previous tab end position
                        // as start position. Some browsers provide imprecise positions therefore we use the correct
                        // previous position from the last calculation.

                        // Check previous node and that text is empty - this is a precondition for the chaining
                        // code.
                        if (tabNode.prev().length && tabNode.prev().text().length === 0) {
                            var checkPrevPrevNode = tabNode.prev().length ? tabNode.prev().prev() : null;
                            if (checkPrevPrevNode.length && DOM.isTabNode(checkPrevPrevNode) && prevTabEndPos.top === topHMM) {
                                leftHMM = prevTabEndPos.center || prevTabEndPos.right;
                            }
                        }
                    }

                    // Special case for first line indent. For the first line the negative indent
                    // must be used as a tab stop position using the absolute value. We need to set
                    // at least one character for the tabSpan to have a valid width otherwise we
                    // won't get a correct height which is needed for the check.
                    if (indentFirstLine < 0) {
                        indentFirstLineTabStopValue = Math.abs(indentFirstLine);
                        // Fix for 29265 and 30847: Removing empty text node in span with '.contents().remove().end()'.
                        // Otherwise there are two text nodes in span after '.text('abc')' in IE.
                        tabSpan.contents().remove().end().text(NBSP);
                        heightSpanHMM = Utils.convertLengthToHmm(tabSpan.height(), 'px');
                        topSpanHMM = Utils.convertLengthToHmm(tabSpan.position().top, 'px');
                        // checkout if the first line indent is active for this special tab stop
                        if (intersect(topFirst, topFirst + heightFirst, topSpanHMM, topSpanHMM + heightSpanHMM) &&
                            (leftHMM + 10) < indentFirstLineTabStopValue) {
                            width = Math.max(0, indentFirstLineTabStopValue - leftHMM);
                        }

                        if (width > 1) {
                            // find index for the first line indent position within the paragraph tab stop array
                            indentFirstLineTabStopIndex = Utils.findLastIndex(paraTabStops, function (tab) {
                                return (indentFirstLineTabStopValue > tab.pos);
                            }) + 1;
                            // create a copy of the paragraph tab stops and add the first indent tab stop at
                            // the correct position to the active paragraph tab stop array
                            indentFirstLineTabStop = { value: 'left', pos: indentFirstLineTabStopValue, fillChar: NBSP, processed: false };
                            activeParaTabStops = paraTabStops.slice();
                            activeParaTabStops.splice(indentFirstLineTabStopIndex, 0, indentFirstLineTabStop);
                            // reset width - the tab stop position must be calculated using all tab stop positions
                            width = 0;
                        } else {
                            // reset text within span
                            tabSpan.contents().remove();
                            tabSpan[0].appendChild(document.createTextNode(''));
                            // use only paragraph tab stops
                            activeParaTabStops = paraTabStops;
                        }
                    } else {
                        // reset text within span
                        tabSpan.contents().remove();
                        tabSpan[0].appendChild(document.createTextNode(''));
                        // use only paragraph tab stops
                        activeParaTabStops = paraTabStops;
                    }

                    // Paragraph tab stops. Only paragraph tab stop can have a fill character and
                    // define a new alignment
                    if (width <= 1) {
                        tabStop = _(activeParaTabStops).find(function (tab) {
                            return ((leftHMM + 10) < tab.pos) && !tab.processed;
                        });
                        if (tabStop) {

                            tabStopType = tabStop.value || 'left';

                            // calculate tab stop size based on tab stop properties
                            if (tabStopType === 'left') {
                                // left bound tab stop
                                width = calcLeftTab();
                            } else if (tabStopType === 'right') {
                                // right bound tab stop
                                width = calcRightTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);

                                if (!ignore && _.indexOf(activeParaTabStops, tabStop) === activeParaTabStops.length - 1) {
                                    if (!_.browser.IE) {
                                        paragraph.css('width', 'calc(100% + 5mm)');
                                    }
                                }

                            } else if (tabStopType === 'center') {
                                // right bound tab stop combinded with left bound tab stop
                                width = calcMiddleTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);
                            }

                            // insert fill character
                            if (width > 1) {
                                fillChar = _.isString(tabStop.fillChar) ? TAB_FILL_CHARS[tabStop.fillChar] : NBSP;
                                if (fillChar) {
                                    insertTabFillChar(tabSpan, fillChar, width);
                                }
                                // Set processed flag to prevent using the same tab again due to rounding errors
                                // resulting by the browser calculation.
                                tabStop.processed = true;
                            }
                        }
                    }

                    if (!ignore) {
                        // only process default tab stop if tab stop is not set to ignore
                        if (width <= 1) {
                            // tab size calculation based on default tab stop
                            width = Math.max(0, defaultTabStop - (leftHMM % defaultTabStop));
                            width = (width <= 10) ? defaultTabStop : width; // no 0 tab size allowed, check for <= 10 to prevent rounding errors
                            // reset possible fill character
                            insertTabFillChar(tabSpan, NBSP, width);
                        }
                        prevTabEndPos = { right: leftHMM + width, top: topHMM };
                    }
                    // always set a width to the tab div (even an ignored tab div needs to set the size to zero)
                    tabNode.css('width', (width / 100) + 'mm');
                });

                calculateMarginForMixedTab(docModel, paragraph, mergedAttributes, mixedTab);

                // reset processed flag for tab stops again
                paraTabStops.forEach(function (tab) {
                    tab.processed = false;
                });
            }
        }

        /**
         * right tab can lay right above from the border, this cannot be triggered by MS-WORD interface,
         * but it is a typical behavior for MS-"table of content"
         * so we calculate if right tab is outside of the margin, and if true,
         * we reduce the margin-right of the paragraph
         */
        function calculateMarginForMixedTab(docModel, paragraph, paraAttributes, mixedTab) {
            paragraph.css('margin-right', null);

            if (!mixedTab) {
                return;
            }

            var pageAttrs = docModel.getDefaultAttributes('page');
            var maxParaWidth = pageAttrs.width - pageAttrs.marginLeft - pageAttrs.marginRight;
            var indentRight = paraAttributes.paragraph.indentRight;
            var currentParaWidth = mixedTab.pos + indentRight;

            if (currentParaWidth > maxParaWidth) {
                var newIndentRight = indentRight - (currentParaWidth - maxParaWidth);
                paragraph.css('margin-right', (newIndentRight / 100) + 'mm');
            }
        }

        function getMixedTab(paraTabStops) {
            var leftTab = _.find(paraTabStops, function (tab) {
                return !tab.value || tab.value === 'left';
            });
            var rightTab = _.find(paraTabStops, function (tab) {
                return tab.value === 'center' || tab.value === 'right';
            });
            if (leftTab && rightTab) {
                return rightTab;
            }
        }

        /**
         * Calculates the width of a right aligned tab element
         * depending on the position and the remaining size of
         * the following nodes.
         *
         * @param {Object} tabStop
         *  The right aligned tab stop.
         *
         * @param {jQuery} tabNode
         *  The right aligned tab node as jQuery.
         *
         * @param {Number} tabNodePosHMM
         *  The current position of the tab node in 1/100th mm.
         *
         * @returns {Number}
         *  The width of the right aligned tab stop. May be zero if the
         *  tab stop is not active.
         */
        function calcRightAlignTabstop(tabStop, tabNode, tabNodePosHMM) {
            var nextNodes = tabNode.nextAll(),
                node,
                rightPos = tabNodePosHMM,
                width = 0, i = 0, len = nextNodes.length;

            // Loop through all following nodes and sum up their width. Exceeding
            // the tab stop position or a new tab stops the iteration.
            for (i = 0; i < len && rightPos < tabStop.pos; i++) {
                node = $(nextNodes.get(i));
                if (DOM.isTabNode(node)) {
                    break;
                }
                rightPos += Utils.convertLengthToHmm(node.width(), 'px');
            }

            if (rightPos < tabStop.pos) {
                // Active right aligned tab stop. Subtract at least 1/10mm
                // rounding error regarding position values from the browser.
                width = tabStop.pos - rightPos - 10;
                width = Math.max(0, width);
            }

            return width;
        }

        /**
         * Will be called for every paragraph whose character attributes have
         * been changed.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateParagraphFormatting(paragraph, mergedAttributes) {

            var // the paragraph attributes of the passed attribute map
                paragraphAttributes = mergedAttributes.paragraph,
                // the character styles/formatter
                characterStyles = docModel.getStyleCollection('character'),

                leftMargin = 0,
                rightMargin = 0,

                leftMarginTmp = 0,
                rightMarginTmp = 0,

                prevParagraph = paragraph.prev(),
                prevAttributes = (prevParagraph.length > 0) ? self.getElementAttributes(prevParagraph) : { paragraph: {} },

                nextParagraph = paragraph.next(),
                nextAttributes = (nextParagraph.length > 0) ? self.getElementAttributes(nextParagraph) : { paragraph: {} },
                collectedAttrs =  {},
                prevCollectedAttrs =  {},
                textIndent = 0,
                // target for targetChain for themes
                target;
                // explicitParaAttributes = null;

            function setBorder(border, position, addSpace, para) {
                var space = addSpace + ((border && border.space) ? border.space : 0),
                    paddingPosition,
                    borderPosition;

                if (para) {
                    para.css('padding-' + position, Utils.convertHmmToCssLength(space, 'px', 1));
                    para.css('border-' + position, self.getCssBorder(border));

                } else {
                    position = position.charAt(0).toUpperCase() + position.slice(1);
                    paddingPosition = 'padding' + position;
                    borderPosition = 'border' + position;

                    collectedAttrs[paddingPosition] = Utils.convertHmmToCssLength(space, 'px', 1);
                    collectedAttrs[borderPosition] = self.getCssBorder(border);
                }

                return -space;
            }

            function calcSpacingHmm(lineHeight) {
                // fix for Bug 48442
                if (!lineHeight.value) { return 0; }

                var height = 0;
                var fontHeight = Utils.convertLength(mergedAttributes.character.fontSize, 'pt', 'px');

                if (lineHeight.type === 'fixed') {
                    height = Utils.convertHmmToLength(lineHeight.value, 'px');
                } else if (lineHeight.type === 'percent') {
                    height = fontHeight * lineHeight.value / 100;
                } else {
                    Utils.warn('ParagraphStyles.calcSpacingHmm(): invalid line height type', lineHeight.type);
                    return 0;
                }
                // workaround for bug 47960 & Bug 48476
                var fontCollection = docModel.getFontCollection();
                var options = { zoom: 10 };

                if (!target && docModel.useSlideMode() && AttributeUtils.isCharacterFontThemed(mergedAttributes.character)) { target = docModel.getTargetChainForNode(paragraph); }
                if (target) { options.theme = docModel.getThemeCollection().getTheme('', target); }

                var normalLineHeight = fontCollection.getNormalLineHeight(mergedAttributes.character, options) / 10;
                height -= (normalLineHeight - fontHeight);
                height = Math.max(0, height);
                return Utils.convertLengthToHmm(height, 'px');

            }

            // Always update character formatting of all child nodes which may
            // depend on paragraph settings, e.g. automatic text color which
            // depends on the paragraph fill color. Also visit all helper nodes
            // containing text spans, e.g. numbering labels.
            Utils.iterateDescendantNodes(paragraph, function (node) {

                // visiting the span inside a hard break node
                // -> this is necessary for change tracking attributes
                if (DOM.isHardBreakNode(node)) {
                    characterStyles.updateElementFormatting(node.firstChild);
                }
                if (DOM.isComplexFieldNode(node)) {
                    characterStyles.updateElementFormatting(node);
                }

                DOM.iterateTextSpans(node, function (span) {
                    characterStyles.updateElementFormatting(span, { baseAttributes: mergedAttributes });
                });
            }, undefined, { children: true });

            // update borders
            var leftPadding = paragraphAttributes.borderLeft && paragraphAttributes.borderLeft.space ? paragraphAttributes.borderLeft.space : 0;
            leftMarginTmp = setBorder(paragraphAttributes.borderLeft, 'left', 0);
            rightMarginTmp = setBorder(paragraphAttributes.borderRight, 'right', 0);

            // Fix for BUG #36298:
            // only add the border-margins (left/right) if the paragraph isn't in a textframe
            // or if they were positive. Otherwise the left/right border will be hidden.
            if (!DOM.isNodeInsideTextFrame(paragraph) || (leftMarginTmp > 0 && rightMarginTmp > 0)) {
                leftMargin += leftMarginTmp;
                rightMargin += rightMarginTmp;
            }

            var topMargin = paragraphAttributes.marginTop;
            var bottomMargin = paragraphAttributes.marginBottom;

            // inside comments the paragraph bottom margin must be 0 (even if this was not applied via operation)
            // and the paragraph must be left aligned
            if (!docModel.getApp().isODF() && !docModel.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(paragraph)) {
                bottomMargin = 0;
                collectedAttrs.textAlign = 'left';
            }

            // top border is not set if previous paragraph uses the same border settings
            if (isMergeBorders(paragraphAttributes, prevAttributes.paragraph)) { // #36293
                setBorder({ style: 'none' }, 'top', topMargin);
                _.extend(prevCollectedAttrs, {
                    paddingBottom: self.getCssBorder(paragraphAttributes.borderInside.space + bottomMargin),
                    borderBottom: self.getCssBorder(paragraphAttributes.borderInside),
                    marginBottom: 0
                });
                topMargin = 0;
            } else {
                setBorder(paragraphAttributes.borderTop, 'top', 0);
            }

            // bottom border is replaced by inner border if next paragraph uses the same border settings
            if (isMergeBorders(paragraphAttributes, nextAttributes.paragraph)) { // #36293
                setBorder(paragraphAttributes.borderInside, 'bottom');
                setBorder(paragraphAttributes.borderInside, 'top', 0, nextParagraph);  // Task 30473: Updating merged borders
                prevCollectedAttrs.paddingBottom = self.getCssBorder(bottomMargin);
                bottomMargin = 0;
            } else {
                setBorder(paragraphAttributes.borderBottom, 'bottom', 0);
            }

            //calculate list indents
//            var listLabel = $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR);
//            var listStyleId = paragraphAttributes.listStyleId;
//            if (listStyleId.length) {
//                var listLevel = paragraphAttributes.listLevel,
//                    lists = docModel.getListCollection();
//                if (listLevel < 0) {
//                    // is a numbering level assigned to the current paragraph style?
//                    listLevel = lists.findIlvl(listStyleId, mergedAttributes.styleId);
//                }
//                if (listLevel !== -1 && listLevel < 10) {
//                    var listItemCounter = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
//                    var listObject = lists.formatNumber(paragraphAttributes.listStyleId, listLevel, listItemCounter, 1);
//
//                    if (listObject.indent > 0) {
//                        leftPadding += listObject.indent - listObject.firstLine;
//                        leftMargin += listObject.firstLine;
//                    }
//                    if (listLabel.length) {
//                        var listSpan = listLabel.children('span');
//                        if (listObject.fontSize) {
//                            listSpan.css('font-size', listObject.fontSize + 'pt');
//                        }
//                        if (listObject.color) {
//                            listSpan.css('color', self.getCssTextColor(listObject.color, [paragraphAttributes.fillColor, listObject.fillColor]));
//                        }
//                    }
//                } else {
//                    //fix for Bug 37594 some list-helper dont disappear
//                    listLabel.detach();
//                }
//            } else {
//                //fix for Bug 37594 some list-helper dont disappear
//                listLabel.detach();
//            }

            // paragraph margin attributes - also applying to paragraphs in a list, if they are defined as explicit attribute
            // -> handle both cases correctly: 40792 and 41118
//            if (listStyleId === '') {
            leftMargin += paragraphAttributes.indentLeft ? paragraphAttributes.indentLeft : 0;
            rightMargin += paragraphAttributes.indentRight ? paragraphAttributes.indentRight : 0;
            textIndent = paragraphAttributes.indentFirstLine ? paragraphAttributes.indentFirstLine : 0;
            collectedAttrs.textIndent = textIndent / 100 + 'mm';
//            } else {
//                // but explicit attributes need to be handled (40792)
//                explicitParaAttributes = AttributeUtils.getExplicitAttributes(paragraph);
//                explicitParaAttributes = (explicitParaAttributes && explicitParaAttributes.paragraph) || {};
//                if (explicitParaAttributes.indentLeft) { leftMargin += explicitParaAttributes.indentLeft; }
//                if (explicitParaAttributes.indentRight) { rightMargin += explicitParaAttributes.indentRight; }
//                if (explicitParaAttributes.indentFirstLine) {
//                    textIndent += explicitParaAttributes.indentRight;
//                    collectedAttrs.textIndent = textIndent / 100 + 'mm';
//                }
//            }

            if (textIndent < 0) {
                leftPadding -= textIndent;
                leftMargin += textIndent;
            }

            _.extend(collectedAttrs, {
                paddingLeft: (leftPadding / 100) + 'mm',
                // now set left/right margins
                marginLeft: (leftMargin / 100) + 'mm',
                marginRight: (rightMargin / 100) + 'mm',
                textIndent: (textIndent / 100) + 'mm'
            });

            // Overwrite of margin left for lists in draft mode (conversion from fixed unit mm to %)
            if (Utils.SMALL_DEVICE && paragraph.data('draftRatio')) {
                collectedAttrs.marginLeft = (parseInt($(paragraph).css('margin-left'), 10) * paragraph.data('draftRatio')) + '%';
            }

            //no distance between paragraph using the same style if contextualSpacing is set
            var noDistanceToPrev = prevAttributes.paragraph.contextualSpacing && (mergedAttributes.styleId === prevAttributes.styleId),
                noDistanceToNext = paragraphAttributes.contextualSpacing && (mergedAttributes.styleId === nextAttributes.styleId);
            if (noDistanceToPrev) {
                //remove bottom margin from previous paragraph
                prevCollectedAttrs.marginBottom = 0 + 'mm';
                collectedAttrs.paddingTop = 0 + 'mm';
                topMargin = 0;
            }
            if (noDistanceToNext) {
                collectedAttrs.paddingBottom = 0 + 'mm';
                bottomMargin = 0;
            }

            if (paragraphAttributes.spacingBefore) {
                topMargin += calcSpacingHmm(paragraphAttributes.spacingBefore);
            }
            if (paragraphAttributes.spacingAfter && nextParagraph.length) {
                bottomMargin += calcSpacingHmm(paragraphAttributes.spacingAfter);
            }

            _.extend(collectedAttrs, {
                marginTop: (topMargin / 100) + 'mm',
                marginBottom: (bottomMargin / 100) + 'mm'
            });

            // taking care of implicit paragraph nodes behind tables after loading the document
            // (also handling final empty paragraphs behind tables inside table cell (36537)).
            if (DOM.isIncreasableParagraph(paragraph) && (!self.isImportFinished() || !docModel.selectionInNode(paragraph))) {
                collectedAttrs.height = 0;
            }

            if (!_.browser.IE) {
                collectedAttrs.width = null;
            }

            // apply collected attributes at the end
            prevParagraph.css(prevCollectedAttrs);
            paragraph.css(collectedAttrs);

            // update the size of all tab stops in this paragraph (but only if the paragraph contains tabs (Performance))
            if (paragraph.find(DOM.TAB_NODE_SELECTOR).length > 0) { updateTabStops(paragraph, mergedAttributes); }

        }

        /**
         * Getting the merged place holder attributes from master slide and layout slide.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {String} placeHoderType
         *  The drawing place holder type.
         *
         * @param {Number} placeHoderIndex
         *  The drawing place holder index.
         *
         * @param {String} paraLevel
         *  The paragraph level.
         *
         * @returns {Object}
         *  An object containing the attributes from the layout slide and the master slide. If attributes are defined
         *  at both slide, the layout slide overwrites the values from the master slide.
         */
        function getMergedPlaceHolderAttributes(layoutId, masterId, placeHolderType, placeHolderIndex, paraLevel) {

            var // the drawing styles object
                drawingStyles = docModel.getDrawingStyles(),
                // the layout drawing attributes
                layoutDrawingAttrs = layoutId ? drawingStyles.getPlaceHolderAttributes(layoutId, placeHolderType, placeHolderIndex) : null,
                // the layout slide attributes
                layoutSlideAttrs = layoutId ? docModel.getListStylesAttributesOfSlide(layoutId, placeHolderType) : null,
                // the master drawing attributes
                masterDrawingAttrs = masterId ? drawingStyles.getPlaceHolderAttributes(masterId, placeHolderType, placeHolderIndex) : null,
                // the master slide attributes
                masterSlideAttrs = masterId ? docModel.getListStylesAttributesOfSlide(masterId, placeHolderType) : null,
                // the attributes of layout and master slide
                attributes = null;

            // using only the 'listStyle' property from the drawing attribute set
            layoutDrawingAttrs = layoutDrawingAttrs && layoutDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] ? layoutDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] : null;
            masterDrawingAttrs = masterDrawingAttrs && masterDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] ? masterDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] : null;

            // using only selected list level from drawing 'listStyle' attribute
            layoutDrawingAttrs = layoutDrawingAttrs && layoutDrawingAttrs[paraLevel] ? layoutDrawingAttrs[paraLevel] : null;
            masterDrawingAttrs = masterDrawingAttrs && masterDrawingAttrs[paraLevel] ? masterDrawingAttrs[paraLevel] : null;

            // using only selected list level from slide attributes
            layoutSlideAttrs = layoutSlideAttrs && layoutSlideAttrs[paraLevel] ? layoutSlideAttrs[paraLevel] : null;
            masterSlideAttrs = masterSlideAttrs && masterSlideAttrs[paraLevel] ? masterSlideAttrs[paraLevel] : null;

            // merging the collected attributes from master and layout slides and drawings (never overwrite attribute set)
            if (masterSlideAttrs) { attributes = docModel.extendAttributes({}, masterSlideAttrs); }
            if (masterDrawingAttrs) { attributes = attributes ? docModel.extendAttributes(attributes, masterDrawingAttrs) : docModel.extendAttributes({}, masterDrawingAttrs); }
            if (layoutSlideAttrs) { attributes = attributes ? docModel.extendAttributes(attributes, layoutSlideAttrs) : docModel.extendAttributes({}, layoutSlideAttrs); }
            if (layoutDrawingAttrs) { attributes = attributes ? docModel.extendAttributes(attributes, layoutDrawingAttrs) : docModel.extendAttributes({}, layoutDrawingAttrs); }

            return attributes;
        }

        /**
         * Getting the merged place holder attributes for non place holder drawings.
         *
         * @param {jQuery|Node} drawing
         *  The drawing containing the paragraph with the specified level.
         *
         * @param {String} paraLevel
         *  The paragraph level.
         *
         * @returns {Object}
         *  An object containing the merged attributes for a specified paragraph level. Merged are the
         *  default list attributes and the list attributes that are assigned to the drawing node.
         */
        function getMergedNonPlaceHolderAttributes(drawing, paraLevel) {

            var // the default attributes for a specified level
                defaultAttrs = docModel.getDefaultTextListStylesForLevel(paraLevel),
                // the hard list style attributes at the drawing (45279)
                explicitDrawingAttrs = docModel.getDrawingStyles().getDrawingListStyleAttributes(drawing),
                // the explicit character attributes at the drawing
                explicitCharAttributes = null;

            defaultAttrs = docModel.extendAttributes({}, defaultAttrs);

            if (explicitDrawingAttrs && explicitDrawingAttrs[paraLevel]) {
                defaultAttrs = docModel.extendAttributes(defaultAttrs, explicitDrawingAttrs[paraLevel]);
            }

            // also checking explicit character attributes at the drawing
            explicitCharAttributes = AttributeUtils.getExplicitAttributes(drawing, { family: 'character' });

            if (explicitCharAttributes && !_.isEmpty(explicitCharAttributes)) {
                defaultAttrs = docModel.extendAttributes(defaultAttrs, { character: explicitCharAttributes });
            }

            return defaultAttrs;
        }

        /**
         * Getting all available list level attributes specified for a given set of layout slide id, master slide
         * id, place holder type and place holder index.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {String} placeHoderType
         *  The drawing place holder type.
         *
         * @param {Number} placeHoderIndex
         *  The drawing place holder index.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [nameOnly=false]
         *      If set to true, only the names of the list styles are returned, not the complete definitions.
         *
         * @returns {Object|Null}
         *  An object containing the list level attributes that are available inside a drawing, that is specified
         *  by layout id, master id, place holder type and place holder index. If no list styles are defined for
         *  this drawing, null is returned.
         *  The returned object has the top level keys 'l1', 'l2', ...
         */
        function getAllAvailableListStyleAttributes(layoutId, masterId, placeHolderType, placeHolderIndex, options) {

            var // the drawing styles object
                drawingStyles = docModel.getDrawingStyles(),
                // the layout drawing attributes
                layoutDrawingAttrs = layoutId ? drawingStyles.getPlaceHolderAttributes(layoutId, placeHolderType, placeHolderIndex) : null,
                // the layout slide attributes
                layoutSlideAttrs = layoutId ? docModel.getListStylesAttributesOfSlide(layoutId, placeHolderType) : null,
                // the master drawing attributes
                masterDrawingAttrs = masterId ? drawingStyles.getPlaceHolderAttributes(masterId, placeHolderType, placeHolderIndex) : null,
                // the master slide attributes
                masterSlideAttrs = masterId ? docModel.getListStylesAttributesOfSlide(masterId, placeHolderType) : null,
                // the attributes of layout or master slide
                attributes = null,
                // whether only the names shall be returned, not the definition of the list level styles
                nameOnly = Utils.getBooleanOption(options, 'nameOnly', false),
                // a collector for the list level keys
                allKeys = null;

            // using only the 'listStyle' property from the drawing attribute set
            layoutDrawingAttrs = layoutDrawingAttrs && layoutDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] ? layoutDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] : null;
            masterDrawingAttrs = masterDrawingAttrs && masterDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] ? masterDrawingAttrs[ParagraphStyles.LISTSTYLE_NAME] : null;

            if (nameOnly) {

                allKeys = [];

                if (layoutDrawingAttrs) { allKeys = allKeys.concat(_.keys(layoutDrawingAttrs)); }
                if (layoutSlideAttrs) { allKeys = allKeys.concat(_.keys(layoutSlideAttrs)); }
                if (masterDrawingAttrs) { allKeys = allKeys.concat(_.keys(masterDrawingAttrs)); }
                if (masterSlideAttrs) { allKeys = allKeys.concat(_.keys(masterSlideAttrs)); }

                if (allKeys.length > 0) {
                    attributes = {};
                    _.each(allKeys, function (key) {
                        attributes[key] = 1;
                    });
                }

            } else {

                // Merging all attributes for all list level -> this should be used with care because of performance reasons.
                // -> merging the collected attributes from master and layout slides and drawings (never overwrite attribute set)
                if (masterSlideAttrs) {
                    _.each(_.keys(masterSlideAttrs), function (key) {
                        attributes = attributes || {};
                        if (!attributes[key]) { attributes[key] = {}; }
                        attributes[key] = docModel.extendAttributes(attributes[key], masterSlideAttrs[key]);
                    });
                }

                if (masterDrawingAttrs) {
                    _.each(_.keys(masterDrawingAttrs), function (key) {
                        attributes = attributes || {};
                        if (!attributes[key]) { attributes[key] = {}; }
                        attributes[key] = docModel.extendAttributes(attributes[key], masterDrawingAttrs[key]);
                    });
                }

                if (layoutSlideAttrs) {
                    _.each(_.keys(layoutSlideAttrs), function (key) {
                        attributes = attributes || {};
                        if (!attributes[key]) { attributes[key] = {}; }
                        attributes[key] = docModel.extendAttributes(attributes[key], layoutSlideAttrs[key]);
                    });
                }

                if (layoutDrawingAttrs) {
                    _.each(_.keys(layoutDrawingAttrs), function (key) {
                        attributes = attributes || {};
                        if (!attributes[key]) { attributes[key] = {}; }
                        attributes[key] = docModel.extendAttributes(attributes[key], layoutDrawingAttrs[key]);
                    });
                }
            }

            return attributes;
        }

        /**
         * A helper function to set the valid paragraph and character attributes to paragraphs
         * inside tables.
         *
         * @param {Object} attributes
         *  The attribute set specified for the given paragraph. For paragraphs inside tables this
         *  already includes the table specific attributes. This makes it difficult in this case,
         *  because the table specific attributes must not be overwritten in this step.
         *
         * @param {Object} placeHolderAttributes
         *  The collection of attributes that are collected in the function 'addLayoutAttributes'.
         *  For paragraphs inside tables this are the 'other' attributes specified on the master
         *  slide for place holder drawings.
         */
        function handleParagraphInTableAttributes(attributes, placeHolderAttributes) {

            // Some attributes are inherited from the table style and must not be overwritten, for
            // example font color, that is dependent from the position inside the table.
            // But other attributes are not inherited and also the default value should not be used.
            // This is the case for the font size, where the default of 11pt is too small. But the
            // default font color must not overwrite the value specified by the table style.
            // Unfortunately it is not clear at this point, which attributes are inherited from the
            // table and which are the default values.
            //
            // -> Removing those values from the place holder attributes, for that the given attributes
            //    are NOT the default values

            var allFamilies = null;
            var allDefaultAttributes = null;

            if (placeHolderAttributes && attributes) {

                allFamilies = _.keys(placeHolderAttributes);
                allDefaultAttributes = docModel.getDefaultAttributeSet(allFamilies);

                _.each(allFamilies, function (family) {

                    var placeHolderAttrsFamily = placeHolderAttributes[family];

                    if (attributes[family]) {
                        // iterating over all place holder attributes and check if the given attribute set uses the default value
                        _.each(_.keys(placeHolderAttrsFamily), function (attr) {
                            if (!_.isEqual(attributes[family][attr], allDefaultAttributes[family][attr])) {
                                delete placeHolderAttrsFamily[attr]; // -> not overwriting specified non-default value
                            }
                        });
                    }
                });
            }
        }

        /**
         * Adding attributes of master and layout slide into the merge attributes. This has to happen
         * before the explicit attributes are set into the specified attribute set.
         *
         * @param {jQuery} element
         *  The node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} attributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets, but without explicit attributes.
         */
        function addLayoutAttributes(paragraph, attributes) {

            var // the attribute set of the place holder elements in layout and/or master slide
                placeHolderAttributes = null,
                // the IDs of document slide, layout slide and master slide
                id = null, layoutId = null, masterId = null,
                // the place holder type and index
                placeHolderType = null, placeHolderIndex = null,
                // the closest drawing node upwards in the dom
                drawing = paragraph.closest('div.drawing'), // TODO: Group handling
                // the explicit drawing attributes
                drawingAttrs = null,
                // the paragraph level attribute
                paraLevel = 0,
                // whether the paragraph is inside a table (attributes can be defined at table, never overwrite)
                insideTable = DOM.isCellContentNode(paragraph.parent());

            // check, if the drawing, containing the paragraph, is a 'placeholder' drawing
            if (drawing.length > 0) { drawingAttrs = AttributeUtils.getExplicitAttributes(drawing, { direct: true }); }

            paraLevel = self.getParagraphLevel(paragraph);
            paraLevel = self.generateKeyFromLevel(paraLevel);

            if (PresentationUtils.isPlaceHolderAttributeSet(drawingAttrs) && !insideTable) { // place holder, but not in table

                id = DOM.getTargetContainerId(drawing.parent()); // checking only direct children of slide
                placeHolderType = drawingAttrs.presentation.phType;
                placeHolderIndex = drawingAttrs.presentation.phIndex;

                if (placeHolderType && !placeHolderIndex) { placeHolderIndex = PresentationUtils.getDefaultPlaceHolderIndex(); }
                if (placeHolderIndex && !placeHolderType) { placeHolderType = PresentationUtils.getDefaultPlaceHolderType(); }

                if (docModel.isStandardSlideId(id)) {
                    layoutId = docModel.getLayoutSlideId(id);
                    masterId = docModel.getMasterSlideId(layoutId);
                    placeHolderAttributes = getMergedPlaceHolderAttributes(layoutId, masterId, placeHolderType, placeHolderIndex, paraLevel);
                } else if (docModel.isLayoutSlideId(id)) {
                    masterId = docModel.getMasterSlideId(id);
                    placeHolderAttributes = getMergedPlaceHolderAttributes(id, masterId, placeHolderType, placeHolderIndex, paraLevel);
                } else if (docModel.isMasterSlideId(id)) {
                    placeHolderAttributes = getMergedPlaceHolderAttributes(null, id, placeHolderType, placeHolderIndex, paraLevel);
                }
            } else if (insideTable) {

                id = DOM.getTargetContainerId(drawing.parent()); // checking only direct children of slide
                masterId = docModel.getCorrespondingMasterSlideId(id);
                // using the 'other' attributes from master slide
                placeHolderAttributes = docModel.getListStylesAttributesOfSlide(masterId, docModel.getDefaultPlaceHolderListStyleType());

                if (placeHolderAttributes && placeHolderAttributes[paraLevel]) {
                    placeHolderAttributes = placeHolderAttributes[paraLevel];
                } else {
                    placeHolderAttributes = getMergedNonPlaceHolderAttributes(drawing, paraLevel); // fallback to non-placeholder default attributes
                }

                // merge handling for paragraphs inside tables
                handleParagraphInTableAttributes(attributes, placeHolderAttributes);

            } else {

                // using the default text list styles
                // -> these values will be used for shapes or text frames
                // -> additionally there can be list styles hard defined at the drawing (45279)
                placeHolderAttributes = getMergedNonPlaceHolderAttributes(drawing, paraLevel);
            }

            // merging the new place holder attributes into the specified attribute set
            if (placeHolderAttributes) { docModel.extendAttributes(attributes || {}, placeHolderAttributes); }
        }

        // methods ------------------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated. If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {ParagraphStyles}
         *  A reference to this instance.
         */
        this.updateTabStops = function (paragraph) {
            updateTabStops($(paragraph), self.getElementAttributes(paragraph));
            return this;
        };

        /**
         * Getting the level of a specified paragraph. If the paragraph has no
         * level specified in the attributes, 0 is returned.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph node whose level shall be determined.
         *
         * @returns {Number}
         *  The paragraph level. If it cannot be determined, 0 is returned.
         */
        this.getParagraphLevel = function (paragraph) {

            var // the explicit attributes at a paragraph
                paraAttrs = AttributeUtils.getExplicitAttributes(paragraph, { direct: true });

            return (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.level) || 0;
        };

        /**
         * Generating the list style key from a specified paragraph level.
         *
         * @param {Number} level
         *  The paragraph level.
         *
         * @returns {String}
         *  The key used in list styles corresponding to the specified paragraph level.
         */
        this.generateKeyFromLevel = function (level) {
            return _.isNumber(level) ? 'l' + (level + 1) : null;
        };

        /**
         * Getting all available list level attributes specified for a given drawing. This list styles can be
         * defined at the master or layout slide or at the drawing on the master or layout slide.
         * id, place holder type and place holder index.
         *
         * @param {jQuery} drawing
         *  The drawing, for that the list styles shall be determined.
         *
         * @param {Object} [options]
         *  Supports all options specified at the private function 'getAllAvailableListStyleAttributes'.
         *
         * @returns {Object|Null}
         *  An object containing the list level attributes that are available inside a specified drawing. If no
         *  list styles are defined for this drawing, null is returned.
         *  The returned object has the top level keys 'l1', 'l2', ...
         */
        this.getAllAvailableListStyleAttributes = function (drawing, options) {

            var // the ids of active slide, master slide and layout slide
                id = null, layoutId = null, masterId = null,
                // the place holder type and index
                placeHolderType = null, placeHolderIndex = null,
                // the explicit drawing attributes
                drawingAttrs = null,
                // the list style set
                allListStyles = null;

            // check, if the drawing, containing the paragraph, is a 'placeholder' drawing
            if (drawing && drawing.length > 0) { drawingAttrs = AttributeUtils.getExplicitAttributes(drawing, { direct: true }); }

            if (PresentationUtils.isPlaceHolderAttributeSet(drawingAttrs)) {

                id = DOM.getTargetContainerId(drawing.parent()); // checking only direct children of slide
                placeHolderType = drawingAttrs.presentation.phType;
                placeHolderIndex = drawingAttrs.presentation.phIndex;

                if (placeHolderType && !placeHolderIndex) { placeHolderIndex = PresentationUtils.getDefaultPlaceHolderIndex(); }
                if (placeHolderIndex && !placeHolderType) { placeHolderType = PresentationUtils.getDefaultPlaceHolderType(); }

                if (docModel.isStandardSlideId(id)) {
                    layoutId = docModel.getLayoutSlideId(id);
                    masterId = docModel.getMasterSlideId(layoutId);
                    allListStyles = getAllAvailableListStyleAttributes(layoutId, masterId, placeHolderType, placeHolderIndex, options);
                } else if (docModel.isLayoutSlideId(id)) {
                    masterId = docModel.getMasterSlideId(id);
                    allListStyles = getAllAvailableListStyleAttributes(id, masterId, placeHolderType, placeHolderIndex, options);
                } else if (docModel.isMasterSlideId(id)) {
                    allListStyles = getAllAvailableListStyleAttributes(null, id, placeHolderType, placeHolderIndex, options);
                }
            } else {

                // using the default text list styles, that are saved in the list handler mixin
                // -> these values will be used for shapes or text frames
                allListStyles = docModel.getDefaultTextListStyles();
            }

            return allListStyles;
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('paragraph', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class ParagraphStyles

    // constants --------------------------------------------------------------

    /**
     * Default solid border line for paragraphs and tables.
     * 'auto' is not supported in Presentation app.
     * The 'fallbackValue' is specified for ODF.
     *
     * @constant
     */
    ParagraphStyles.SINGLE_BORDER = { style: 'single', width: 17, space: 140, color: { type: 'scheme', value: 'text1', fallbackValue: '000000' } };

    /**
     * The name of the property in the drawing attributes, that contains the list style definitions.
     *
     * @constant
     */
    ParagraphStyles.LISTSTYLE_NAME = 'listStyle';

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: ParagraphStyles });

});
