/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/portal/spreadsheet/fulltour', [
    'io.ox/office/portal/portaltour',
    'io.ox/office/baseframework/utils/baseconfig',
    'gettext!io.ox/office/portal/main'
], function (PortalTour, BaseConfig, gt) {

    'use strict';

    // class SpreadsheetFullTour ==============================================

    /**
     * The guided tour for the Spreadsheet Portal application.
     *
     * @constructor
     *
     * @extends PortalTour
     */
    var SpreadsheetFullTour = PortalTour.extend({ constructor: function (initOptions) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        PortalTour.call(this, 'spreadsheet', initOptions);

        // private methods ----------------------------------------------------

        /**
         * Returns the localized version of the passed cell formula. Replaces
         * 'SUM' with the translated function name, and all commas with the
         * local function parameter separator.
         */
        function getLocalizedFormula(formula) {
            var formulaGrammar = self.getApp().getModel().getFormulaGrammar('ui');
            return formula
                .replace(/SUM/g, formulaGrammar.getFunctionName('SUM'))
                .replace(/,/g, formulaGrammar.SEP);
        }

        // initialization -----------------------------------------------------

        // the initial welcome message
        this.welcomeStep()
            .title(gt.pgettext('tour', 'The Spreadsheet app'))
            .content(gt.pgettext('tour', 'Welcome to your new Spreadsheet app. This Guided Tour will introduce you to the most important functions of creating and editing spreadsheets online.'));

        // do not show the following steps in the short version of the tour
        if (!this.isShortTour()) {

            // highlight the header of the template area
            this.templatesAreaStep()
                .title(gt.pgettext('tour', 'Templates'))
                .content(gt.pgettext('tour', 'Clicking on the Spreadsheet app in the navigation bar displays an overview of all the templates available to you in Spreadsheet.'));

            // describe the 'Blank document' button
            this.blankDocumentStep()
                .title(gt.pgettext('tour', 'Blank spreadsheet'))
                .content(gt.pgettext('tour', 'You can choose from a variety of templates to help you create professional documents.'))
                .content($('<br>'))
                .content(gt.pgettext('tour', 'Let\'s start with a blank document.'));
        }

        // launch the editor application with a new empty document
        this.launchDocumentStep();

        // do not show the following steps in the short version of the tour
        if (!this.isShortTour()) {

            // insert document contents, show the 'Saving changes' label
            this.autoSaveStep(function () {
                this.executeItem('cell/active/parse', 'Lorem ipsum dolor sit amet.');
            });

            // describe the 'File' toolbar
            this.fileTabStep();
            this.saveAsStep();
            if (BaseConfig.MAIL_AVAILABLE) { this.sendAsMailStep(); }
        }

        // start to type a cell formula
        this.step()
            .title(gt.pgettext('tour', 'Formulas'))
            .content(gt.pgettext('tour', 'You can use all the regular spreadsheet formulas by typing them into a cell.'))
            .on('before:show', function () {
                this.getDocView().cancelCellEditMode();
                this.executeItem('cell/active/parse', getLocalizedFormula('\'=SUM'));
            })
            .waitForAndReferTo(this.EDITOR_APP_PANE_SELECTOR + ' .selection-layer .active-cell', { position: 'bottom', hotspot: true });

        // show auto-completion drop-down menu for functions
        this.step()
            .title(gt.pgettext('tour', 'Autocompletion'))
            .content(gt.pgettext('tour', 'As soon as you start typing a formula, autocompletion offers you a list of matching functions to choose from.'))
            .on('before:show', function () {
                var docView = this.getDocView();
                docView.cancelCellEditMode();
                docView.enterCellEditMode({ text: getLocalizedFormula('=SUM'), stickyList: true });
            })
            .waitForAndReferToPopup({ position: 'bottom', hotspot: true });

        // describe the formula label in the status pane
        this.step()
            .title(gt.pgettext('tour', 'Displaying hidden formulas'))
            .content(gt.pgettext('tour', 'If a result is displayed in a cell, the calculating formula will be displayed in the lower right hand corner of the screen.'))
            .on('before:show', function () {
                // need to launch explicitly, in case this step is reached by using the Back button in a short tour
                self.launchEditorApp().done(function () {
                    this.getDocView().cancelCellEditMode();
                    this.executeItem('cell/active/parse', getLocalizedFormula('=SUM(1, 2, 3)'));
                }.bind(this));
            })
            .waitForAndReferToGroup(this.EDITOR_WINDOW_SELECTOR + ' .status-pane', 'view/formula', { position: 'top', hotspot: true });

        // describe the 'Close' button, switch to 'Documents' folder in Drive
        if (!this.isShortTour()) {
            this.closeButtonStep();
            this.documentsFolderStep();
        }

        // final tour step describing how to restart the tour, return to correct application with the Back button
        this.restartTourStep().on('back', (this.isShortTour() ? this.launchEditorApp : this.launchDriveApp).bind(this));

    } }); // class SpreadsheetFullTour

    // exports ================================================================

    return SpreadsheetFullTour;

});
