/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/editframework/utils/texture', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/tk/utils/deferredutils',
    'io.ox/office/tk/render/canvas'
], function (Utils, ImageUtil, DeferredUtils, Canvas) {

    'use strict';

    var isNumber = _.isNumber;

    // private static functions ===============================================

    function calculateOffset(rectAlignment, offsetX, offsetY, widthPx, heightPx, patternWidth, patternHeight, options) {
        var tempX = 0,
            tempY = 0,
            flipMode = Utils.getBooleanOption(options, 'flipMode', false);

        switch (rectAlignment) {
            case 'topLeft':
                tempX = 0;
                tempY = 0;
                break;
            case 'top':
                tempX = ((widthPx - patternWidth) / 2) % patternWidth;
                if (flipMode) {
                    tempX += patternWidth / 4;
                }
                break;
            case 'topRight':
                tempX = widthPx % patternWidth;
                if (flipMode) {
                    tempX += patternWidth / 2;
                }
                break;
            case 'left':
                tempY = ((heightPx - patternHeight) / 2) % patternHeight;
                if (flipMode) {
                    tempY += patternHeight / 4;
                }
                break;
            case 'center':
                tempX = ((widthPx - patternWidth) / 2) % patternWidth;
                tempY = ((heightPx - patternHeight) / 2) % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 4;
                    tempY += patternHeight / 4;
                }
                break;
            case 'right':
                tempX = widthPx % patternWidth;
                tempY = ((heightPx - patternHeight) / 2) % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 2;
                    tempY += patternHeight / 4;
                }
                break;
            case 'bottomLeft':
                tempY = heightPx % patternHeight;
                if (flipMode) {
                    tempY += patternHeight / 2;
                }
                break;
            case 'bottom':
                tempX = ((widthPx - patternWidth) / 2) % patternWidth;
                tempY = heightPx % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 4;
                    tempY += patternHeight / 2;
                }
                break;
            case 'bottomRight':
                tempX = widthPx % patternWidth;
                tempY = heightPx % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 2;
                    tempY += patternHeight / 2;
                }
                break;
            default:
                Utils.error('Texture.calculateOffset(): rectAlignment is unknown!');
        }
        offsetX %= patternWidth;
        offsetY %= patternHeight;
        // if the values are positive, we need to shift them top left for the 1 pattern dimensions
        // this way pattern will always cover whole canvas object, and there will be no gaps if offset is positive
        if (offsetX > 0) { offsetX -= patternWidth; }
        if (offsetY > 0) { offsetY -= patternHeight; }
        if (tempX > 0) { tempX -= patternWidth; }
        if (tempY > 0) { tempY -= patternHeight; }
        tempX += offsetX;
        tempY += offsetY;
        tempX %= patternWidth;
        tempY %= patternHeight;

        return { offsetX: tempX, offsetY: tempY };
    }

    function getStrechedCoords(stretchingObj, cWidth, cHeight) {
        var sLeft = stretchingObj.left || 0;
        var sRight = stretchingObj.right || 0;
        var sTop = stretchingObj.top || 0;
        var sBottom = stretchingObj.bottom || 0;

        var tLeft = cWidth * (sLeft / 100);
        var tWidth = cWidth * ((100 - sRight - sLeft) / 100);
        var tTop = cHeight * (sTop / 100);
        var tHeight = cHeight * ((100 - sBottom - sTop) / 100);

        return { left: Utils.round(tLeft, 1), top: Utils.round(tTop, 1), width: Utils.round(tWidth, 1), height: Utils.round(tHeight, 1) };
    }

    // static class Texture ===================================================

    /**
     * Provides canvas patterns representing an image as background, and
     * texture as a background, generated from the image.
     */
    var Texture = {};

    // public methods ---------------------------------------------------------

    /**
     * Creates and returns texture fill for given object, based on bitmap and its attributes.
     *
     * @param  {PresentationModel} docModel
     *         Presentation doc model
     * @param  {ContextWrapper} context
     *         Context wrapper context as provided by the class Canvas.
     * @param  {Object} fillAttrs
     *         Shape fill attributes
     * @param  {Number} widthPx
     *         Canvas width in pixels
     * @param  {Number} heightPx
     *          Canvas height in pixels
     * @param  {Number|null} rotationAngle
     *         If exist, angle of rotation in radians
     * @param  {Object} [options]
     *         Optional set of properties.
     *  @param {Boolean} [options.isSlide=false]
     *         If texture is applied to slide background.
     *  @param {Boolean} [options.noAlpha=false]
     *         If alpha should be ignored.
     *
     * @return {jQuery.Promise}
     *  A promise that will be resolved with the generated canvas pattern.
     */
    Texture.getTextureFill = function (docModel, context, fillAttrs, widthPx, heightPx, rotationAngle, options) {

        var imageUrl = ImageUtil.getFileUrl(docModel.getApp(), fillAttrs.bitmap.imageUrl);
        var promise = docModel.getApp().createImageNode(imageUrl);

        promise = promise.then(function (imgNode) {
            var isSlideBackground = Utils.getBooleanOption(options, 'isSlide', false);
            var ignoreAlpha = Utils.getBooleanOption(options, 'noAlpha', false);
            var tilingObj = fillAttrs.bitmap.tiling;
            var isTiling = !_.isEmpty(tilingObj);
            var stretchingObj = fillAttrs.bitmap.stretching;
            var isStretching = !_.isEmpty(stretchingObj);
            var isRepeat = isTiling ? 'repeat' : 'no-repeat';
            var isRotateWithShape = isSlideBackground ? true : fillAttrs.bitmap.rotateWithShape !== false;
            var stretchX = (isTiling && isNumber(tilingObj.stretchX)) ? tilingObj.stretchX / 100 : 1;
            var stretchY = (isTiling && isNumber(tilingObj.stretchY)) ? tilingObj.stretchY / 100 : 1;
            var offsetX = (isTiling && isNumber(tilingObj.offX)) ? Utils.convertHmmToLength(tilingObj.offX, 'px', 1) : 0;
            var offsetY = (isTiling && isNumber(tilingObj.offY)) ? Utils.convertHmmToLength(tilingObj.offY, 'px', 1) : 0;
            var nTransparency = isNumber(fillAttrs.bitmap.transparency) ? (1 - fillAttrs.bitmap.transparency) : 1;
            var rectAlignment = (isTiling && tilingObj.rectAlignment) || null;
            var flipMode = (isTiling && tilingObj.flipMode) || null;
            var isFlipMode = flipMode && flipMode !== 'none';
            var dpiDiff = 96 / 96; // TODO: some images have different dpi, filter needs to send them
            var cWidth = isTiling ? Utils.round(imgNode[0].width * stretchX * dpiDiff, 1) : widthPx;
            var cHeight = isTiling ? Utils.round(imgNode[0].height * stretchY * dpiDiff, 1) : heightPx;
            var tempCanvas = new Canvas();
            var offsetCanvas = null;

            if (!isTiling && !isRotateWithShape) {
                cWidth = widthPx * Math.abs(Math.cos(rotationAngle)) + heightPx * Math.abs(Math.sin(rotationAngle));
                cHeight = heightPx * Math.abs(Math.cos(rotationAngle)) + widthPx * Math.abs(Math.sin(rotationAngle));
            }

            tempCanvas.initialize({ width: cWidth, height: cHeight });

            // $('body').append(tempCanvas.getNode().css({ zIndex: 99, position: 'fixed' })); // DEBUG

            // scaling of the image
            tempCanvas.render(function (tempCtx) {
                var coordObj = isStretching ? getStrechedCoords(stretchingObj, cWidth, cHeight) : { left: 0, top: 0, width: cWidth, height: cHeight };
                if (!ignoreAlpha) {
                    tempCtx.setGlobalAlpha(nTransparency);
                }
                tempCtx.drawImage(imgNode[0], coordObj);
            });
            // offset
            if (offsetX || offsetY || rectAlignment || flipMode) {
                offsetCanvas = new Canvas();
                offsetCanvas.initialize({ width: 2 * cWidth, height: 2 * cHeight });
                // $('body').append(offsetCanvas.getNode().css({ zIndex: 99, position: 'fixed', top: (cHeight + 20) })); // DEBUG

                if (isFlipMode) {
                    switch (flipMode) {
                        case 'x':
                            offsetCanvas.render(function (offsetCtx) {
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: cHeight, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(-1, 1);
                                offsetCtx.translate(-cWidth * 2, 0);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: cHeight, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                            });
                            break;
                        case 'y':
                            offsetCanvas.render(function (offsetCtx) {
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: cWidth, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(1, -1);
                                offsetCtx.translate(0, -cHeight * 2);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: cWidth, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                            });
                            break;
                        case 'xy':
                            offsetCanvas.render(function (offsetCtx) {
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(-1, 1);
                                offsetCtx.translate(-cWidth * 2, 0);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(1, -1);
                                offsetCtx.translate(0, -cHeight * 2);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(-1, 1);
                                offsetCtx.translate(-cWidth * 2, 0);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                            });
                            break;
                        default:
                            Utils.error('Texture.getTextureFill(): flipMode is unknown!');
                    }

                    offsetCanvas.render(function (offsetCtx) {
                        var calcOffset = calculateOffset(rectAlignment, offsetX, offsetY, widthPx, heightPx, cWidth * 2, cHeight * 2, { flipMode: true });
                        var pattern = offsetCtx.createPattern(offsetCanvas, 'repeat');
                        offsetCtx.clearRect(0, 0, 2 * cWidth, 2 * cHeight);
                        offsetCtx.translate(calcOffset.offsetX, calcOffset.offsetY);
                        offsetCtx.setFillStyle(pattern).drawRect(0, 0, 2 * cWidth - calcOffset.offsetX, 2 * cHeight - calcOffset.offsetY, 'fill');
                    });
                } else {
                    tempCanvas.render(function (tempCtx) {
                        var calcOffset = calculateOffset(rectAlignment, offsetX, offsetY, widthPx, heightPx, cWidth, cHeight);
                        var pattern = tempCtx.createPattern(tempCanvas, 'repeat');
                        tempCtx.clearRect(0, 0, cWidth, cHeight);
                        tempCtx.translate(calcOffset.offsetX, calcOffset.offsetY);
                        tempCtx.setFillStyle(pattern).drawRect(0, 0, cWidth - calcOffset.offsetX, cHeight - calcOffset.offsetY, 'fill');
                    });
                }
            }
            // DEBUG
            // tempCanvas.getNode().remove();
            // if (offsetCanvas) {
            //     offsetCanvas.getNode().remove();
            // }

            return context.createPattern(isFlipMode ? offsetCanvas : tempCanvas, isRepeat);
        });

        promise.fail(function () {
            Utils.warn('Texture.getTextureFill(): failed to load shape background image!');
        });

        return promise;
    };

    // exports ================================================================

    return Texture;
});
