/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/editframework/utils/operationutils', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // static class OperationUtils ============================================

    var OperationUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Predefined property name of the operation state number.
     *
     * @constant
     */
    OperationUtils.OSN = 'osn';

    /**
     * Predefined property name of the operation length.
     *
     * @constant
     */
    OperationUtils.OPL = 'opl';

    /**
     * Predefined property name of actions array.
     *
     * @constant
     */
    OperationUtils.ACTIONS = 'actions';

    /**
     * Predefined property name of the operations array within an action object
     *
     * @constant
     */
    OperationUtils.OPERATIONS = 'operations';

    // static methods ---------------------------------------------------------

    /**
     * Retrieves the operation state number from an operation object.
     *
     * @param {Object} operation
     *  A JSON operation object with properties.
     *
     * @returns {Number}
     *  The operation state number or -1, if it cannot be retrieved.
     */
    OperationUtils.getOSN = function (operation) {
        return Utils.getIntegerOption(operation, OperationUtils.OSN, -1);
    };

    /**
     * Retrieves the operation length from an operation object.
     *
     * @param {Object} operation
     *  A JSON operation object with properties.
     *
     * @returns {Number}
     *  The operation length or -1, if it cannot be retrieved.
     */
    OperationUtils.getOPL = function (operation) {
        return Utils.getIntegerOption(operation, OperationUtils.OPL, -1);
    };

    /**
     * Retrieves the actions array from an actions object or the actions array
     * itself.
     *
     * @param {Array<Object>|Object} actions
     *
     * @returns {Array<Object>}
     */
    OperationUtils.getActionsArray = function (actions) {
        return (_.isObject(actions) && _.isArray(actions[OperationUtils.ACTIONS])) ?
               actions[OperationUtils.ACTIONS] : actions;
    };

    /**
     * Returns the next operation state number which must follow
     * on the operations stored in the actions array.
     *
     * @param {Array<Object>} opsArray
     *  The array with JSON operation objects.
     *
     * @returns {Number}
     *  The next operation state number or -1 if the next number
     *  cannot be determined (e.g. empty array).
     */
    OperationUtils.getNextOperationStateNumber = function (opsArray) {
        var // calculated osn
            osn = -1,
            // the operation length
            opl = -1,
            // last operation
            operation = null;

        if (_.isArray(opsArray)) {
            operation = _.last(opsArray);
            osn = OperationUtils.getOSN(operation);
            opl = OperationUtils.getOPL(operation);
            osn = ((osn !== -1) && (opl !== -1)) ? osn + opl : -1;
        }

        return osn;
    };

    /**
     * Returns the starting operation state number from an operations
     * array.
     *
     * @params {Array<Object>} opsArray
     *  An array with JSON operation objects.
     *
     * @returns {Number}
     *  The operation state number of the first operation in the array, or -1
     *  if the operation state number cannot be determined (e.g. empty array).
     */
    OperationUtils.getStartingOperationStateNumber = function (opsArray) {
        return (_.isArray(opsArray)) ? OperationUtils.getOSN(_.first(opsArray)) : -1;
    };

    // exports ================================================================

    return OperationUtils;

});
