/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/operationsgenerator', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/utils/clienterror',
    'io.ox/office/baseframework/utils/errorcontext',
    'io.ox/office/editframework/utils/operations'
], function (Utils, ClientError, ErrorContext, Operations) {

    'use strict';

    // private global functions ===============================================

    /**
     * Returns whether the options object contains the undo flag.
     */
    function isUndo(options) {
        return options && options.undo;
    }

    // class OperationsGenerator ==============================================

    /**
     * An instance of this class contains an array of document operations, and
     * an array of associated undo operations, and provides methods to generate
     * new operations for either of the operation arrays.
     *
     * @constructor
     *
     * @param {EditModel} docModel
     *  The document model containing this generator.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Boolean} [initOptions.applyImmediately=false]
     *      If set to true, all document operations that will be generated and
     *      inserted into this generator instance will be applied immediately
     *      at the document model. In this case, it is NOT allowed to insert
     *      operations at the beginning of the internal array of document
     *      operation (the option 'prepend' MUST NOT be used except for undo
     *      operations). It MUST be possible to apply the operations without
     *      error. If the operation handler throws an exception, the entire
     *      application will immediately switch to internal error state.
     */
    function OperationsGenerator(docModel, initOptions) {

        // protected properties -----------------------------------------------

        // the cached document model (protected property, can be used in derived classes)
        this._docModel = docModel;

        // private properties -------------------------------------------------

        // the buffer for regular document operations (private property)
        this._docOperations = [];

        // the buffer for undo operations (private property)
        this._undoOperations = [];

        // whether to apply generated/inserted operations immediately
        this._applyImmediately = Utils.getBooleanOption(initOptions, 'applyImmediately', false);

    } // class OperationsGenerator

    // static methods ---------------------------------------------------------

    /**
     * Converts the passed document operations to a plain JS array.
     *
     * @param {Object|Array<Object>|OperationsGenerator} operations
     *  A single JSON operation, or an array of JSON operations, or an
     *  operations generator instance with operations.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      Whether to return the undo operations from an operations generator
     *      instance.
     *
     * @returns {Array<Object>}
     *  The document operations, as plain JS array.
     */
    OperationsGenerator.getArray = function (operations, options) {
        return (operations instanceof OperationsGenerator) ? operations.getOperations(options) : _.getArray(operations);
    };

    // private methods --------------------------------------------------------

    OperationsGenerator.prototype._get = function (options) {
        return isUndo(options) ? this._undoOperations : this._docOperations;
    };

    OperationsGenerator.prototype._set = function (operations, options) {
        if (isUndo(options)) { this._undoOperations = operations; } else { this._docOperations = operations; }
    };

    OperationsGenerator.prototype._internalError = function (msg) {
        Utils.error('OperationsGenerator.appendOperations(): ' + msg);
        this._docModel.getApp().setInternalError(ClientError.ERROR_WHILE_MODIFYING_DOCUMENT, ErrorContext.GENERAL);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the generated document operations will be applied
     * immediately at the document model. See constructor option
     * 'applyImmediately' for more details.
     *
     * @returns {Boolean}
     *  Whether the generated document operations will be applied immediately.
     */
    OperationsGenerator.prototype.isImmediateApplyMode = function () {
        return this._applyImmediately;
    };

    /**
     * Returns the number of document operations that have been generated so
     * far.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      Whether to return the number of undo operations.
     *
     * @returns {Number}
     *  The number of document operations that have been generated so far.
     */
    OperationsGenerator.prototype.getOperationCount = function (options) {
        return this._get(options).length;
    };

    /**
     * Returns the array with all document operations that have been generated
     * so far. Note that this method does not remove the document operations
     * from this generator instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      Whether to return the undo operations array.
     *
     * @returns {Array<Object>}
     *  The array with all generated document operations. For performance
     *  reasons, this array is a reference to the internal stoirage, and MUST
     *  NOT be changed.
     */
    OperationsGenerator.prototype.getOperations = function (options) {
        return this._get(options);
    };

    /**
     * Reverses the entire operations array in-place. This method MUST NOT be
     * used to reverse regular document operations, if they have been applied
     * immediately (see constructor option 'applyImmediately' for details).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the undo operations array will be reversed.
     *
     * @returns {OperationsGenerator}
     *  A reference to this instance.
     */
    OperationsGenerator.prototype.reverseOperations = function (options) {

        // regular operations must not be reversed when they have been applied immediately
        if (this._applyImmediately && !isUndo(options)) {
            this._internalError('cannot reverse operations that have been applied immediately');
        }

        this._get(options).reverse();
        return this;
    };

    /**
     * Removes the JSON operations from the operations array.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the undo operations will be removed from the undo
     *      operations array.
     *
     * @returns {OperationsGenerator}
     *  A reference to this instance.
     */
    OperationsGenerator.prototype.clearOperations = function (options) {

        // regular operations must not be cleared when they have been applied immediately
        if (this._applyImmediately && !isUndo(options)) {
            this._internalError('cannot clear operations that have been applied immediately');
        }

        this._get(options).splice(0);
        return this;
    };

    /**
     * Appends the passed JSON operations to the operations array.
     *
     * @param {Object|Array<Object>|OperationsGenerator} operations
     *  A single JSON operation, or an array of JSON operations, or an
     *  operations generator instance with operations to be appended to the
     *  operations array. Note that an operations generator passed to this
     *  method will NOT be cleared after its operations have been appended.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the operations will be appended to the undo
     *      operations array. If the passed parameter is an instance of this
     *      class, its undo operations will be inserted.
     *
     * @returns {OperationsGenerator}
     *  A reference to this instance.
     */
    OperationsGenerator.prototype.appendOperations = function (operations, options) {

        // the own operations, and the new operations, as JS arrays
        var ownOperations = this._get(options);
        var newOperations = OperationsGenerator.getArray(operations, options);

        // append the new operations (do not create new array instance for a single operation)
        if (newOperations.length === 1) {
            ownOperations.push(newOperations[0]);
        } else if (newOperations.length > 1) {
            this._set(ownOperations.concat(newOperations), options);
        }

        // apply the new operations immediately if specified (but skip other operation generators
        // that are in immediate mode by themselves)
        if (this._applyImmediately && !isUndo(options) && !((operations instanceof OperationsGenerator) && operations.isImmediateApplyMode())) {
            if (!newOperations.every(this._docModel.invokeOperationHandler, this._docModel)) {
                this._internalError('error while applying generated operations');
            }
        }

        return this;
    };

    /**
     * Prepends the passed JSON operations to the operations array. This method
     * MUST NOT be used to insert regular document operations, if they will be
     * applied immediately (see constructor option 'applyImmediately' for more
     * details).
     *
     * @param {Object|Array<Object>|OperationsGenerator} operations
     *  A single JSON operation, or an array of JSON operations, or an
     *  operations generator instance with operations to be prepended to the
     *  operations array. Note that an operations generator passed to this
     *  method will NOT be cleared after its operations have been prepended.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the operations will be prepended to the undo
     *      operations array. If the passed parameter is an instance of this
     *      class, its undo operations will be inserted.
     *
     * @returns {OperationsGenerator}
     *  A reference to this instance.
     */
    OperationsGenerator.prototype.prependOperations = function (operations, options) {

        // regular operations must not be prepended when they will be applied immediately
        if (this._applyImmediately && !isUndo(options)) {
            this._internalError('cannot prepend operations, and apply them immediately');
        }

        // the own operations, and the new operations, as JS arrays
        var ownOperations = this._get(options);
        var newOperations = OperationsGenerator.getArray(operations, options);

        // prepend the new operations (do not create new array instance for a single operation)
        if (newOperations.length === 1) {
            ownOperations.unshift(newOperations[0]);
        } else if (newOperations.length > 1) {
            this._set(newOperations.concat(ownOperations), options);
        }

        return this;
    };

    /**
     * Appends or prepends the passed JSON operations to the operations array.
     * This method MUST NOT be used to insert regular document operations at
     * the beginning, if they will be applied immediately (see constructor
     * option 'applyImmediately' for more details).
     *
     * @param {Object|Array<Object>|OperationsGenerator} operations
     *  A single JSON operation, or an array of JSON operations, or an
     *  operations generator instance with operations to be inserted into the
     *  operations array. Note that an operations generator passed to this
     *  method will NOT be cleared after its operations have been inserted.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the operations will be inserted into the undo
     *      operations array. If the passed parameter is an instance of this
     *      class, its undo operations will be inserted.
     *  @param {Boolean} [options.prepend=false]
     *      If set to true, the operations will be prepended, otherwise
     *      appended to the operations array.
     *
     * @returns {OperationsGenerator}
     *  A reference to this instance.
     */
    OperationsGenerator.prototype.insertOperations = function (operations, options) {
        return (options && options.prepend) ?
            this.prependOperations(operations, options) :
            this.appendOperations(operations, options);
    };

    /**
     * Creates and appends a new operation to the operations array.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the JSON operation.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the operation will be inserted into the undo
     *      operations array.
     *  @param {Boolean} [options.prepend=false]
     *      If set to true, the operation will be prepended, otherwise appended
     *      to the operations array.
     *
     * @returns {Object}
     *  The created operation object.
     */
    OperationsGenerator.prototype.generateOperation = function (name, properties, options) {
        var operation = _.extend({ name: name }, properties);
        this.insertOperations(operation, options);
        return operation;
    };

    /**
     * Creates and appends a new operation to the operations array that changes
     * one or more global document properties.
     *
     * @param {Object} attributes
     *  The document attributes to be inserted into the operation, as simple
     *  key/value map.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, the operation will be inserted into the undo
     *      operations array.
     *  @param {Boolean} [options.prepend=false]
     *      If set to true, the operation will be prepended, otherwise appended
     *      to the operations array.
     *
     * @returns {Object}
     *  The created operation object.
     */
    OperationsGenerator.prototype.generateDocAttrsOperation = function (attributes, options) {
        return this.generateOperation(Operations.SET_DOCUMENT_ATTRIBUTES, { attrs: { document: attributes } }, options);
    };

    // exports ================================================================

    return _.makeExtendable(OperationsGenerator);

});
