/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/fontcollection', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/deferredutils',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/render/font',
    'io.ox/office/baseframework/model/modelobject'
], function (Config, Utils, DeferredUtils, SimpleMap, Font, ModelObject) {

    'use strict';

    // all known fonts, mapped by font name
    var PREDEFINED_FONTS = {

        Helvetica:              { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
        Arial:                  { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] } },
        'Arial Black':          { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['sans serif', 'Arial'] }, hidden: true },
        // Arial Narrow is overwritten in themecollection.js
        Verdana:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
        Tahoma:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
        Calibri:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Carlito', 'Arial'] } },
        'Calibri Light':        { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Carlito', 'Arial'] } },
        Carlito:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Calibri', 'Arial'] }, hidden: true },
        Albany:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] }, hidden: true },
        Impact:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['sans serif', 'Arial'] } },
        // Bug 49056 in Linux is "sans serif" not the same as "sans-serif"

        'Helvetica Neue':       { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] }, hidden: true },
        'Helvetica Light':      { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] }, hidden: true },
        'Open Sans':            { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica Neue'] }, hidden: true },
        'Open Sans Light':      { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
        'Open Sans Semibold':   { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
        'Open Sans Extrabold':  { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans Semibold'] }, hidden: true },

        Times:                  { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
        'Times New Roman':      { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times'] } },
        Georgia:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] } },
        Palatino:               { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] } },
        Cambria:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Caladea', 'Times New Roman'] } },
        Caladea:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Cambria', 'Times New Roman'] }, hidden: true },
        Thorndale:              { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
        'Book Antiqua':         { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Palatino'] } },
        'Calisto MT':           { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Book Antiqua'] }, hidden: true },

        Courier:                { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true },
        'Courier New':          { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier'] } },
        'Andale Mono':          { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] } },
        Consolas:               { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] } },
        Cumberland:             { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true }
    };

    // a regular expression that matches valid font names
    var RE_VALID_FONTNAME = new RegExp('^[^<>[\\]()"\'&/\\\\]+$');

    // key in localStorage were the font data is saved
    var LOCAL_STORAGE_FONT_KEY = 'appsuite.office-fonts';

    // key in localStorage were blockDownloadToken token is saved
    var LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY = LOCAL_STORAGE_FONT_KEY + '-do-not-download-again';

    // class FontCollection ===================================================

    /**
     * Contains the definitions of all known fonts in a document.
     *
     * Instances of this class trigger the following events:
     * - 'insert:font'
     *      After a new font definition has been inserted into this collection,
     *      or an existing font definition has been changed. Event handlers
     *      receive the name of the changed font.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    function FontCollection(docModel) {

        // all registered fonts, mapped by capitalized font name
        var fontMap = new SimpleMap();

        var debugStateImportReplacementFonts = {
            fontLoadedWithAjax: false,
            fontReadyInDOM: false,
            fontAttachedFromLocalStorage: false,
            installedOrReplacementAvailable: false
        };

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Removes the calculated CSS font families from all registered fonts.
         */
        function deleteAllCssFontFamilies() {
            fontMap.forEach(function (font) { delete font.cssFontFamily; });
        }

        /**
         * Calculates and returns the CSS font family attribute containing the
         * specified font name and all alternative font names.
         */
        function calculateCssFontFamily(fontName) {

            // the resulting collection of CSS font names
            var cssFontNames = [fontName];
            // the last found CSS default font family
            var lastCssDefault = null;

            // collects all alternative font names of the fonts in the passed array
            function collectAltNames(fontNames) {

                // alternative font names of the passed fonts
                var altFontNames = [];

                // collect all alternative font names
                fontNames.forEach(function (fontName2) {
                    fontMap.with(fontName2, function (fontDesc) {
                        if (_.isArray(fontDesc.fontProps.altNames)) {
                            altFontNames = altFontNames.concat(fontDesc.fontProps.altNames);
                            if (typeof fontDesc.cssDefault === 'string') {
                                lastCssDefault = fontDesc.cssDefault;
                            }
                        }
                    });
                });

                // unify the new alternative font names and remove font names
                // already collected in cssFontNames (prevent cyclic references)
                altFontNames = _.chain(altFontNames).unique().difference(cssFontNames).value();

                // insert the new alternative font names into result list, and
                // call recursively until no new font names have been found
                if (altFontNames.length > 0) {
                    cssFontNames = cssFontNames.concat(altFontNames);
                    collectAltNames(altFontNames);
                }
            }

            // collect alternative font names by starting with passed font name
            collectAltNames([fontName]);

            // finally, add the CSS default family
            if (_.isString(lastCssDefault)) {
                cssFontNames.push(lastCssDefault);
            }

            // remove invalid/dangerous characters
            cssFontNames = cssFontNames.filter(RE_VALID_FONTNAME.test, RE_VALID_FONTNAME);

            // enclose all fonts with white-space into quote characters
            cssFontNames.forEach(function (cssFontName, index) {
                if (/\s/.test(cssFontName)) {
                    cssFontNames[index] = '\'' + cssFontName.replace(/\s+/g, ' ') + '\'';
                }
            });

            // return the CSS font-family attribute value
            return cssFontNames.join(',');
        }

        /**
         * Check if the the given font is available in the browser. It's available
         * when it is installed at the client or when it is already in the DOM as a web font.
         * It was created to work with Calibri', 'Carlito', 'Cambria' and 'Caladea', so in other
         * cases see the restriction below.
         *
         * Important restriction: This does only work 100% reliable for the fonts 'Calibri', 'Carlito',
         * 'Cambria' and 'Caladea' at the moment. That said, it should work for all fonts that have a
         * different metric to 'monospace'. So just check if the detection for the font that should
         * be checked works when installed and when not installed.
         *
         * But to be reliable for every font without checking it before, this function needs to be enhanced
         * with at least a second check in addition to the monospace font.
         */
        function isFontAvailableInBrowser(font) {
            var // width from the container with the test font
                testFontWidth,
                // width from the container with the fallback font
                fallbackFontWidth,
                // name from the fallback font that is compared against the test font
                fallBackFont = 'monospace';

            // Calculates and returns the container width for the given font.
            // When this font is not available, it uses the fallback font and
            // therefore returns the container width for the fallback font.
            function calcFontMetric(font, fontFallback) {
                var
                    // were the hiddenCalcContainer is appended
                    rootNode = $('body'),
                    // container for calculation, with the font and the fallback font attached, it's moved offscreen
                    hiddenCalcContainer = $('<div style="position:absolute; width:auto; font-size: 100px; left:-9000px; font-family:' + font + ',' + fontFallback + ';"</div>'),
                    // the span with a string for measurements
                    textToMeasure = $('<span>abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234</span>'),
                    // width from the container with the current font
                    fontWidth;

                rootNode.append(hiddenCalcContainer.append(textToMeasure));
                fontWidth = hiddenCalcContainer.outerWidth();
                hiddenCalcContainer.remove();
                return fontWidth;
            }

            testFontWidth = calcFontMetric(font, fallBackFont);
            fallbackFontWidth = calcFontMetric(fallBackFont, fallBackFont);

            // When the testFontWidth is equal to the fallbackFontWidth it means that the testFont was not used by the browser.
            // Therefore the font is available when both width values are different.
            return testFontWidth !== fallbackFontWidth;
        }

        // public methods -----------------------------------------------------

        /**
         * Inserts a new font definition into this container. An existing font
         * definition with the specified name will be replaced.
         *
         * @param {String} fontName
         *  The name of of the new font.
         *
         * @param {Object} fontProps
         *  The definition properties of the font.
         *
         * @returns {FontCollection}
         *  A reference to this instance.
         */
        this.insertFont = function (fontName, fontProps) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // store passed font attributes
            fontProps = _.copy(fontProps, true);
            var fontDesc = fontMap.insert(fontName, { fontProps: fontProps, hidden: false });

            // validate alternative names
            if (_.isArray(fontProps.altNames)) {
                fontProps.altNames = _.chain(fontProps.altNames).without('').map(Utils.capitalizeWords).value();
            } else {
                fontProps.altNames = [];
            }

            // add predefined alternative names
            if ((fontName in PREDEFINED_FONTS) && _.isArray(PREDEFINED_FONTS[fontName].fontProps.altNames)) {
                fontProps.altNames = fontProps.altNames.concat(PREDEFINED_FONTS[fontName].fontProps.altNames);
            }

            // unify alternative names, remove own font name
            fontProps.altNames = _.chain(fontProps.altNames).unique().without(fontName).value();

            // add default CSS font family if known
            if (fontName in PREDEFINED_FONTS) {
                fontDesc.cssDefault = PREDEFINED_FONTS[fontName].cssDefault;
            }
            // remove calculated CSS font families of all fonts, notify listeners
            deleteAllCssFontFamilies();
            this.trigger('insert:font', fontName);

            return this;
        };

        /**
         * Returns the names of all predefined and registered fonts. Predefined
         * fonts with the hidden flag that have not been registered will not be
         * included.
         *
         * @returns {Array<String>}
         *  The names of all predefined and registered fonts, as array.
         */
        this.getFontNames = function () {
            return fontMap.reduce([], function (fontNames, fontDesc, fontName) {
                if (!fontDesc.hidden) { fontNames.push(fontName); }
                return fontNames;
            });
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @param {Theme} [theme]
         *  The document theme used to resolve abstract font names.
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName, theme) {

            // resolve abstract font names from the passed document theme
            if (theme) { fontName = theme.getFontName(fontName); }

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // default for unregistered fonts: just the passed font name
            var fontDesc = fontMap.get(fontName, null);
            if (!fontDesc) { return fontName; }

            // calculate and cache the CSS font family
            if (!_.isString(fontDesc.cssFontFamily)) {
                fontDesc.cssFontFamily = calculateCssFontFamily(fontName);
            }

            return fontDesc.cssFontFamily;
        };

        /**
         * Calculates the scaled font size for the passed font size and zoom
         * factor.
         *
         * @param {Number} fontSize
         *  The original font size, in points.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor. The value 1 represents the normal zoom (of 100%).
         *
         * @returns {Number}
         *  The scaled font size, in points, rounded to 1/10 of points.
         */
        this.getScaledFontSize = function (fontSize, zoom) {
            if (typeof zoom === 'number') { fontSize *= zoom; }
            return Utils.round(fontSize, 0.1);
        };

        /**
         * Converts the passed character formatting attributes to a font
         * descriptor, according to the font registered in this collection, and
         * the passed zoom factor.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations, to be
         *  converted to the CSS font style, with the following properties:
         *  @param {String} charAttributes.fontName
         *      The original font name (case-insensitive).
         *  @param {Number} charAttributes.fontSize
         *      The original font size, in points.
         *  @param {Boolean} [charAttributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [charAttributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Theme} [options.theme]
         *      The document theme used to resolve abstract font names.
         *  @param {Number} [options.zoom=1]
         *      The zoom factor used to scale the font size. The value 1 (the
         *      default of this option) represents the normal zoom (of 100%).
         *
         * @returns {Font}
         *  The resulting font. The property 'family' includes all known
         *  fall-back fonts for the font. The property 'size' contains the font
         *  size as number (scaled according to the zoom factor if specified).
         */
        this.getFont = function (charAttributes, options) {
            return new Font(
                this.getCssFontFamily(charAttributes.fontName, Utils.getOption(options, 'theme', null)),
                this.getScaledFontSize(charAttributes.fontSize, Utils.getNumberOption(options, 'zoom', 1)),
                charAttributes.bold,
                charAttributes.italic
            );
        };

        /**
         * Calculates the normal line height for the passed character
         * attributes, as reported by the current browser. Note that this value
         * may vary for the same font settings in different browsers.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FontCollection.getFont() for
         *  details.
         *
         * @returns {Number}
         * The normal line height for the specified font, in pixels (with a
         * precision of 1/100 pixel for browsers using fractional line heights
         * internally).
         */
        this.getNormalLineHeight = function (charAttributes, options) {
            return this.getFont(charAttributes, options).getNormalLineHeight();
        };

        /**
         * Calculates the position of the font base line for the passed
         * character attributes and line height, as reported by the current
         * browser. Note that this value may vary for the same font settings in
         * different browsers.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {Number} [lineHeight]
         *  The line height to be used to determine the position of the font
         *  base line. If omitted, the normal line height, as reported by the
         *  method FontCollection.getNormalLineHeight(), will be used.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FontCollection.getFont() for
         *  details.
         *
         * @returns {Number}
         *  The position of the font base line for the passed character
         *  attributes and line height, in pixels, relative to the top border
         *  of the text line.
         */
        this.getBaseLineOffset = function (charAttributes, lineHeight, options) {
            return this.getFont(charAttributes, options).getBaseLineOffset(lineHeight);
        };

        /**
         * Returns the effective width of the passed text, in pixels.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {String} text
         *  The text whose width will be calculated. If the text consists of
         *  one character only, the calculated text width will be cached
         *  internally for subsequent invocations with the same text and
         *  character attributes.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FontCollection.getFont() for
         *  details.
         *
         * @returns {Number}
         *  The effective width of the passed text, in pixels.
         */
        this.getTextWidth = function (charAttributes, text, options) {
            // empty string: always zero (do not create a font)
            return (text.length === 0) ? 0 : this.getFont(charAttributes, options).getTextWidth(text);
        };

        /**
         * Returns measures for the digits '0' to '9'.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FontCollection.getFont() for
         *  details.
         *
         * @returns {Array<Number>}
         *  An array with the widths of the digits '0' to '9', in pixels, and
         *  additional properties. See method Font.getDigitWidths() for more
         *  details.
         */
        this.getDigitWidths = function (charAttributes, options) {
            return this.getFont(charAttributes, options).getDigitWidths();
        };

        /**
         * Splits the passed string to text lines that all fit into the passed
         * width, using the specified font settings.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {String} text
         *  The text that will be split into single text lines.
         *
         * @param {Number} width
         *  The maximum width of the text lines, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FontCollection.getFont() for
         *  details.
         *
         * @returns {Array}
         *  An array of strings containing the single text lines.
         */
        this.getTextLines = function (charAttributes, text, width, options) {
            return this.getFont(charAttributes, options).getTextLines(text, width);
        };

        /**
         * Make replacement fonts for 'Calibri' and 'Cambria' available in the browser,
         * when neither these two nor the replacement fonts exists on the client.
         *
         * Loading:
         * When they are not existing, the replacement fonts are loaded from the server,
         * stored in localStorage and added to the DOM to be useable.
         *
         * font data:
         * The replacement fonts (tk/fonts/fonts.css) are base64 encoded woff files that are
         * attached directly to the <head> in the DOM.
         *
         * Already stored:
         * When the replacement fonts are already in localStorage, they don't need to be loaded
         * again. And when they are already in the DOM they even must not attached again
         * (font data would be two times in DOM then).
         *
         * LocalStorage and logout:
         * It's important that in the 'LOCAL_STORAGE_FONT_KEY' ('appsuite.office-fonts')
         * is added to the 'fileList' in '/core/cache/localstorage.js', see web repo
         * commit: 5a5a47c6d9a220ca501563cdc226c09ac9f34a99), so that this key is
         * not deleted after a logout from the appSuite.
         *
         * Prevent multiple downloads:
         * Also, to prevent constant attempts to download fonts when they can't be stored
         * in the localStorage, e.g. because of the size (which should not happen normally), only a
         * small token is set in the localStorage to prevent downloading until it is cleared
         * (e.g. when the localStorage is cleared, this token is cleared too and therefore a
         * new attempt is made to download and store the font in localStorage).
         *
         * Debug-Mode:
         * There is a special debug-mode to test font-loading with an own created test font.
         * This is useful to test the font loading in windows (selenium) where the Calibri and Cambria
         * fonts are always available. Or in Linux when 'Fontconfig' has Calibri replacement
         * fonts on OS level that are not possible to detect (for more information see our wiki).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the font is available in the browser,
         *  or rejected when an error has occurred.
         */
        this.importReplacementFonts = function () {

            var
                loadDeferred = DeferredUtils.createDeferred(docModel.getApp(), 'FontCollection.importReplacementFonts()'),
                // whether 'Calibri' or the metric compatible replacement 'Carlito' is available at the device
                calibriOrReplacementAvailible = isFontAvailableInBrowser('Calibri') || isFontAvailableInBrowser('Carlito'),
                // whether 'Cambria' or the metric compatible replacement 'Caladea' is available at the device
                cambriaOrReplacementAvailible = isFontAvailableInBrowser('Cambria') || isFontAvailableInBrowser('Caladea'),
                // data from the localStorage
                localStorageData,
                // whether a possible font download should be allowed
                blockDownloadToken = false,
                // needed as an exit condition/timeout
                pollingStartTime,
                // web font name that is used in debug mode
                debugFontName = 'oxdebugwebfontox';

            // When used with 'office:testautomation=true', the importReplacementFonts is tested with a own, unique font (see 'debugFontName') that doesn't exists on Windows and Linux.
            // Since Calibri and Cambria are system fonts in Windows, it would not work otherwise. In Linux 'Fontconfig' may generate Calibri replacement fonts on OS level that are not possible to detect.
            function debugReplacementFonts() {
                return !Config.AUTOTEST || isFontAvailableInBrowser(debugFontName);
            }

            // Attach the CSS content which contains the web font to DOM
            function addFontToDOM(cssFontData) {

                // when this node exists, the font data is also attached to the DOM, so don't do it again
                if (Utils.findHiddenNodes('.font-preloading-helper').length < 1) {
                    $('head').append('<style type="text/css">' + cssFontData + '</style>');

                    // insert a hidden node to reference the replacement fonts in the DOM. This starts using the font-face rule (note: 'A' is a used glyph in 'debugFontName' font )
                    Utils.insertHiddenNodes($('<div class="font-preloading-helper"><div style="font-family: Carlito;">g</div><div style="font-family: Caladea;">g</div><div style="font-family: ' +  debugFontName + ';">A</div></div>'));

                    // trigger a reflow in the browser, at least Chrome needs this to make
                    // the font really available for rendering in the browser
                    Utils.findHiddenNodes('.font-preloading-helper').outerWidth();

                    pollingStartTime = Date.now();

                    // Check and resolve the Promise if the font is available in the browser. This must
                    // be checked async, because when the code runs synchronous e.g. Chrome is not able to make
                    // the font available for rendering in the browser. This check should normally resolve with
                    // true in the first round, but browser can behave differently here. So better be save and
                    // check until the font is finally available or the interval is timed out.
                    var isFontLoadedInBrowserPolling = window.setInterval(function () {

                        // exit condition: break when the font is still not ready in the browser after 2000ms
                        if ((2000 < (Date.now() - pollingStartTime))) {
                            //console.log(Date.now() - pollingStartTime);
                            window.clearInterval(isFontLoadedInBrowserPolling);
                            loadDeferred.resolve();

                            Utils.warn('ImportReplacementFonts - Timeout while checking font availability for rendering in the Browser after appending font data to DOM');

                        // check if the font is available now
                        } else if (isFontAvailableInBrowser('Carlito') && isFontAvailableInBrowser('Caladea') && debugReplacementFonts()) {

                            window.clearInterval(isFontLoadedInBrowserPolling);
                            loadDeferred.resolve();
                            Utils.log('ImportReplacementFonts - Font is finally available for rendering in the browser after appending font data to DOM');
                            // set debug state
                            debugStateImportReplacementFonts.fontReadyInDOM = true;

                        }
                    }, 100);
                }
            }

            // only for debug
            Utils.withLogging(function () {
                Utils.log('ImportReplacementFonts - Calibri:', isFontAvailableInBrowser('Calibri'), ' - Carlito:', isFontAvailableInBrowser('Carlito'), ' - Cambria:', isFontAvailableInBrowser('Cambria'), ' - Caladea:', isFontAvailableInBrowser('Caladea'), '| FontDebugMode:', Config.AUTOTEST, '- ' + debugFontName + ':', isFontAvailableInBrowser(debugFontName));
            });

            // check if font or replacement is installed locally, or already available in the browser
            if (calibriOrReplacementAvailible && cambriaOrReplacementAvailible && debugReplacementFonts()) {
                loadDeferred.resolve();
                Utils.log('ImportReplacementFonts - Needed fonts are installed or a replacement fonts is already available for rendering in the browser');
                // set debug state
                debugStateImportReplacementFonts.installedOrReplacementAvailable = true;

            // when font is not available in the browser
            } else {
                // get the data from local storage
                localStorageData = localStorage.getItem(LOCAL_STORAGE_FONT_KEY);

                // add font when it's cached in localStorage
                if (localStorageData !== null && localStorageData !== undefined) {

                    // add font to DOM to be useable
                    addFontToDOM(localStorageData);

                    Utils.log('ImportReplacementFonts - Attaching font data from localStorage to DOM');
                    // set debug state
                    debugStateImportReplacementFonts.fontAttachedFromLocalStorage = true;

                // when not cached, load them from server
                } else {

                    blockDownloadToken = localStorage.getItem(LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY) === 'true';

                    // Check if the localStorage is available and really writable (possible problem with browser incognito mode).
                    // Also do not try to download again when localStorage has no free size left
                    // important: the token and also the whole localStorage, besides the fonts,
                    // should be cleared after a logout. Since the token is also deleted, it will
                    // try to download it again after that.
                    if ((typeof (Storage) !== 'undefined') && Utils.isLocalStorageSupported() && !blockDownloadToken) {

                        // get the css file which contains the base64 encoded fonts
                        $.ajax({
                            url: ox.root + '/apps/io.ox/office/tk/render/font.css',
                            dataType: 'text',
                            timeout: 40000 // with 3G 512 kbit/s the 1.7mb font.css needs about 27s to download, add some safety buffer
                        }).done(function (data) {

                            Utils.log('ImportReplacementFonts - Loaded fonts via AJAX-request, attach data to localStorage');
                            // set debug state
                            debugStateImportReplacementFonts.fontLoadedWithAjax = true;

                            try {
                                // put to local storage
                                localStorage.setItem(LOCAL_STORAGE_FONT_KEY, data);
                                // add font to DOM to be useable
                                addFontToDOM(data);

                            } catch (ex) {
                                Utils.warn('ImportReplacementFonts - Failed to store data in localStorage', ex);

                                try {
                                    // set token
                                    localStorage.setItem(LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY, true);
                                } catch (ex2) {
                                    Utils.warn('ImportReplacementFonts - Failed to store do-not-download-again token in localStorage', ex2);
                                }

                                loadDeferred.resolve();
                            }

                        }).fail(function () {

                            // fail mean that just the replacement is not available, everything works like before the replacement fonts existed
                            loadDeferred.resolve();

                            Utils.warn('ImportReplacementFonts - Loading font via AJAX-request failed, probably timed out');
                        });

                    // do nothing when no localStorage is available, everything works like before the replacement fonts existed
                    } else {
                        loadDeferred.resolve();
                        Utils.warn('ImportReplacementFonts - No localStorage supported or "do-not-download-again" token set', blockDownloadToken);
                    }
                }
            }

            return docModel.createAbortablePromise(loadDeferred, function () { Utils.warn('ImportReplacementFonts -  Promise aborted'); });
        };

        /**
         * Deletes all web replacement fonts related items from local storage.
         */
        this.deleteWebFontsFromLocalStorage = function () {
            localStorage.removeItem(LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY);
            localStorage.removeItem(LOCAL_STORAGE_FONT_KEY);
        };

        this.getImportReplacementFontsState = function () {
            return debugStateImportReplacementFonts;
        };

        // initialization -----------------------------------------------------

        // start with all predefined fonts
        _.each(PREDEFINED_FONTS, function (fontDesc, fontName) {
            fontMap.insert(fontName, _.copy(fontDesc, true));
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = fontMap = null;
        });

    } // class FontCollection

    // static methods ---------------------------------------------------------

    /**
     * Returns the first font name specified in the passed font family string
     * received from a CSS font-family attribute. The CSS font family may
     * consist of a comma separated list of font names, where the font names
     * may be enclosed in quote characters.
     *
     * @param {String} cssFontFamily
     *  The value of a CSS font-family attribute.
     *
     * @returns {String}
     *  The first font name from the passed string, without quote characters.
     */
    FontCollection.getFirstFontName = function (cssFontFamily) {

        // extract first font name from comma separated list
        var fontName = $.trim(cssFontFamily.split(',')[0]);

        // Remove leading and trailing quote characters (be tolerant and allow
        // mixed quotes such as "Font' or even a missing quote character on one
        // side of the string). After that, trim again and return.
        return Utils.capitalizeWords($.trim(fontName.match(/^["']?(.*?)["']?$/)[1]).toLowerCase());
    };

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: FontCollection });

});
