/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/editframework/app/mailactions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/core/notifications',
    'io.ox/mail/api',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/io',
    'io.ox/office/baseframework/utils/infomessages',
    'io.ox/office/baseframework/app/extensionregistry',
    'gettext!io.ox/office/editframework/actions'
], function (ext, links, CoreNotification, MailApi, Utils, DriveUtils, IO, InfoMessages, ExtensionRegistry, gt) {

    'use strict';

    // static class MailActions ===============================================

    var MailActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Mail application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {FileExtensions} fileExtensions
     *  A singleton object containing configuration settings of all file
     *  extensions supported by this application instance.
     *
     */
    MailActions.create = function (appBaseName) {

        var // the module name of the edit application
            ACTION_POINT = 'io.ox/mail/office/' + appBaseName,
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(options) {
            ox.launch(MODULE_NAME + '/main', options);
        }

        // initialization -----------------------------------------------------

        // Copy mail attachment to Files and edit
        new links.Action(ACTION_POINT + '-edit-asnew', {
            id: 'save',
            capabilities: 'infostore',
            requires: function (e) {

                if (!e.collection.has('one')) {
                    return false;
                }

                var filename = null;
                if (_.isArray(e.context)) {
                    // Fix for 46557: e.context can be an object or array dependent on the context
                    // Show operations for attachments or show all attachments in a list with operations
                    filename = e.context[0].filename;
                } else {
                    filename = e.context.filename;
                }

                // whether the "edit" link should be shown for the attachment
                var editable = ExtensionRegistry.isEditable(filename, MODULE_NAME);
                if (!editable) {
                    return false;
                }

                // prefetch application source code
                IO.prefetchModuleSource(MODULE_NAME);
                return true;
            },
            multiple: function (list) {
                MailApi.saveAttachments(list, DriveUtils.getStandardDocumentsFolderId()).done(function (result) {
                    if (result && result[0] && result[0].error) {
                        Utils.error('MailApi.saveAttachments makes an error', result[0]);
                        require(['io.ox/office/baseframework/utils/errormessages']).done(function (ErrorMessages) {
                            CoreNotification.yell('error', ErrorMessages.getMessageData('GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR').message);
                        });
                        return;
                    }

                    // the baton data of the created Drive file returned by the mail API
                    var baton = result[0];
                    // the file descriptor passed to the OX Documents application launcher
                    var file = _.clone(baton.data);

                    // Bug 40376 add the origin property to the file descriptor
                    file.source = 'drive';
                    file.origin = { source: 'mail', folder_id: file.mailFolder, id: file.mailUID };
                    delete file.mailFolder;
                    delete file.mailUID;

                    // show a notification to the user that a new file has been created in Drive
                    CoreNotification.yell('success', InfoMessages.getMessageData('INFO_DOC_STORED_IN_DEFAULTFOLDER_AS', { fullFileName: file.filename }).message);
                    // launch the OX Documents editor application
                    if (ExtensionRegistry.isNative(file.filename)) {
                        launchApplication({ action: 'load', file: file });
                    } else {
                        launchApplication({ action: 'convert', target_folder_id: file.folder_id, templateFile: file, preserveFileName: true });
                    }
                })
                .fail(CoreNotification.yell);
            }
        });

        var link = {
            id: appBaseName + '-edit',
            index: 1000,
            prio: 'lo',
            mobile: 'lo',
            label: gt('Edit as new'),
            ref: ACTION_POINT + '-edit-asnew'
        };

        // an action link to edit files attached to an e-mail
        ext.point('io.ox/mail/attachment/links').extend(new links.Link(link));

        // an action link to edit files attached to an e-mail
        ext.point('io.ox/core/viewer/toolbar/links/mail').extend(new links.Link(link));

    };

    // exports ================================================================

    return MailActions;

});
