/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/editcontroller', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/app/basecontroller',
    'io.ox/office/editframework/utils/editconfig'
], function (Utils, BaseController, Config) {

    'use strict';

    // default search configuration
    var DEFAULT_SEARCH_SETTINGS = {
        // current search query
        query: '',
        // whether 'query' is a regular expression
        regexp: false,
        // the replacement string
        replace: ''
    };

    // class EditController ===================================================

    /**
     * The base class for controller classes of all OX Documents allowing to
     * edit a document.
     *
     * @constructor
     *
     * @extends BaseController
     *
     * @param {EditApplication} app
     *  The application that has created this controller instance.
     *
     * @param {EditModel} docModel
     *  The document model created by the passed application.
     *
     * @param {EditView} docView
     *  The document view created by the passed application.
     *
     * @param {Function} searchHandler
     *  A callback function that will implement searching for a specific query
     *  string entered in the search/replace tool bar. Receives the following
     *  parameters:
     *  (1) {String} command
     *      The search/replace command to be executed. Will be one of the
     *      following commands:
     *      - 'search:start'
     *          Start a new search/replace session. Will be sent after entering
     *          some text in the query input field, or changing any search
     *          options which influence the matching document contents.
     *      - 'search:prev'
     *          Search for the previous occurrence of the current query string.
     *      - 'search:next'
     *          Search for the next occurrence of the current query string.
     *      - 'search:end'
     *          Finish the current search/replace session, remove all feedback
     *          for search/replace in the user interface. Will be sent after
     *          closing the search pane.
     *      - 'replace:next'
     *          Replace the next occurrence of the current query string with
     *          the replacement text.
     *      - 'replace:all'
     *          Replace all occurrences of the current query string with the
     *          replacement text.
     *  (2) {Object} settings
     *      The current search settings. The object contains the following
     *      properties:
     *      - {String} query
     *          The search query string.
     *      - {Boolean} regexp
     *          Whether the query string shall be interpreted as regular
     *          expression, instead of a literal search string.
     *      - {String} replace
     *          The string containing the replacement text for the matches.
     *  The function may return a promise, if the search action involves
     *  asynchronous operations. The view will be locked until the promise has
     *  been resolved or rejected. The promise may be rejected with an error
     *  code string which may result in an alert message box shown in the user
     *  interface. Alternatively, the function may return an explicit error
     *  code string synchronously (without using a promise). The function will
     *  be called in the context of this view instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  BaseController.
     */
    function EditController(app, docModel, docView, searchHandler, initOptions) {

        // self reference
        var self = this;

        // the current search configuration
        var searchSettings = _.clone(DEFAULT_SEARCH_SETTINGS);

        // base constructor ---------------------------------------------------

        BaseController.call(this, app, docModel, docView, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Invokes the search/replace handler passed to the constructor of this
         * instance.
         *
         * @param {String} command
         *  The search/replace command to be executed.
         *
         * @param {Object} [newSettings]
         *  New properties for the search configuration object. Will be
         *  inserted into the configuration before invoking the search handler.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the search handler has
         *  finished the specified operation, or that will be rejected on any
         *  error.
         */
        function executeSearch(command, newSettings) {

            // change the specified configuration properties
            _.extend(searchSettings, newSettings);

            // invoke the search handler
            var result = searchHandler.call(this, command, _.clone(searchSettings));

            // convert error code (non-empty strings) to rejected promise
            var promise = (_.isString(result) && result) ? $.Deferred().reject(result) : $.when(result);

            // TODO: show alert messages for generic errors
            promise.fail(function (result) { Utils.error('executeSearch failed', result); });

            return promise;
        }

        // initialization -----------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // application and view

            // enabled unless the application is in 'internal error' state
            'app/valid': {
                parent: 'app/imported',
                enable: function () { return !app.isInternalError(); }
            },

            // toggle visibility of the tool bars
            'view/toolbars/show': {
                // always enabled (also in read-only mode, and in application error state)
                get: function () { return docView.getToolPane().isVisible(); },
                set: function (state) { docView.getToolPane().toggle(state); }
            },

            // the identifier of the active tool bar
            'view/toolbars/tab': {
                // always enabled (also in read-only mode, and in application error state)
                get: function () { return docView.getToolBarTabs().getActiveTabId(); },
                set: function (tabId) { docView.getToolBarTabs().activateTab(tabId); },
                focusTarget: function (sourceType) {
                    // when activating via ENTER key, move focus into tool pane
                    return (sourceType === 'keyboard') ? docView.getToolPane() : null;
                }
            },

            // dummy item for the 'Save in Drive' drop-down menu
            'view/saveas/menu': {
                parent: 'app/valid',
                enable: function () { return (!app.getLaunchOption('disableSaveAs')) && (!app.isDocumentEncrypted()); }
            },

            'view/searchgroup': {
                enable: 'app/valid',
                get: function () { return docView.getSearchGroup().isVisible(); },
                set: function (state) { docView.getSearchGroup().toggle(state); },
                focusTarget: function () { return docView.getSearchGroup(); },
                // shortcut always enables the search pane (no toggling)
                shortcut: { keyCode: 'F', ctrlOrMeta: true, value: true }
            },

            // document

            // parent iten that is enabled if the edited document is based on OOXML file format
            'document/ooxml': {
                enable: function () { return app.isOOXML(); }
            },

            'document/acquireedit': {
                parent: 'app/valid',
                enable: function () { return app.isAcquireEditRightsEnabled(); }
            },

            'document/acquireeditOrPending': {
                parent: 'app/valid',
                enable: function () { return app.isAcquireEditRightsEnabledPendingUpdates(); },
                set: function () { app.acquireEditRights(); }
            },

            'document/reload': {
                parent: 'app/imported',
                enable: function () { return (app.isInternalError() && app.isReloadEnabled()); },
                set: function () { app.reloadDocument(); }
            },

            // to be used as parent item for all items that require edit mode
            'document/editable': {
                parent: 'app/valid',
                enable: function () { return docModel.getEditMode(); }
            },

            'document/readonly': {
                parent: 'app/valid',
                enable: function () { return !docModel.getEditMode(); }
            },

            'document/fullname': {
                get: function () { return app.getFullFileName(); }
            },

            'document/rename': {
                parent: ['document/editable', 'app/bundled'],
                enable: function () { return !Config.RENAME_DISABLED && (!app.isDocumentEncrypted() || !app.doesRenameNeedReload()); },
                get: function () { return app.getShortFileName(); },
                set: function (fileName) { return app.rename(fileName); }
            },

            'document/rename/dialog': {
                parent: 'document/rename',
                set: function () { return docView.showRenameDialog(); }
            },

            'document/undo': {
                parent: 'document/editable',
                enable: function () { return docView.isUndoAvailable(); },
                set: function () { return docView.undo(); },
                shortcut: [
                   // bug 33077: restrict to application pane, to not interfere with text field's
                   // native undo/redo and automatic item execution after losing focus
                    { keyCode: 'Z', ctrlOrMeta: true, selector: '.app-pane' },
                    { keyCode: 'BACKSPACE', alt: true, selector: '.app-pane' }
                ]
            },

            'document/redo': {
                parent: 'document/editable',
                enable: function () { return docView.isRedoAvailable(); },
                set: function () { return docView.redo(); },
                shortcut: [
                    // bug 33077: restrict to application pane, to not interfere with text field's
                    // native undo/redo and automatic item execution after losing focus
                    { keyCode: 'Y', ctrlOrMeta: true, selector: '.app-pane' },
                    { keyCode: 'BACKSPACE', shift: true, alt: true, selector: '.app-pane' }
                ]
            },

            'document/saveas/dialog': {
                parent: 'app/valid',
                set: function (type) { return docView.showSaveAsDialog(type); }
            },

            'document/exportaspdf/dialog': {
              //parent: 'document/saveas/dialog',
                parent: 'app/valid',
                enable: function () { return Config.CONVERTER_AVAILABLE; },
                set: function (type) { return docView.showSaveAsDialog(type); }
            },

            'document/autosave': {
                parent: ['document/editable'],
                get: function () { return app.isAutoSaveEnabled(); },
                set: function (state) { return app.toggleAutoSave(state); },
                // auto save button must be always fresh
                alwaysDirty: true
            },

            'document/users': {
                parent: 'app/valid',
                enable: function () { return app.getActiveClients().length > 1; },
                get: function () { return docView.getCollaboratorPopup().isVisible(); },
                set: function (state) { docView.toggleCollaboratorPopup(state); }
            },

            // base item for search/replace
            'document/search': {
                // enabled in read-only mode
                parent: 'app/valid'
            },

            'document/search/text': {
                parent: 'document/search',
                get: function () { return searchSettings.query; },
                set: function (text, evt) {
                    // workaround for Bug 46401: 'custom' sourceType is triggered on blur on the textfield
                    if (evt && evt.changeEvent && evt.changeEvent.sourceType === 'custom') {
                        searchSettings.query = text;
                    } else {
                        return executeSearch('search:start', { query: text });
                    }
                }
            },

            'document/search/regexp': {
                parent: 'document/search',
                get: function () { return searchSettings.regexp; },
                set: function (state) { return executeSearch('search:start', { regexp: state }); }
            },

            'document/search/start': {
                parent: 'document/search',
                set: function () { return executeSearch('search:start'); }
            },

            'document/search/prev': {
                parent: 'document/search',
                set: function () { return executeSearch('search:prev'); },
                shortcut: [{ keyCode: 'G', ctrlOrMeta: true, shift: true }, { keyCode: 'F3', shift: true }]
            },

            'document/search/next': {
                parent: 'document/search',
                set: function () { return executeSearch('search:next'); },
                shortcut: [{ keyCode: 'G', ctrlOrMeta: true }, { keyCode: 'F3' }]
            },

            'document/search/end': {
                parent: 'document/search',
                set: function () { return executeSearch('search:end', { query: '', replace: '' }); }
            },

            'document/search/close': {
                parent: 'document/search',
                set: function () { return docView.getSearchGroup().hide(); }
            },

            // base item for replace, disabled in read-only documents
            'document/replace': {
                parent: ['document/search', 'document/editable']
            },

            'document/replace/text': {
                parent: 'document/replace',
                get: function () { return searchSettings.replace; },
                set: function (text) { searchSettings.replace = text; }
            },

            'document/replace/next': {
                parent: 'document/replace',
                set: function () { return executeSearch('replace:next'); }
            },

            'document/replace/all': {
                parent: 'document/replace',
                set: function () { return executeSearch('replace:all'); }
            }
        });

        // update controller after operations, or changed state of edit mode
        this.waitForImport(function () {
            this.listenTo(app, 'docs:state:error docs:users', function () { self.update(); });
            this.listenTo(docModel, 'operations:after change:editmode', function () { self.update(); });
            this.listenTo(docModel.getUndoManager(), 'change:count', function () { self.update(); });
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = docModel = docView = null;
        });

    } // class EditController

    // exports ================================================================

    // derive this class from class BaseController
    return BaseController.extend({ constructor: EditController });

});
