/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author York Richter <york.richtert@open-xchange.com>
 */

define.async('io.ox/office/drawinglayer/view/drawinglabels', [
    'io.ox/office/tk/io',
    'gettext!io.ox/office/drawinglayer/main'
], function (IO, gt) {

    'use strict';

    // maps drawing object types as defined/used in operations to GUI names and icons
    var DRAWING_TYPE_INFOS = {
        chart:     { icon: 'fa-bar-chart-o',        label: /*#. bar charts, line charts, pie charts, etc. */ gt.pgettext('drawing', 'Chart') },
        image:     { icon: 'fa-picture-o',          label: /*#. bitmaps, vector graphics, etc. */ gt.pgettext('drawing', 'Image') },
        shape:     { icon: 'docs-drawing-shape',    label: /*#. rectangles, circles, lines, stars, etc. */ gt.pgettext('drawing', 'Shape') },
        textframe: { icon: 'docs-insert-textframe', label: /*#. rectangles, circles, etc. with arbitrary text contents */ gt.pgettext('drawing', 'Text frame') },
        diagram:   { icon: 'fa-sitemap',            label: /*#. complex diagrams, organigrams, etc. */ gt.pgettext('drawing', 'Diagram') }
    };

    // fall-back settings for unknown/unsupported drawing objects
    var DRAWING_TYPE_DEFAULT_INFO = { icon: 'fa-picture-o', label: gt.pgettext('drawing', 'Drawing') };

    // chart type descriptors for all supported chart type
    var CHART_TYPE_STYLES = {
        'column standard':         { cat: 1, title: /*#. Chart type: vertical bars (clustered) */                                                                     gt.pgettext('chart-type', 'Column'),           type: 'column2d',  stacking: 'standard',       cjs: 'column' },
        'column stacked':          { cat: 1, title: /*#. Chart type: vertical bars (stacked from bottom to top) */                                                    gt.pgettext('chart-type', 'Column (stacked)'), type: 'column2d',  stacking: 'stacked',        cjs: 'stackedColumn' },
        'column percentStacked':   { cat: 1, title: /*#. Chart type: vertical bars (stacked from bottom to top with percent scaling) */                               gt.pgettext('chart-type', 'Column (percent)'), type: 'column2d',  stacking: 'percentStacked', cjs: 'stackedColumn100' },
        'bar standard':            { cat: 1, title: /*#. Chart type: horizontal bars (clustered) */                                                                   gt.pgettext('chart-type', 'Bar'),              type: 'bar2d',     stacking: 'standard',       cjs: 'bar' },
        'bar stacked':             { cat: 1, title: /*#. Chart type: horizontal bars (stacked from left to right) */                                                  gt.pgettext('chart-type', 'Bar (stacked)'),    type: 'bar2d',     stacking: 'stacked',        cjs: 'stackedBar' },
        'bar percentStacked':      { cat: 1, title: /*#. Chart type: horizontal bars (stacked from left to right with percent scaling) */                             gt.pgettext('chart-type', 'Bar (percent)'),    type: 'bar2d',     stacking: 'percentStacked', cjs: 'stackedBar100' },
        'line standard':           { cat: 2, title: /*#. Chart type: data points connected with lines */                                                              gt.pgettext('chart-type', 'Line'),             type: 'line2d',    stacking: 'standard',       cjs: 'line' },
        'line standard curved':    { cat: 2, title: /*#. Chart type: data points connected with curved lines */                                                       gt.pgettext('chart-type', 'Line (curved)'),    type: 'line2d',    stacking: 'standard',       cjs: 'spline', curved: true },
        'scatter standard':        { cat: 2, title: /*#. Chart type: data points with free X/Y coordinates connected with lines */                                    gt.pgettext('chart-type', 'Scatter'),          type: 'scatter2d', stacking: 'standard',       cjs: 'line' },
        'scatter standard curved': { cat: 2, title: /*#. Chart type: data points with free X/Y coordinates connected with curved lines */                             gt.pgettext('chart-type', 'Scatter (curved)'), type: 'scatter2d', stacking: 'standard',       cjs: 'spline', curved: true },
        'bubble standard':         { cat: 3, title: /*#. Chart type: data points with free X/Y coordinates drawn as circles/bubbles */                                gt.pgettext('chart-type', 'Bubble'),           type: 'bubble2d',  stacking: 'standard',       cjs: 'bubble' },
        'pie standard':            { cat: 3, title: /*#. Chart type: single pie */                                                                                    gt.pgettext('chart-type', 'Pie'),              type: 'pie2d',     stacking: 'standard',       cjs: 'pie', varyColors: true },
        'donut standard':          { cat: 3, title: /*#. Chart type: one or multiple concentric circles */                                                            gt.pgettext('chart-type', 'Donut'),            type: 'donut2d',   stacking: 'standard',       cjs: 'doughnut', varyColors: true },
        'area standard':           { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (areas overlay each other) */                        gt.pgettext('chart-type', 'Area'),             type: 'area2d',    stacking: 'standard',       cjs: 'area' },
        'area stacked':            { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top) */                      gt.pgettext('chart-type', 'Area (stacked)'),   type: 'area2d',    stacking: 'stacked',        cjs: 'stackedArea' },
        'area percentStacked':     { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top with percent scaling) */ gt.pgettext('chart-type', 'Area (percent)'),   type: 'area2d',    stacking: 'percentStacked', cjs: 'stackedArea100' }
    };

    // set missing Boolean properties explicitly to false
    _.each(CHART_TYPE_STYLES, function (CHART_TYPE) {
        CHART_TYPE.curved = CHART_TYPE.curved || false;
        CHART_TYPE.varyColors = CHART_TYPE.varyColors || false;
    });

    // geometries of predefined drawing shapes
    var PRESET_GEOMETRIES = null;

    // description labels for all custom shape types, grouped by shape categories
    var SHAPE_LABELS_MAP = {

        line: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Lines'),
            shapes: {
                line: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Line'),
                straightConnector1: {
                    tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Line arrow'),
                    lineAttrs: { headEndType: 'arrow' }
                },
                straightConnector1_2: {
                    shapeID: 'straightConnector1', tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Line arrow: double'),
                    lineAttrs: { headEndType: 'arrow', tailEndType: 'arrow' }
                },

                bentConnector3: 'Connector: elbow',
                bentConnector3_1: {
                    shapeID: 'bentConnector3',
                    tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Connector: elbow arrow'),
                    lineAttrs: { headEndType: 'arrow' },
                    format: 'scale-vertical'
                },
                bentConnector3_2: {
                    shapeID: 'bentConnector3',
                    tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Connector: elbow double-arrow'),
                    lineAttrs: { headEndType: 'arrow', tailEndType: 'arrow' },
                    format: 'scale-vertical-double'
                },

                curvedConnector3: 'Connector: curved',
                curvedConnector3_1: {
                    shapeID: 'curvedConnector3',
                    tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Connector: curved arrow'),
                    lineAttrs: { headEndType: 'arrow' }, format: 'scale-vertical'
                },
                curvedConnector3_2: {
                    shapeID: 'curvedConnector3', tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Connector: curved double-arrow'),
                    lineAttrs: { headEndType: 'arrow', tailEndType: 'arrow' },
                    format: 'scale-vertical-double'
                }
            }
        },

        shape: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Basic shapes'),
            shapes: {
                ellipse: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ellipse'),
                triangle: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Isosceles triangle'),
                rtTriangle: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Right triangle'),
                parallelogram: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Parallelogram'),
                trapezoid: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Trapezoid'),
                diamond: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Diamond'),
                pentagon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Pentagon'),
                hexagon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Hexagon'),
                heptagon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Heptagon'),
                octagon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Octagon'),
                decagon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Decagon'),
                dodecagon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Dodecagon'),
                pie: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Circle pie'),
                chord: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Circle segment'),
                teardrop: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Teardrop'),
                frame: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Frame'),
                halfFrame: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Half frame'),
                corner: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Corner'),
                diagStripe: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Diagonal stripe'),
                plus: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Plus'),
                plaque: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Plaque'),
                can: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Cylinder'),
                cube: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Cube'),
                funnel: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Funnel'), // no excel
                bevel: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle bevel'),
                donut: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ring'),
                noSmoking: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'No symbol'),
                blockArc: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Block arc'),
                foldedCorner: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Folded corner'),
                smileyFace: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Smiley face'),
                heart: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Heart'),
                lightningBolt: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Lightning bolt'),
                sun: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Sun'),
                moon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Moon'),
                cloud: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Cloud'),
                arc: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arc'),
                bracketPair: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Double bracket'),
                bracePair: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Double brace'),
                leftBracket: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Left bracket'), format: 'landscape' },
                rightBracket: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Right bracket'), format: 'landscape' },
                leftBrace: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Left brace'), format: 'landscape' },
                rightBrace: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Right brace'), format: 'landscape' },
                // no excel
                squareTabs: /*#. predefiend drawing shape type (4 small squares, one per corner, like photo corners) */ gt.pgettext('shape', 'Square tabs'),
                cornerTabs: /*#. predefiend drawing shape type (4 triangles, one per corner, like photo corners with straight borders inside) */ gt.pgettext('shape', 'Corner tabs'),
                plaqueTabs: /*#. predefiend drawing shape type (4 quarters of a circle, one per corner, like photo corners with rounded borders inside) */ gt.pgettext('shape', 'Plaque tabs'),
                pieWedge: /*#. predefiend drawing shape type (one quarter of a circle) */ gt.pgettext('shape', 'Quarter pie'),
                chartPlus: /*#. predefiend drawing shape type ("plus" sign) */ gt.pgettext('shape', 'Plus sign'),
                chartX: /*#. predefiend drawing shape type (X letter, "times" sign) */ gt.pgettext('shape', 'Times sign')
            }
        },

        rect: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Rectangles'),
            shapes: {
                rect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle'),
                roundRect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: rounded corners'),
                snip1Rect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: single corner snipped'),
                snip2SameRect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: top corners snipped'),
                snip2DiagRect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: diagonal corners snipped'),
                snipRoundRect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: top corners one rounded one snipped'),
                round1Rect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: single corner rounded'),
                round2SameRect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: top corners rounded'),
                round2DiagRect: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Rectangle: diagonal corners rounded')
            }
        },

        arrow: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Block arrows'),
            shapes: {
                rightArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: right'),
                leftArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: left'),
                upArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: up'),
                downArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: down'),
                leftRightArrow: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: left-right'), format: 'portrait' },
                upDownArrow: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: up-down'), format: 'landscape' },
                quadArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: quad'),
                leftRightUpArrow: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: left-right-up'), format: 'portrait' },
                bentArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: right-bent'),
                uturnArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: u-turn'),
                leftUpArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: left-up'),
                bentUpArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: up-bent'),
                curvedRightArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: curved right'),
                curvedLeftArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: curved left'),
                curvedUpArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: curved up'),
                curvedDownArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: curved down'),
                leftCircularArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: circular right'),
                circularArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: circualr left'),
                stripedRightArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: striped right'),
                notchedRightArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: notched right'),
                homePlate: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: pentagon'),
                chevron: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: chevron'),
                swooshArrow: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Arrow: swoosh')
            }
        },

        flowchart: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Flow charts'),
            shapes: {
                flowChartProcess: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: process'),
                flowChartAlternateProcess: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: alternate process'),
                flowChartDecision: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: decision'),
                flowChartInputOutput: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: data'),
                flowChartPredefinedProcess: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: predefined process'),
                flowChartInternalStorage: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: internal storage'),
                flowChartDocument: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: document'),
                flowChartMultidocument: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: multidocument'),
                flowChartTerminator: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: terminator'),
                flowChartPreparation: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: preparation'),
                flowChartManualInput: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: manual input'),
                flowChartManualOperation: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: manual operation'),
                flowChartConnector: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: connector'),
                flowChartOffpageConnector: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: off-page connector'),
                flowChartPunchedCard: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: Card'),
                flowChartPunchedTape: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: punched tape'),
                flowChartSummingJunction: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: summing junction'),
                flowChartOr: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: or'),
                flowChartCollate: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: collate'),
                flowChartSort: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: sort'),
                flowChartExtract: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: extract'),
                flowChartMerge: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: merge'),
                flowChartOnlineStorage: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: stored data'),
                flowChartDelay: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: delay'),
                flowChartMagneticTape: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: magnetic tape'),
                flowChartMagneticDisk: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: magnetic disk'),
                flowChartMagneticDrum: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: magnetic drum'),
                flowChartDisplay: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: display'),
                flowChartOfflineStorage: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Flowchart: offline storage')
            }
        },

        starbanner: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Stars and banners'),
            shapes: {
                irregularSeal1: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Explosion: 12 points'),
                irregularSeal2: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Explosion: 14 points'),
                star4: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 4 points'),
                star5: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 5 points'),
                star6: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 6 points'),
                star7: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 7 points'),
                star8: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 8 points'),
                star10: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 10 points'),
                star12: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 12 points'),
                star16: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 16 points'),
                star24: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 24 points'),
                star32: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Star: 32 points'),
                ribbon2: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ribbon: tilted up'),
                ribbon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ribbon: tilted down'),
                ellipseRibbon2: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ribbon: curved and tilted up'),
                ellipseRibbon: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ribbon: curved and tilted down'),
                leftRightRibbon: { tooltip: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Ribbon: left-right arrow'), format: 'portrait' },
                verticalScroll: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Scroll: vertical'),
                horizontalScroll: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Scroll: horizontal'),
                wave: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Wave'),
                doubleWave: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Double wave')
            }
        },

        math: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Equation'),
            shapes: {
                mathPlus: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Plus sign'),
                mathMinus: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Minus sign'),
                mathMultiply: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Multiplication sign'),
                mathDivide: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Division sign'),
                mathEqual: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Equal'),
                mathNotEqual: /*#. predefiend drawing shape type */ gt.pgettext('shape', 'Not equal')
            }
        },

        callouts: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Callouts'),
            shapes: {}
        }
    };

    // shape category descriptors for all custom shapes, mapped by shape identifier
    var SHAPE_CATEGORIES_MAP = {};

    // static class DrawingLabels =============================================

    /**
     * Provides generic caption texts for control groups, menu headers, and
     * other GUI elements in a single map for convenience.
     */
    var DrawingLabels = {};

    // drawing types ----------------------------------------------------------

    /**
     * Returns an appropriate CSS icon class for the passed drawing type.
     *
     * @param {String} type
     *  The type identifier of a drawing object.
     *
     * @returns {String}
     *  A CSS icon class for the passed drawing type.
     */
    DrawingLabels.getDrawingTypeIcon = function (type) {
        return (DRAWING_TYPE_INFOS[type] || DRAWING_TYPE_DEFAULT_INFO).icon;
    };

    /**
     * Returns an appropriate text label for the passed drawing type.
     *
     * @param {String} type
     *  The type identifier of a drawing object.
     *
     * @returns {String}
     *  A text label for the passed drawing type.
     */
    DrawingLabels.getDrawingTypeLabel = function (type) {
        return (DRAWING_TYPE_INFOS[type] || DRAWING_TYPE_DEFAULT_INFO).label;
    };

    // constant labels and tooltips -------------------------------------------

    /**
     * Standard options for the 'Delete drawing' label.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_LABEL = /*#. delete a drawing object from the document */ gt.pgettext('drawing', 'Delete');

    /**
     * Standard options for the 'Delete drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Delete the drawing object');

    /**
     * Standard options for the 'Insert drawing' label.
     *
     * @constant
     */
    DrawingLabels.INSERT_DRAWING_LABEL = DrawingLabels.getDrawingTypeLabel('image');

    /**
     * Standard options for the 'Insert drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.INSERT_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Insert an image');

    /**
     * Standard options for the 'Group drawing' label.
     *
     * @constant
     */
    DrawingLabels.GROUP_DRAWING_LABEL = /*#. group the selected drawings */ gt.pgettext('drawing', 'Group');

    /**
     * Standard options for the 'Group drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.GROUP_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Group the selected drawings');

    /**
     * Standard options for the 'Ungroup drawing' label.
     *
     * @constant
     */
    DrawingLabels.UNGROUP_DRAWING_LABEL = /*#. ungroup the selected drawing(s) */ gt.pgettext('drawing', 'Ungroup');

    /**
     * Standard options for the 'Ungroup drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.UNGROUP_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Ungroup the selected drawings');

// constants for controls -------------------------------------------------

    /**
     * Standard options for the 'Delete drawing' button.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_BUTTON_OPTIONS = {
        icon: 'fa-trash-o',
        tooltip: DrawingLabels.DELETE_DRAWING_TOOLTIP
    };

    /**
     * Standard options for insert image.
     *
     * @constant
     */
    DrawingLabels.INSERT_IMAGE_OPTIONS = {
        label: DrawingLabels.INSERT_DRAWING_LABEL,
        tooltip: DrawingLabels.INSERT_DRAWING_TOOLTIP
    };

    /**
     * Standard options for insert textframe.
     *
     * @constant
     */
    DrawingLabels.INSERT_TEXTFRAME_OPTIONS = {
        label: DrawingLabels.getDrawingTypeLabel('textframe'),
        tooltip: gt.pgettext('drawing', 'Insert a text frame')
    };

    /**
     * Standard options for the 'Insert textframe' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_TEXTFRAME_BUTTON_OPTIONS = {
        icon: DrawingLabels.getDrawingTypeIcon('textframe'),
        label: DrawingLabels.INSERT_TEXTFRAME_OPTIONS.label,
        tooltip: DrawingLabels.INSERT_TEXTFRAME_OPTIONS.tooltip,
        smallerVersion: {
            css: { width: 35 },
            hideLabel: true
        }
    };

    /**
     * Standard options for insert comment.
     *
     * @constant
     */
    DrawingLabels.INSERT_COMMENT_OPTIONS = {
        label: gt('Comment'),
        tooltip: /*#. insert a comment into the text */ gt('Insert a comment')
    };

    /**
     * Standard options for a 'Insert chart' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_CHART_OPTIONS = {
        label: DrawingLabels.getDrawingTypeLabel('chart'),
        tooltip: gt.pgettext('drawing', 'Insert a chart')
    };

    /**
     * Standard options for the 'Insert chart' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_CHART_BUTTON_OPTIONS = {
        width: null,
        icon: DrawingLabels.getDrawingTypeIcon('chart'),
        label: DrawingLabels.INSERT_CHART_OPTIONS.label,
        tooltip: DrawingLabels.INSERT_CHART_OPTIONS.tooltip,
        updateCaptionMode: 'none',
        value: 'column standard'
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_DATA_POINTS_BUTTON_OPTIONS = {
        label: //#. menu title: options to format data points (bars, lines, pie segments) in chart objects
            gt.pgettext('chart-format', 'Data points')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS = {
        label: //#. check box label: show/hide small text labels next to the single data points in charts objects
            gt.pgettext('chart-format', 'Show data point labels')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS = {
        label: //#. check box label: if active, a bar/pie chart will have different colors for each data point (default is the same color for all points)
            gt.pgettext('chart-format', 'Vary colors by point')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_MARKER_ONLY_BUTTON_OPTIONS = {
        label: //#. check box label: for line & scatter charts, show/hide lines between points (markers)
            gt.pgettext('chart-format', 'Markers only')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_LABELS_BUTTON_OPTIONS = {
        label: //#. button label to open a menu to modify chart objects
            gt.pgettext('chart-format', 'Labels'),
        tooltip: //#. tooltip: menu to modify chart objects
            gt.pgettext('chart-format', 'Settings for labels and axes'),
        toggle: true,
        haspopup: true
    };

    /**
     * A map with predefined border styles (line width, line style) supported by OOXML.
     *
     * @constant
     */
    DrawingLabels.BORDER_OOXML_PRESET_STYLES = [
        { value: 'none:none',         label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'No line'),                           icon: { style: 'none',       width: -1 } },
        { value: 'solid:hair',        label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'Hair line'),                         icon: { style: 'solid',      width: 0 } },
        { value: 'solid:thin',        label: /*#. border line style (width 1 pixel, solid) */ gt.pgettext('borders', 'Thin line'),                            icon: { style: 'solid',      width: 1 } },
        { value: 'dashed:thin',       label: /*#. border line style (width 1 pixel, dashed) */ gt.pgettext('borders', 'Dashed thin line'),                    icon: { style: 'dashed',     width: 1 } },
        { value: 'dotted:thin',       label: /*#. border line style (width 1 pixel, dotted) */ gt.pgettext('borders', 'Dotted thin line'),                    icon: { style: 'dotted',     width: 1 } },
        { value: 'dashDot:thin',      label: /*#. border line style (width 1 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash thin line'),            icon: { style: 'dashDot',    width: 1 } },
        { value: 'dashDotDot:thin',   label: /*#. border line style (width 1 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash thin line'),   icon: { style: 'dashDotDot', width: 1 } },
        { value: 'solid:medium',      label: /*#. border line style (width 2 pixels, solid) */ gt.pgettext('borders', 'Medium line'),                         icon: { style: 'solid',      width: 2 } },
        { value: 'dashed:medium',     label: /*#. border line style (width 2 pixels, dashed) */ gt.pgettext('borders', 'Dashed medium line'),                 icon: { style: 'dashed',     width: 2 } },
        { value: 'dotted:medium',     label: /*#. border line style (width 2 pixel, dotted) */ gt.pgettext('borders', 'Dotted medium line'),                  icon: { style: 'dotted',     width: 2 } },
        { value: 'dashDot:medium',    label: /*#. border line style (width 2 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash medium line'),          icon: { style: 'dashDot',    width: 2 } },
        { value: 'dashDotDot:medium', label: /*#. border line style (width 2 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash medium line'), icon: { style: 'dashDotDot', width: 2 } },
        { value: 'solid:thick',       label: /*#. border line style (width 3 pixels, solid) */ gt.pgettext('borders', 'Thick line'),                          icon: { style: 'solid',      width: 3 } }
    ];

    /**
     * A map with predefined border styles (line width, line style) supported
     * by ODF.
     *
     * @constant
     */
    DrawingLabels.BORDER_ODF_PRESET_STYLES = [
        { value: 'none:none',         label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'No line'),                           icon: { style: 'none',       width: -1 } },
        { value: 'solid:hair',        label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'Hair line'),                         icon: { style: 'solid',      width: 0 } },
        { value: 'solid:thin',        label: /*#. border line style (width 1 pixel, solid) */ gt.pgettext('borders', 'Thin line'),                            icon: { style: 'solid',      width: 1 } },
        { value: 'dashed:thin',       label: /*#. border line style (width 1 pixel, dashed) */ gt.pgettext('borders', 'Dashed thin line'),                    icon: { style: 'dashed',     width: 1 } },
        { value: 'dotted:thin',       label: /*#. border line style (width 1 pixel, dotted) */ gt.pgettext('borders', 'Dotted thin line'),                    icon: { style: 'dotted',     width: 1 } },
        { value: 'solid:medium',      label: /*#. border line style (width 2 pixels, solid) */ gt.pgettext('borders', 'Medium line'),                         icon: { style: 'solid',      width: 2 } },
        { value: 'dashed:medium',     label: /*#. border line style (width 2 pixels, dashed) */ gt.pgettext('borders', 'Dashed medium line'),                 icon: { style: 'dashed',     width: 2 } },
        { value: 'dotted:medium',     label: /*#. border line style (width 2 pixel, dotted) */ gt.pgettext('borders', 'Dotted medium line'),                  icon: { style: 'dotted',     width: 2 } },
        { value: 'solid:thick',       label: /*#. border line style (width 3 pixels, solid) */ gt.pgettext('borders', 'Thick line'),                          icon: { style: 'solid',      width: 3 } }
    ];

    /**
     * A map with chart type descriptors for all supported chart type. Each
     * map element contains the following properties:
     * - {Number} cat
     *      The type category, used for grouping in the type picker control.
     * - {String} title
     *      The localized name of the chart type.
     * - {String} type
     *      The base chart type, as used in document operations.
     * - {String} stacking
     *      The stacking mode: 'standard', 'stacked', or 'percentStacked'.
     * - {String} cjs
     *      The chart type identifier for the CanvasJS library.
     * - {Boolean} curved
     *      Whether the lines in the chart area are smoothed.
     * - {Boolean} varyColors
     *      Whether the data points of a single series show different colors.
     *
     * @constant
     */
    DrawingLabels.CHART_TYPE_STYLES = CHART_TYPE_STYLES;

    // custom shapes ----------------------------------------------------------

    /**
     * A map with tooltips and other settings for all custom shapes, grouped by
     * category names.
     *
     * @constant
     */
    DrawingLabels.SHAPE_LABELS_MAP = SHAPE_LABELS_MAP;

    /**
     * Return the shape by id.
     *
     * @param {String} shapeId
     *  to get the shape
     *
     * @returns {Object|Null}
     *      {Object} shapeObj.shape The JSON to draw the shape
     *      {String} shapeObj.id The id/name of the shape
     *      {String} [shapeObj.tooltip] The tooltip string for the shape
     *      {String} [shapeObj.format] The format of the shape menu icon portrait/landscape, default is square.
     *      {Object} [shapeObj.lineAttrs] Optional line attributes.
     */
    DrawingLabels.getPresetShape = function (shapeId) {

        var categoryDef = SHAPE_CATEGORIES_MAP[shapeId];
        if (!categoryDef) { return null; }

        var shapeDef = categoryDef.shapes[shapeId];
        shapeId = shapeDef.shapeID || shapeId;
        var shapeGeometry = PRESET_GEOMETRIES[shapeId];
        if (!shapeGeometry) { return null; }

        var presetShape = { shape: shapeGeometry, id: shapeId };
        if (_.isObject(shapeDef)) {
            _.extend(presetShape, shapeDef);
        } else {
            presetShape.tooltip = shapeDef;
        }
        return presetShape;
    };

    // exports ================================================================

    return IO.loadJSON('io.ox/office/drawinglayer/view/presetGeometries').then(function (geometries) {

        PRESET_GEOMETRIES = geometries;

        _.each(SHAPE_LABELS_MAP, function (categoryDef) {
            _.each(categoryDef.shapes, function (shapeDef, shapeId, shapeMap) {
                var linkShapeId = (_.isObject(shapeDef) && shapeDef.shapeID) ? shapeDef.shapeID : null;
                if (PRESET_GEOMETRIES[shapeId] || (linkShapeId && PRESET_GEOMETRIES[linkShapeId])) {
                    SHAPE_CATEGORIES_MAP[shapeId] = categoryDef;
                } else {
                    delete shapeMap[shapeId]; // remove shapes without geometry definition
                }
            });
        });

        return DrawingLabels;
    });

});
