/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingcontrols', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/controls',
    'io.ox/office/baseframework/view/control/compoundbutton',
    'io.ox/office/baseframework/view/control/compoundsplitbutton',
    'io.ox/office/drawinglayer/view/control/shapetypepicker',
    'io.ox/office/drawinglayer/view/control/charttypepicker',
    'io.ox/office/drawinglayer/view/control/chartcolorsetpicker',
    'io.ox/office/drawinglayer/view/control/chartstylesetpicker',
    'io.ox/office/drawinglayer/view/control/chartlegendpicker',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Controls, CompoundButton, CompoundSplitButton, ShapeTypePicker, ChartTypePicker, ChartColorSetPicker, ChartStyleSetPicker, ChartLegendPicker, DrawingLabels, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;

    // private global functions ===============================================

    /**
     * Inserts all buttons needed to change the Z order of drawing objects to
     * the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingOrderButtons(control) {

        control.addGroup('drawing/order', new Button({
            value: 'front',
            icon: 'docs-arrange-to-front',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Bring to front'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Bring object to the front')
        }));

        control.addGroup('drawing/order', new Button({
            value: 'forward',
            icon: 'docs-arrange-forward',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Bring forward'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Move object up one placement to the front')
        }));

        control.addGroup('drawing/order', new Button({
            value: 'backward',
            icon: 'docs-arrange-backward',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send backward'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send object one placement to the back')
        }));

        control.addGroup('drawing/order', new Button({
            value: 'back',
            icon: 'docs-arrange-to-back',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send to back'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send object to the back')
        }));
    }

    /**
     * Inserts all buttons needed to change the rotation of drawing objects to
     * the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingRotateButtons(control) {

        control.addGroup('drawing/rotate', new Button({
            value: 'cw90',
            icon: 'docs-drawing-rotate-right',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate right 90\xb0'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate right 90 degrees')
        }));

        control.addGroup('drawing/rotate', new Button({
            value: 'ccw90',
            icon: 'docs-drawing-rotate-left',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate left 90\xb0'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate left 90 degrees')
        }));

        control.addGroup('drawing/rotate', new Button({
            value: 'flipV',
            icon: 'docs-drawing-flip-v',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip vertical'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip object vertical')
        }));

        control.addGroup('drawing/rotate', new Button({
            value: 'flipH',
            icon: 'docs-drawing-flip-h',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip horizontal'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip object horizontal')
        }));
    }

    /**
     * Inserts all buttons needed to change the horizontal alignment of drawing
     * objects to the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingHorAlignButtons(control) {

        control.addGroup('drawing/align', new Button({
            value: 'left',
            icon: 'docs-drawing-h-align-left',
            label: /*#. horizontal alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align left')
        }));

        control.addGroup('drawing/align', new Button({
            value: 'center',
            icon: 'docs-drawing-h-align-center',
            label: /*#. horizontal alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align center')
        }));

        control.addGroup('drawing/align', new Button({
            value: 'right',
            icon: 'docs-drawing-h-align-right',
            label: /*#. horizontal alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align right')
        }));
    }

    /**
     * Inserts all buttons needed to change the vertical alignment of drawing
     * objects to the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingVertAlignButtons(control) {

        control.addGroup('drawing/align', new Button({
            value: 'top',
            icon: 'docs-drawing-v-align-top',
            label: /*#. vertical alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align top')
        }));

        control.addGroup('drawing/align', new Button({
            value: 'middle',
            icon: 'docs-drawing-v-align-middle',
            label: /*#. vertical alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align middle')
        }));

        control.addGroup('drawing/align', new Button({
            value: 'bottom',
            icon: 'docs-drawing-v-align-bottom',
            label: /*#. vertical alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align bottom')
        }));
    }

    /**
     * Inserts all buttons needed to distribute multiple drawing objects to the
     * passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingDistributionButtons(control) {

        control.addGroup('drawing/distribute', new Button({
            value: 'horzSlide',
            icon: 'docs-distribute-slide-h',
            label: gt.pgettext('drawing-pos', 'Distribute horizontally on slide'),
            tooltip: gt('Distribute selected objects horizontally on slide')
        }));

        control.addGroup('drawing/distribute', new Button({
            value: 'vertSlide',
            icon: 'docs-distribute-slide-v',
            label: gt.pgettext('drawing-pos', 'Distribute vertically on slide'),
            tooltip: gt('Distribute selected objects vertically on slide')
        }));

        control.addGroup('drawing/distributeamong', new Button({
            value: 'horzSelection',
            icon: 'docs-distribute-objects-h',
            label: gt.pgettext('drawing-pos', 'Distribute horizontally among objects'),
            tooltip: gt('Distribute selected objects horizontally among themselves')
        }));

        control.addGroup('drawing/distributeamong', new Button({
            value: 'vertSelection',
            icon: 'docs-distribute-objects-v',
            label: gt.pgettext('drawing-pos', 'Distribute vertically among objects'),
            tooltip: gt('Distribute selected objects vertically among themselves')
        }));
    }

    // static class DrawingControls ===========================================

    /**
     * Provides different classes for GUI form controls for drawing objects in
     * a single map for convenience.
     */
    var DrawingControls = {
        ChartTypePicker: ChartTypePicker,
        ChartColorSetPicker: ChartColorSetPicker,
        ChartStyleSetPicker: ChartStyleSetPicker,
        ChartLegendPicker: ChartLegendPicker,
        ShapeTypePicker: ShapeTypePicker
    };

    // class DrawingOrderPicker ===============================================

    /**
     * Drop-down menu with commands to change the Z order of drawing objects.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {EditView} docView
     *  The document view instance owning this control.
     */
    DrawingControls.DrawingOrderPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            //#. drop-down menu to change the Z order of drawing objects
            label: gt.pgettext('drawing-pos', 'Reorder'),
            icon: 'docs-z-order',
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        addDrawingOrderButtons(this);

    } });

    // class DrawingArrangementPicker =========================================

    /**
     * Drop-down menu with commands to change the Z order of drawing objects,
     * and additional rotation and flipping commands.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {EditView} docView
     *  The document view instance owning this control.
     */
    DrawingControls.DrawingArrangementPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            //#. drop-down menu to change the Z order and rotation of drawing objects
            label: gt.pgettext('drawing-pos', 'Arrange'),
            icon: 'docs-z-order',
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        this.addSectionLabel(gt('Reorder'));
        addDrawingOrderButtons(this);

        this.addSectionLabel(gt('Rotate'));
        addDrawingRotateButtons(this);

    } }); // class DrawingArrangementPicker

    // class DrawingAlignmentPicker ===========================================

    /**
     * Drop-down menu with commands to change the alignment of drawing objects
     * relative to each other.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {EditView} docView
     *  The document view instance owning this control.
     */
    DrawingControls.DrawingAlignmentPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            //#. drop-down menu to change the alignment of drawing objects
            label: gt.pgettext('drawing-pos', 'Align'),
            icon: 'docs-drawing-h-align-left',
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        addDrawingHorAlignButtons(this);
        this.addSeparator();
        addDrawingVertAlignButtons(this);
        this.addSeparator();
        addDrawingDistributionButtons(this);

    } }); // class DrawingAlignmentPicker

    // class ImagePicker ======================================================

    DrawingControls.ImagePicker = CompoundSplitButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundSplitButton.call(this, docView, Utils.extendOptions({
            icon: 'docs-image-insert',
            label: DrawingLabels.INSERT_DRAWING_LABEL,
            tooltip: DrawingLabels.INSERT_DRAWING_TOOLTIP,
            smallerVersion: {
                css: { width: 35 },
                hideLabel: true
            },
            value: 'drive'
        }, initOptions));

        var itemKey = Utils.getStringOption(initOptions, 'itemKey', 'image/insert/dialog');
        this.addGroup(itemKey, new Button({ value: 'local', label: /*#. Label for insert image from the local file system. */ gt('Local file') }));
        this.addGroup(itemKey, new Button({ value: 'drive', label: /*#. Label for insert image from drive. Please do not translate the productname 'Drive' */ gt('From Drive') }));
        this.addGroup(itemKey, new Button({ value: 'url', label:  /*#. Label for insert image from an url. */ gt('From URL') }));

    } }); // class ImagePicker

    // class CompactImagePicker ===============================================

    /**
     * Control to pick image from local file or from drive.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    DrawingControls.CompactImagePicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        CompoundButton.call(this, docView, Utils.extendOptions({
            icon: 'docs-image-insert',
            label: DrawingLabels.INSERT_DRAWING_LABEL,
            tooltip: DrawingLabels.INSERT_DRAWING_TOOLTIP,
            smallerVersion: {
                hideLabel: true
            }
        }, initOptions));

        var itemKey = Utils.getStringOption(initOptions, 'itemKey', 'slide/setbackgroundimage');
        this.addGroup(itemKey, new Button({ value: 'local', label: /*#. Label for insert image from the local file system. */ gt('Local file') }));
        this.addGroup(itemKey, new Button({ value: 'drive', label: /*#. Label for insert image from drive. Please do not translate the productname 'Drive' */ gt('From Drive') }));

    } }); // class CompactImagePicker

    // exports ================================================================

    return DrawingControls;

});
