/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author York Richter <york.richtert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/shapetypepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/tk/popup/listmenu',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/view/drawingframe',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Forms, RadioList, ListMenu, Labels, DrawingFrame, gt) {

    'use strict';

    var ICON_SIZE = Utils.SMALL_DEVICE ? 20 : 25;

    var ICON_CONTAINER_SIZE = Utils.SMALL_DEVICE ? 35 : 40;

    var SHAPES_PER_ROW = Math.min(12, (Utils.getInnerWindowWidth() / ICON_CONTAINER_SIZE) - 1);

    // class ShapeTypePicker ==================================================

    /**
     * @constructor
     *
     * @extends RadioList
     */
    function ShapeTypePicker(docView, initOptions) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // the application
        var app = docView.getApp();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Insert a shape'),
            label: gt('Shape'),
            icon: Labels.getDrawingTypeIcon('shape'),
            updateCaptionMode: 'none',
            toggle: true,
            itemDesign: 'grid',
            gridColumns: SHAPES_PER_ROW,
            smallerVersion: { hideLabel: true },
            mruSize: SHAPES_PER_ROW,
            mruSettingsKey: 'shapetypepicker/mostrecentlyused',
            mruLabel: gt('Recently used')
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function createItemHandler(event, buttonNode) {
            buttonNode.addClass('mini-caption').css({
                minWidth: ICON_CONTAINER_SIZE,
                maxWidth: ICON_CONTAINER_SIZE,
                maxHeight: ICON_CONTAINER_SIZE,
                padding: 0,
                lineHeight: 'normal'
            }).append('<img style="margin:2px;">');
        }

        /**
         * called initial and on theme-change
         * creates CanvasJS structure
         * and draws deferred all types in the chain to images
         */
        var updatePreviewShapes = this.createDebouncedMethod($.noop, function () {
            var boundRect = { width: ICON_SIZE, height: ICON_SIZE, left: 0, top: 0 };
            var fillAttrs = { type: 'none', color: { type: 'preset', value: 'white' } };
            var lineAttrs = { color: { type: 'preset', value: 'black' }, width: 25 };

            var canvas = DrawingFrame.initializeCanvas(null, boundRect, 1);

            self.iterateArraySliced(self.getOptionButtons(), function (buttonNode) {
                var id = Forms.getButtonValue(buttonNode);
                var preset = Labels.getPresetShape(id);
                var img = $(buttonNode).find('>img');
                var shapeLineAttrs = _.extend(_.clone(lineAttrs), preset.lineAttrs);

                if ('headEndType' in shapeLineAttrs) {
                    shapeLineAttrs.headEndLength = 'small';
                    shapeLineAttrs.headEndWidth = 'small';
                }
                if ('tailEndType' in shapeLineAttrs) {
                    shapeLineAttrs.tailEndLength = 'small';
                    shapeLineAttrs.tailEndWidth = 'small';
                }

                var rect = (function () {
                    switch (preset.format) {
                        case 'portrait':
                            img.css({ marginTop: 10 });
                            return { width: ICON_SIZE, height: ICON_SIZE / 2, left: 0, top: 0 };
                        case 'landscape':
                            img.css({ marginLeft: 10 });
                            return { width: ICON_SIZE / 2, height: ICON_SIZE, left: 0, top: 0 };
                        case 'scale-vertical':
                            return { width: ICON_SIZE, height: ICON_SIZE - 2, left: 0, top: 2 };
                        case 'scale-vertical-double':
                            return { width: ICON_SIZE, height: ICON_SIZE - 4, left: 0, top: 2 };
                        default:
                            return boundRect;
                    }
                }());

                canvas.render(function (context /*, width, height */) {
                    context.clearRect(-100, -100, 400, 400);
                    DrawingFrame.drawShape(docModel, context, rect, preset.shape, fillAttrs, shapeLineAttrs, {});
                });

                if (preset.format === 'scale-vertical') {
                    canvas.height = ICON_SIZE + 2;
                } else if (preset.format === 'scale-vertical-double') {
                    canvas.height = ICON_SIZE + 4;
                }

                // update the image element
                img.attr({ src: canvas.getDataURL() });

            }, { infoString: 'ShapeTypePicker: updatePreviewShapes 1', app: app }).done(function () {
                canvas.destroy();
            });
        }, { delay: 50,  infoString: 'ShapeTypePicker: updatePreviewShapes 2', app: app });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('shape-type-picker');

        self.getMenu().on('create:item', createItemHandler);

        // add recently used groups
        self.createMenuSection(ListMenu.MRU_SECTION_ID, { label: gt('Recently used') });

        // create sections
        _.each(Labels.SHAPE_LABELS_MAP, function (categoryDef, categoryId) {
            self.createMenuSection(categoryId, { label: categoryDef.title });
            // create shape buttons
            _.each(categoryDef.shapes, function (shapeDef, shapeId) {
                var tooltip = _.isObject(shapeDef) ? (shapeDef.tooltip || '') : shapeDef;
                self.createOptionButton(shapeId, { section: categoryId, tooltip: tooltip });
            });
        });

        this.registerUpdateHandler(function (newState, lastState) {
            if (lastState !== newState) {
                //add the shape to the recently used list if it's not contained
                if (Labels.getPresetShape(newState)) {
                    updatePreviewShapes();
                }
                Forms.selectNodes(self.getNode().find(Forms.BUTTON_SELECTOR), newState !== false);
            }
        });

        // lazy initialization of the charts before first usage of the menu
        this.getMenu().one('popup:beforeshow', updatePreviewShapes);

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = initOptions = docModel = docView = app = null;
        });

    } // class ShapeTypePicker

    // exports ================================================================

    return RadioList.extend({ constructor: ShapeTypePicker });

});
