/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/chartformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, BaseObject, gt) {

    'use strict';

    //'io.ox/office/spreadsheet/model/numberformatter'
    // null date (corresponding to cell value zero) TODO: use null date from document settings

    // class ChartFormatter ===================================================

    function ChartFormatter(docModel, chartModel) {

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        function isAnyDateTime(format, value) {
            if (format && format.isAnyDateTime() && isFinite(value)) {
                return true;
            }
        }

        function newInfoHolder(type, label) {
            return {
                max: Number.NEGATIVE_INFINITY,
                date: false,
                type: type,
                label: label,
                unclean: false,
                exp: 0
            };
        }

        function handleData(info, source, dataPoint, nullAsZero) {
            var realNr = source.value;
            var display = source.display;
            var format = source.format;

            if (format && !info.format) {
                if (isAnyDateTime(format, realNr)) {
                    info.date = format;
                }
                info.format = format;
            }

            var label = null;
            var res = null;
            if (info.date) {
                res = realNr;
                info.max = Math.max(info.max, Math.abs(realNr));
                label = display;

            } else if (realNr === undefined || realNr === null || !isFinite(realNr)) {
                info.unclean = true;
                res = nullAsZero ? 0 : undefined;
                label = display;
            } else if (info.format) {
                info.max = Math.max(info.max, Math.abs(realNr));
                res = realNr;
                label = docModel.getNumberFormatter().formatValue(info.format, realNr);
            } else {
                info.max = Math.max(info.max, Math.abs(realNr));
                res = realNr;
                label = docModel.getNumberFormatter().formatStandardNumber(realNr, 12);
            }

            dataPoint[info.label] = label;
            dataPoint[info.type] = res;
        }

        function calcMaxExp(info) {
            if (isFinite(info.max)) {
                var norm = Utils.normalizeNumber(info.max);
                if (Math.abs(norm.exp) < 6) {
                    info.exp = 0;
                } else {
                    info.exp = norm.exp;
                }
            } else {
                info.exp = 0;
            }
        }

        function handleMantissa(info, dataPoint) {
            if (info.exp) {
                dataPoint[info.type] = Utils.mant(dataPoint[info.type], info.exp);
            }
        }

        function handleLabelLength(info, dataPoint) {
            var label = dataPoint[info.label];
            if (label && label.length > 30) {
                dataPoint[info.label] = label.substring(0, 27) + '...';
            }
        }

        // public methods -----------------------------------------------------

        this.format = function () {
            var data = chartModel.getModelData();
            var attrs = chartModel.getMergedAttributeSet(true);
            var chartType = attrs.chart.type;
            var stacking = attrs.chart.stacking;
            var xyChart = chartModel.isXYType() || chartType.indexOf('area') === 0;
            var nullAsZero = chartType.indexOf('pie') === -1 && chartType.indexOf('donut') === -1;
            var dateChart = false;

            var x = newInfoHolder('x', 'label');
            var y = newInfoHolder('y', 'name');

            data.linearX = false;

            _.each(data.series, function (dataSeries, seriesIndex) {
                dataSeries.seriesIndex = seriesIndex;
                if (dataSeries.nameHolder) {
                    dataSeries.name = dataSeries.nameHolder.display;
                } else {
                    dataSeries.name =
                    //#. A data series in a chart object (a sequence of multiple data points with a title)
                    //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
                    //#. This label should match the default title for data series in other spreadsheet
                    //#. applications (OpenOffice/LibreOffice Calc, Microsoft Excel).
                    //#, c-format
                    gt('Series %1$d', _.noI18n(seriesIndex + 1));
                }

                var dataPoints = dataSeries.dataPoints;
                _.each(dataPoints, function (dataPoint, n) {
                    dataPoint.x = null;
                    dataPoint.y = null;
                    dataPoint.z = null;

                    handleData(y, dataPoint.valueSource, dataPoint, nullAsZero);

                    if (!n && !seriesIndex && dataPoint.nameSource) {
                        //first of first
                        if (attrs.chart.stacking === 'clustered' || attrs.chart.stacking === 'standard') {
                            var format = dataPoint.nameSource ? dataPoint.nameSource.format : null;
                            if (isAnyDateTime(format, dataPoint.nameSource.value)) {
                                dateChart = true;
                            }
                        }
                    }

                    if (dateChart) {
                        handleData(x, dataPoint.nameSource, dataPoint, nullAsZero);
                    } else if (xyChart && dataPoint.nameSource) {
                        handleData(x, dataPoint.nameSource, dataPoint, nullAsZero);
                    } else {
                        dataPoint[x.type] = n + 1;
                        data.linearX = true;
                    }

                    //ugly part against double formatting!!!
                    if (dataPoint.nameSource) {
                        dataPoint[x.label] = dataPoint.nameSource.display;
                    } else {
                        dataPoint[x.label] = String(dataPoint[x.type]);
                    }

                    if (dataPoint.label) { dataPoint.legendLabel = dataPoint.label; }

                    handleLabelLength(x, dataPoint);
                    handleLabelLength(y, dataPoint);

                    if (dataPoint.sizeSource) {
                        dataPoint.z = Math.abs(dataPoint.sizeSource.value);

                        //fix for funny bug in canvasjs
                        dataPoint.markerBorderColor = 'transparent';
                        dataPoint.markerBorderThickness = 1;

                    } else {
                        dataPoint.z = null;
                    }
                    dataPoint.legendText = dataPoint.legendLabel;
                });

                if (xyChart && x.unclean) {
                    _.each(dataPoints, function (dataPoint, n) {
                        dataPoint.x = n + 1;
                    });
                }
            });

            if (xyChart) {
                calcMaxExp(x);
            }
            calcMaxExp(y);

            chartModel.getAxisModel('x').setFormat(x);
            chartModel.getAxisModel('y').setFormat(y);

            var thresholdY = 0.005;

            _.each(data.series, function (dataSeries) {
                _.each(dataSeries.dataPoints, function (dataPoint) {
                    if (!xyChart) {
                        //workaround that too small values stay visible
                        var rel = dataPoint.y / y.max;
                        if (Math.abs(rel) < thresholdY) {
                            if (rel < 0) {
                                dataPoint.y = -thresholdY * y.max;
                            } else {
                                dataPoint.y = thresholdY * y.max;
                            }
                        }
                    }

                    handleMantissa(y, dataPoint);
                    handleMantissa(x, dataPoint);
                });
            });

            data.data = data.series.slice(0);
            var mustReverse = chartType.indexOf('bar') === 0 && (stacking === 'clustered' || stacking === 'standard');
            if (mustReverse) {
                data.data.reverse();
            }
        };

        this.update = function (seriesIndex, values, titles, names, bubbles) {
            var data = chartModel.getModelData();

            var dataSeries = data.series[seriesIndex];

            if (!dataSeries) {
                //workaround for Bug 46928
                Utils.warn('no dataSeries found for ' + seriesIndex + ' title: ' + titles + ' names: ' + names);
                return;
            }

            // build series title
            if (titles) { dataSeries.nameHolder = titles[0]; }
            if (!seriesIndex) {
                if (dataSeries.nameHolder && data.series.length === 1) {
                    chartModel.setMainName(dataSeries.nameHolder.display);
                } else {
                    chartModel.setMainName(null);
                }
            }
            dataSeries.dataPoints = [];
            _.each(values, function (valueCell, n) {
                var dataPoint = {};
                dataSeries.dataPoints.push(dataPoint);
                dataPoint.nameSource = names[n];    //x
                dataPoint.valueSource = valueCell;  //y
                dataPoint.sizeSource = bubbles[n];  //z
                dataPoint.color = null;
                dataPoint.markerColor = null;
            });
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = chartModel = null;
        });

    } // class ChartFormatter

    // exports ================================================================

    return BaseObject.extend({ constructor: ChartFormatter });

});
