/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/utils/drawingutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border'
], function (Utils, Border) {

    'use strict';

    // static class DrawingUtils ==============================================

    /**
     * Contains common static helper functions to handle drawing objects.
     */
    var DrawingUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * Sorts the passed array of logical drawing positions in-place. The
     * positions of embedded drawings will be kept behind the positions of
     * their parent drawings.
     *
     * @param {Array} positions
     *  (in/out) An array with logical drawing positions. Each drawing position
     *  (each array element) MUST be an array with integers. The passed array
     *  will be sorted in-place.
     *
     * @returns {Array}
     *  A reference to the passed original array that has been sorted in-place.
     */
    DrawingUtils.sortDrawingPositions = function (positions) {
        positions.sort(Utils.compareNumberArrays);
        return positions;
    };

    /**
     * Sorts the passed array of logical drawing positions in-place, and
     * removes the positions of all embedded drawings whose parent drawing
     * positions are contained in the array too.
     *
     * @param {Array} positions
     *  (in/out) An array with logical drawing positions. Each drawing position
     *  (each array element) MUST be an array with integers. The passed array
     *  will be sorted and modified in-place.
     *
     * @returns {Array}
     *  A reference to the passed original array that has been sorted and
     *  modified in-place.
     */
    DrawingUtils.optimizeDrawingPositions = function (positions) {

        var // array index for deleting positions from the array
            index = 0,
            // array elements while deleting positions
            pos1 = null, pos2 = null;

        // first, sort the positions
        DrawingUtils.sortDrawingPositions(positions);

        // remove positions of embedded drawings whose parent position is contained too
        while (index + 1 < positions.length) {
            pos1 = positions[index];
            pos2 = positions[index + 1];
            if ((pos1.length < pos2.length) && _.isEqual(pos1, pos2.slice(0, pos1.length))) {
                positions.splice(index + 1, 1);
            } else {
                index += 1;
            }
        }

        return positions;
    };

    /**
     * Returns the identifier of a predefined border style for the passed
     * drawing attribute set.
     *
     * @param {Object|Null} lineAttrs
     *  An attribute map width line attributes. May be null to indicate missing
     *  line formatting (invisible line).
     *
     * @returns {String}
     *  The identifier of a predefined border style, as string with two tokens
     *  separated by a colon character. The first token represents the line
     *  style (or 'none' for invisible lines), the second token represents the
     *  line width (one of 'hair', 'thin', 'medium', or 'thick').
     */
    DrawingUtils.getPresetBorder = function (lineAttrs) {

        // special value 'none:none', if line attributes are not not supported
        // by the drawing object, or if the line style is set to invisible
        if (!lineAttrs || !lineAttrs.type || (lineAttrs.type === 'none')) { return 'none:none'; }

        return lineAttrs.style + ':' + Border.getPresetForWidth(lineAttrs.width);
    };

    /**
     * Returns some line attributes for the passed predefined border style.
     *
     * @param {String} preset
     *  The identifier of a predefined border style, as returned by the method
     *  DrawingUtils.getPresetBorder().
     *
     * @returns {Object}
     *  An attribute map with line attributes, containing the properties
     *  'type' (set to 'none' or 'solid'), 'style' (line style), and 'width'
     *  (line width in 1/100 of millimeters).
     */
    DrawingUtils.resolvePresetBorder = function (preset) {

        var // the tokens in the passed preset style
            tokens = preset.split(':');

        // return attributes, if the preset has been split into two non-empty strings
        if ((tokens.length === 2) && (tokens[0].length > 0) && (tokens[1].length > 0)) {
            var width = Border.getWidthForPreset(tokens[1]);
            return ((tokens[0] === 'none') || (width === 0)) ? { type: 'none' } : { type: 'solid', style: tokens[0], width: width };
        }

        Utils.warn('DrawingUtils.resolvePresetBorder(): invalid preset style "' + preset + '"');
        return { type: 'none' };
    };

    DrawingUtils.convertDegreeToRadian = function (deg) {
        deg = _.isNumber(deg) ? deg : 0;
        return deg * Math.PI / 180;
    };

    /**
     * Rotate a point.
     * @param {Number} cLeft left position of the rotation center
     * @param {Number} cTop top position of the rotation center
     * @param {Number} left left position of the point to rotate
     * @param {Number} left top position o the point to rotate
     * @param {Number} cos the cosine of the radian to rotate the point
     * @param {Number} sin the sine of the radian to rotate the point
     * @returns {Object} the rotated point.
     *  @param {Number} left the new left position
     *  @param {Number} top the new top position
     */
    DrawingUtils.rotatePoint = function (cLeft, cTop, left, top, cos, sin) {
        var // new left position
            nLeft = (cos * (left - cLeft)) + (sin * (top - cTop)) + cLeft,
            // new right position
            nRight = (cos * (top - cTop)) - (sin * (left - cLeft)) + cTop;
        return { left: nLeft, top: nRight };
    };

    /**
     * Rotate a point with the given angle.
     * See@ rotatePoint(cLeft, cTop, left, top, cos, sin)
     */
    DrawingUtils.rotatePointWithAngle = function (cLeft, cTop, left, top, angle) {
        var radians = this.convertDegreeToRadian(angle),
            cos = Math.cos(radians),
            sin = Math.sin(radians);
        return this.rotatePoint(cLeft, cTop, left, top, cos, sin);
    };

    /**
     * Rotate a drawing with the given angle.
     * @param {type} drawing to rotate
     * @param {type} angle the angle to rotate the drawing.
     * @returns {Object}
     *  @param {Number} top the top position of the drawing
     *  @param {Number} right the right position of the drawing
     *  @param {Number} bottom the bottom position of the drawing
     *  @param {Number} left the left position of the drawing
     *  @param {Number} width the width of the drawing
     *  @param {Number} height the height of the drawing
     *  @param {Object} leftTopPoint the rotated left-top point { left, top }
     */
    DrawingUtils.getRotatedDrawingPoints = function (drawing, angle) {
        var centerLeft = drawing.left + drawing.width / 2,
            centerTop = drawing.top  + drawing.height / 2,
            radians = this.convertDegreeToRadian(angle),
            left = null,
            right = null,
            top = null,
            bottom = null,
            width,
            height,
            cos = Math.cos(radians),
            sin = Math.sin(radians),
            points = [
                //left-top
                this.rotatePoint(centerLeft, centerTop, drawing.left, drawing.top, cos, sin),
                //right-top
                this.rotatePoint(centerLeft, centerTop, drawing.left + drawing.width, drawing.top, cos, sin),
                //right-bottom
                this.rotatePoint(centerLeft, centerTop, drawing.left + drawing.width, drawing.top + drawing.height, cos, sin),
                //left-bottom
                this.rotatePoint(centerLeft, centerTop, drawing.left, drawing.top + drawing.height, cos, sin)
            ];

        points.forEach(function (point) {
            if (left === null || point.left < left) {
                left = point.left;
            }
            if (right === null || point.left > right) {
                right = point.left;
            }
            if (top === null || point.top < top) {
                top = point.top;
            }
            if (bottom === null || point.top > bottom) {
                bottom = point.top;
            }
        });

        width = left > right ? left - right : right - left;
        height = top > bottom ? top - bottom : bottom - top;

        return {
            top: top,
            right: right,
            bottom: bottom,
            left: left,
            width: width,
            height: height,
            leftTopPoint: points[0]
        };
    };

    // exports ================================================================

    return DrawingUtils;

});
