/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/popup/layermenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/baseframework/view/popup/compoundmenu'
], function (Utils, Forms, KeyCodes, Tracking, CompoundMenu) {

    'use strict';

    // class LayerMenu ========================================================

    /**
     * An instance of this class represents a floating modeless pop-up menu
     * with a draggable title bar.
     *
     * @constructor
     *
     * @extends CompoundMenu
     *
     * @param {BaseView} docView
     *  The document view instance containing this menu.
     *
     * @param {String} titleLabel
     *  The title shown on top of the menu.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  CompoundMenu. By default, the bounding box (option 'boundingBox') will
     *  be set to the application window node (the menu node will not cover the
     *  top-level global AppSuite tab bar); and the option 'autoClose' will be
     *  set to false (the browser focus may be located outside the menu node).
     */
    function LayerMenu(docView, titleLabel, initOptions) {

        var // self reference
            self = this,

            // the header container node
            headerNode = $('<div class="header-section">'),

            // the title label inside the header node
            titleNode = $('<div class="title-label">').text(titleLabel),

            // the close button element
            closeButton = $('<a class="closer" tabindex="0">' + Forms.createIconMarkup('fa-times') + '</a>'),

            // the container node for this pop-up node
            rootContainerNode = $(Utils.getOption(initOptions, 'rootContainerNode', window.document.body)),

            // original position of the menu node for tracking
            startOffset = null;

        // base constructor ---------------------------------------------------

        CompoundMenu.call(this, docView, Utils.extendOptions({
            boundingBox: docView.getApp().getWindowNode(),
            autoClose: false
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus back to the application before this menu
         * will be hidden.
         */
        function popupBeforeHideHandler() {
            docView.grabFocus();
        }

        /**
         * Initializes mouse/touch tracking to move this menu around.
         */
        function trackingStartHandler(event) {
            if (((rootContainerNode instanceof $) && !_.isEqual(rootContainerNode.get(0), window.document.body)) || (!(rootContainerNode instanceof $) && !_.isEqual(rootContainerNode, window.document.body))) {
                startOffset = self.getNode().position();
            } else {
                startOffset = self.getNode().offset();
            }

            event.preventDefault();
        }

        /**
         * Moves the menu while mouse/touch tracking is active.
         */
        function trackingMoveHandler(event) {
            self.setNodePosition(startOffset.left + event.offsetX, startOffset.top + event.offsetY);
            event.preventDefault();
        }

        /**
         * Click handler for the closer button in the header section. Triggers
         * a 'menu:userclose' event, and closes the menu afterwards.
         */
        function closeButtonHandler(e) {
            var type = e.type,
                key = e.keyCode;

            if ((type === 'keydown' && (key === KeyCodes.SPACE || key === KeyCodes.ENTER)) || type !== 'keydown') {
                self.trigger('menu:userclose').hide();

                //activator buttons should know the changes
                docView.getApp().getController().update();
            }
        }

        // initialization -----------------------------------------------------

        this.getNode().addClass('layer-menu').addClass('f6-target').attr('id', this.getUid());

        // event handlers for the pop-up menu
        this.on('popup:beforehide', popupBeforeHideHandler);

        // insert the header section nodes into this menu
        headerNode.append(titleNode, closeButton);
        this.prependContentNodes(headerNode);

        // make the menu node draggable
        titleNode.css('cursor', 'move');
        Tracking.enableTracking(titleNode);
        titleNode.on({ 'tracking:start': trackingStartHandler, 'tracking:move': trackingMoveHandler });

        // notify listeners and close the menu on click
        closeButton.on('click keydown', closeButtonHandler);

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = initOptions = headerNode = titleNode = closeButton = null;
        });

    } // class LayerMenu

    // exports ================================================================

    // derive this class from class CompoundMenu
    return CompoundMenu.extend({ constructor: LayerMenu });

});
