/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.json;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.ajax.container.FileHolder;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.Properties;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.groupware.ldap.User;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link GetDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
/*
 * MH/KA compilefix for buildsystem
 * @Action(method = RequestMethod.GET, name = "getdocument", description =
 * "Get the whole document in the original format, or converted to any supported document format.", parameters = {
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 * @Parameter(name = "id", description = "Object ID of the requested infoitem."),
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 * @Parameter(name = "version", optional=true, description =
 * "If present, the infoitem data describes the given version. Otherwise the current version is returned."),
 * @Parameter(name = "filter_format", optional=true, description =
 * "If this value is set to 'pdf', the whole document is as converted PDF file, in all other cases, the content of the document's substream, described by the parameter 'fragment, is returned as unmodified file."
 * ),
 * @Parameter(name = "fragment", optional=true, description =
 * "If this value is set and the filter_format is not set or not set to 'pdf', this parameter describes the substream name of the document internal file to be returned."
 * ),
 * @Parameter(name = "filename", optional=true, description =
 * "If present, this parameter contains the name of the infostore item to be used as initial part of the filename for the returned file, other the filename is set to 'file'."
 * ), }, responseDescription = "Response with timestamp: The document in it's original format or converted to the requested format.")
 */

@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class GetDocumentAction extends DocumentConverterAction {

    final private static String KEY_ERRORCODE = "errorCode";

    final private static int ERRORCODE_NONE = 0;
    final private static int ERRORCODE_BAD_REQUEST = 400;

    /**
     * Initializes a new {@link GetDocumentAction}.
     *
     * @param services
     */
    public GetDocumentAction(ServiceLookup services) {
        super(services);
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.json.actions.DocumentConverterAction#perform(com.openexchange.ajax.requesthandler.AJAXRequestData, com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(@NonNull AJAXRequestData request, @NonNull ServerSession session) {
        String fileName = request.getParameter("filename");
        String mimeType = request.getParameter("mimetype");
        final IDocumentConverter docConverter = m_services.getService(IDocumentConverter.class);
        final HashMap<String, Object> jobProperties = new HashMap<>(8);
        final String targetFormat = request.getParameter("documentformat"); // ["native", "pdf"]
        final String encTest = request.getParameter("enctest");
        final boolean convertToPDF = targetFormat.equals("pdf") && ((null == mimeType) || !mimeType.equalsIgnoreCase("application/pdf"));
        String saveAsFileName = request.getParameter("saveas_filename");
        String saveAsFolderId = request.getParameter("saveas_folder_id");
        final String async = request.getParameter("async");
        final boolean isAsync = (null != async) && async.equalsIgnoreCase("true");
        final JobError[] jobError = { JobError.NONE };
        ManagedFile managedResultFile = null;
        JSONObject jsonResult = null;
        IFileHolder resultFileHolder = null;

        if (convertToPDF && (null != docConverter)) {
            final User sessionUser = session.getUser();
            String prefLanguage = null;
            ManagedFile managedSourceFile = null;

            // user language
            if (null != sessionUser) {
                prefLanguage = sessionUser.getPreferredLanguage();

                if (!StringUtils.isEmpty(prefLanguage)) {
                    jobProperties.put(Properties.PROP_LOCALE, prefLanguage);
                }
            }

            final String resultDocumentId = getDocumentId(request, "pdf", prefLanguage);

            try {
                FileIdLocker.lock(resultDocumentId);
                managedResultFile = m_managedFileManagement.contains(resultDocumentId) ? m_managedFileManagement.getByID(resultDocumentId) : null;

                if (null == managedResultFile) {
                    String priority = request.getParameter("priority");
                    priority = (null != priority) ? priority.toLowerCase() : "";
                    final JobPriority jobPriority = priority.equals("background") ? JobPriority.BACKGROUND
                            : (priority.equals("low") ? JobPriority.LOW
                                    : (priority.equals("medium") ? JobPriority.MEDIUM
                                            : (priority.equals("high") ? JobPriority.HIGH : JobPriority.highest())));

                    managedSourceFile = getManagedFileForRequestDocument(request, session, jobError);

                    if(null != managedSourceFile) {
                        final HashMap<String, Object> resultProperties = new HashMap<>(8);

                        jobProperties.put(Properties.PROP_INPUT_FILE, managedSourceFile.getFile());
                        jobProperties.put(Properties.PROP_PRIORITY, jobPriority);

                        if (null != fileName) {
                            jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
                        }

                        if (isAsync) {
                            jobProperties.put(Properties.PROP_ASYNC, Boolean.TRUE);
                            jobProperties.put(Properties.PROP_PRIORITY, JobPriority.BACKGROUND);
                        }

                        // checked warning => stream wil either be directly closed or indirectly after write of the fileHolder
                        try (InputStream pdfResultStm = docConverter.convert("pdf", jobProperties, resultProperties)) {
                            if (null != pdfResultStm) {
                                fileName = getReplacedExtensionFileName(fileName, "pdf");
                                mimeType = "application/pdf";
                                managedResultFile = m_managedFileManagement.createManagedFile(resultDocumentId, pdfResultStm);
                            } else {
                                final Integer errorCode = (Integer) resultProperties.get(Properties.PROP_RESULT_ERROR_CODE);
                                jobError[0] = isAsync ?
                                    JobError.NONE :
                                        ((null != errorCode) ?
                                            JobError.fromErrorCode(errorCode.intValue()) :
                                                JobError.GENERAL);
                            }
                        }
                    }
                } else {
                    fileName = getReplacedExtensionFileName(fileName, "pdf");
                    mimeType = "application/pdf";
                }
            } catch (Exception e) {
                logException(e);
            } finally {
                FileIdLocker.unlock(resultDocumentId);
            }
        } else if (!isAsync) {
            // In case of PDF input and output, the result file is equal to the source file
            managedResultFile = getManagedFileForRequestDocument(request, session, jobError);

            // If source file is valid and encTest is set to 'pdf',
            // do an encryption test to get a valid error code in
            // case of an encrypted PDF
            if ((null != managedResultFile) && (null != encTest) && encTest.equalsIgnoreCase("pdf") && (null != docConverter)) {
                String priority = request.getParameter("priority");

                priority = (null != priority) ? priority.toLowerCase() : "";
                final JobPriority jobPriority = priority.equals("background") ? JobPriority.BACKGROUND
                        : (priority.equals("low") ? JobPriority.LOW
                                : (priority.equals("medium") ? JobPriority.MEDIUM
                                        : (priority.equals("high") ? JobPriority.HIGH : JobPriority.highest())));

                final HashMap<String, Object> resultProperties = new HashMap<>(8);

                jobProperties.put(Properties.PROP_INPUT_FILE, managedResultFile.getFile());
                jobProperties.put(Properties.PROP_PRIORITY, jobPriority);

                if (null != fileName) {
                    jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
                }

                // do the PDF encryption test and close possible result stream immediately
                IOUtils.closeQuietly(docConverter.convert("pdfenctest", jobProperties, resultProperties));

                final Integer errorCode = (Integer) resultProperties.get(Properties.PROP_RESULT_ERROR_CODE);
                jobError[0] = (null != errorCode) ? JobError.fromErrorCode(errorCode.intValue()) : JobError.GENERAL;
            }
        } else {
            // async case for PDF documents
            jobError[0] = JobError.NONE;
        }

        // set the temp file of the managed file as result content
        try {
            if (null != managedResultFile) {
                final File tempFile = managedResultFile.getFile();
                final String sessionId = request.getParameter("session");

                jsonResult = getJSONResultForSaveAs(session, sessionId, tempFile, saveAsFileName, saveAsFolderId, fileName, mimeType);

                if (null == jsonResult) {
                    resultFileHolder = new FileHolder(FileHolder.newClosureFor(tempFile), tempFile.length(), mimeType, fileName);
                } else if (jsonResult.has(KEY_ERRORCODE) && jsonResult.getInt(KEY_ERRORCODE) > ERRORCODE_NONE) {
                    return (new AJAXRequestResult().setHttpStatusCode(ERRORCODE_BAD_REQUEST));
                }
            } else if (isAsync) {
                (jsonResult = new JSONObject()).put("async", true);
            }
        } catch (JSONException e) {
            logException(e);
        }

        return getRequestResult(request, resultFileHolder, jsonResult, jobError[0]);
    }

    /**
     * @param session
     * @param resultFile
     * @param saveAsFileName
     * @param saveAsFolderId
     * @param givenFileName
     * @param givenMimeType
     * @return
     */
    protected JSONObject getJSONResultForSaveAs(@NonNull ServerSession session, String sessionId, File resultFile, String saveAsFileName, String saveAsFolderId,
                                                String givenFileName, String givenMimeType) {
        JSONObject ret = null;

        if ((null != resultFile) && resultFile.canRead() && (resultFile.length() > 0) && !StringUtils.isEmpty(saveAsFileName) && !StringUtils.isEmpty(saveAsFolderId)) {
            // #60025: we need to check for an illegal CSRF attempt by checking the
            // correct sessionId; set errorcode to 400 in case of an invalid request
            try {
                if ((null == sessionId) || !sessionId.equals(session.getSessionID())) {
                    final Logger LOG = LoggerFactory.getLogger(this.getClass());

                    if (null != LOG) {
                        LOG.error("DC Ajax handler was called with invalid sessionId for save request: " + sessionId);
                    }

                    ret = (new JSONObject()).put(KEY_ERRORCODE, ERRORCODE_BAD_REQUEST);
                } else {
                    final IDBasedFolderAccessFactory folderFactory = m_services.getService(IDBasedFolderAccessFactory.class);
                    final IDBasedFolderAccess folderAccess = (null != folderFactory) ? folderFactory.createAccess(session) : null;

                    if (null != folderAccess) {
                        final FileStorageFolder folder = folderAccess.getFolder(saveAsFolderId);
                        final FileStoragePermission permission = (null != folder) ? folder.getOwnPermission() : null;

                        if ((null != permission) && (permission.getFolderPermission() >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER)) {
                            final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                            final IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;

                            if (null != fileAccess) {
                                final com.openexchange.file.storage.File saveAsFileAccessFile = new DefaultFile();

                                saveAsFileAccessFile.setId(FileStorageFileAccess.NEW);
                                saveAsFileAccessFile.setFolderId(saveAsFolderId);
                                saveAsFileAccessFile.setFileName(saveAsFileName);
                                saveAsFileAccessFile.setFileMIMEType(givenMimeType);

                                try (final InputStream resultFileInputStm = FileUtils.openInputStream(resultFile)) {
                                    String resultFileId = fileAccess.saveDocument(saveAsFileAccessFile, resultFileInputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());

                                    if (!StringUtils.isEmpty(resultFileId)) {
                                        (ret = new JSONObject()).put("id", resultFileId).put("filename", saveAsFileAccessFile.getFileName()).put(KEY_ERRORCODE, 0);
                                    }
                                }

                                fileAccess.commit();
                            }
                        }
                    }
                }
            } catch(Exception e) {
                logException(e);
            }
        }

        return ret;
    }
}
