/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.json;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Timer;
import java.util.TimerTask;
import com.openexchange.documentconverter.client.impl.ClientManager;

/**
 * {@link FileIdLocker}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.4
 */
public class FileIdLocker {

    /**
     * @param fileId
     */
    public static void lock(final String fileId) {
        if (null != fileId) {
            FileIdLock fileIdLock = null;

            synchronized (m_fileIdMap) {
                fileIdLock = m_fileIdMap.get(fileId);

                if (null == fileIdLock) {
                    m_fileIdMap.put(fileId, fileIdLock = new FileIdLock());
                } else {
                    fileIdLock.updateTimestamp();
                }
            }

            fileIdLock.lock();
        }
    }

    /**
     * @param fileId
     */
    public static void unlock(final String fileId) {
        if (null != fileId) {
            synchronized (m_fileIdMap) {
                final FileIdLock fileIdLock = m_fileIdMap.get(fileId);

                if (null != fileIdLock) {
                    // update timestamp (touch) and move entry to last position in list
                    fileIdLock.updateTimestamp();

                    m_fileIdMap.remove(fileId);
                    m_fileIdMap.put(fileId, fileIdLock);

                    // finally unlock entry
                    fileIdLock.unlock();
                }
            }
        }
    }

    // - Members ----------------------------------------------------n-----------

    // 6 minutes
    final private static long TIMEOUT_MILLIS = 360 * 1000;

    // 3 minutes
    final private static long PERIOD_MILLIS = 180 * 1000;

    protected static LinkedHashMap<String, FileIdLock> m_fileIdMap = new LinkedHashMap<>();

    protected static Timer m_cleanupTimer = new Timer(true);

    static {
        m_cleanupTimer.scheduleAtFixedRate(new TimerTask() {

            /* (non-Javadoc)
             * @see java.util.TimerTask#run()
             */
            @Override
            public void run() {
                try {
                    final long curTimeMillis = System.currentTimeMillis();
                    final boolean trace = ClientManager.isLogTrace();
                    int mapCountBefore = 0;

                    if (trace) {
                        mapCountBefore = m_fileIdMap.size();
                    }

                    synchronized (m_fileIdMap) {
                        Iterator<String> iter = m_fileIdMap.keySet().iterator();

                        while (iter.hasNext()) {
                            final String curFileId = iter.next();

                            if (null != curFileId) {
                                final FileIdLock curFileIdLock = m_fileIdMap.get(curFileId);

                                if ((curTimeMillis - curFileIdLock.getTimestampMillis()) >= TIMEOUT_MILLIS) {
                                    if (!curFileIdLock.isLocked()) {
                                        iter.remove();
                                    }
                                } else {
                                    // all following entries didn't reach their timeout => leave
                                    break;
                                }
                            }
                        }
                    }

                    if (trace) {
                        final int mapCountAfter = m_fileIdMap.size();

                        ClientManager.logTrace(new StringBuilder(128).
                            append("DC FileIdLocker cleanup removed/left entries: ").
                            append((mapCountBefore - mapCountAfter)).append(" / ").
                            append(mapCountAfter).toString());
                    }
                } catch (Throwable e) {
                    ClientManager.logExcp(new Exception(e));
                }
            }

        }, TIMEOUT_MILLIS, PERIOD_MILLIS);
    }
}
