/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.impl;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;

import com.openexchange.documentconverter.LogData;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.ServerType;

/**
 * {@link RemoteDocumentConverterCheck}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
class RemoteValidator {

    protected class RemoteValidatorCallable implements Callable<Integer> {

        /**
         *
         */
        public RemoteValidatorCallable() {
            super();

            m_engineTypeName = (ServerType.DOCUMENTCONVERTER == m_serverType) ? "documentconverter" : "imageserver";
        }

        /**
         * @return
         * @throws Exception
         */
        @Override
        public Integer call() throws Exception {
            // check remote connection API; may be checked repeatedly, if a valid remote URL is set
            HttpURLConnection connection = null;
            BufferedReader inputReader = null;
            int remoteApiVersion = 0;

            try {
                connection = (HttpURLConnection) m_remoteUrl.openConnection();

                if (null != connection) {
                    final Pattern apiVersionPattern = Pattern.compile("api\\:\\s*v([0-9]+)");

                    connection.setRequestMethod("GET");
                    connection.setUseCaches(false);
                    connection.connect();

                    // read default response and check for 'documentconverter' content
                    inputReader = new BufferedReader(new InputStreamReader(connection.getInputStream()));

                    for (String readLine = null; (readLine = inputReader.readLine()) != null;) {
                        readLine = readLine.toLowerCase();

                        if (readLine.contains("open") && readLine.contains("xchange") && (readLine.contains(m_engineTypeName))) {
                            if (remoteApiVersion <= 0) {
                                remoteApiVersion = 1;
                            }
                        } else if (readLine.contains("api:")) {
                            final Matcher matcher = apiVersionPattern.matcher(readLine);

                            if (matcher.find()) {
                                remoteApiVersion = Integer.valueOf(matcher.group(1)).intValue();
                            }
                        }
                    }

                    inputReader.close();
                }
            } catch (Exception e) {
                remoteApiVersion = -1;
                ClientManager.logExcp(e);
            } finally {
                IOUtils.closeQuietly(inputReader);

                // disconnect and try again, if no valid answer has been received so far
                if (remoteApiVersion < 1) {
                    if (null != connection) {
                        connection.disconnect();
                    }

                    if (m_remoteAPIVersion.get() > 0) {
                        setConnectionLost();
                    } else {
                        synchronized (self) {
                            final long curLogTime = System.currentTimeMillis();

                            if ((0 == m_lastLogTime) || ((curLogTime - m_lastLogTime) >= ENABLE_LOG_TIMEOUT)) {
                                ClientManager.logError("Documentconverter remote connection could not be established => please check remote converter setup", new LogData("remoteurl", m_remoteUrl.toString()), new LogData("async_request", "true"));
                                m_lastLogTime = curLogTime;
                            }
                        }
                    }
                }
            }

            synchronized (self) {
                if (ClientManager.isLogInfo() && (remoteApiVersion > 0) && (m_remoteAPIVersion.get() < 1)) {
                    ClientManager.logInfo("Documentconverter established remote connection", new LogData("remoteurl", m_remoteUrl.toString()), new LogData("async_request", "true"));
                }

                m_remoteAPIVersion.set(remoteApiVersion);
                m_isRunning.compareAndSet(true, false);
            }

            return Integer.valueOf(remoteApiVersion);
        }

        // - Members -----------------------------------------------------------

        final private  static long ENABLE_LOG_TIMEOUT = 60000;

        private String m_engineTypeName = null;

    }

    /**
     * @param remoteUrl
     */
    RemoteValidator(@NonNull ServerType serverType, URL remoteUrl) {
        super();

        m_serverType = serverType;

        if (null != (m_remoteUrl = remoteUrl)) {
            m_callable = new RemoteValidatorCallable();

            // initial trigger at end of construction
            trigger();
        }
    }

    /**
     * @return
     */
    ServerType getSupportedServerType() {
        return m_serverType;
    }

    /**
     * @return
     */
    URL getRemoteUrl() {
        return m_remoteUrl;
    }

    /**
     * @return
     */
    boolean isRemoteServerSupported() {
        return (null != m_callable);
    }

    /**
     * @return
     */
    int getRemoteAPIVersion() {
        return m_remoteAPIVersion.get();
    }

    /**
     * @return
     */
    boolean isValidationRunning() {
        return m_isRunning.get();
    }

    /**
     * @return
     */
    synchronized void trigger() {
        if ((null != m_callable) && m_isRunning.compareAndSet(false, true)) {
            Executors.newSingleThreadExecutor().submit(new FutureTask<>(m_callable));
        }
    }

    /**
     *
     */
    synchronized void resetConnection() {
        if ((null != m_callable) && (m_remoteAPIVersion.get() >= 1) && !m_isRunning.get()) {
            setConnectionLost();
        }
    }

    // - Implementation

    /**
     *
     */
    protected void setConnectionLost() {
        m_remoteAPIVersion.set(-1);

        if (ClientManager.isLogError()) {
            ClientManager.logError("Documentconverter lost remote connection:", new LogData("remoteurl", m_remoteUrl.toString()));
        }
    }

    // - Members ---------------------------------------------------------------

    protected Object self = this;

    protected ServerType m_serverType = null;

    protected AtomicBoolean m_isRunning = new AtomicBoolean(false);

    protected URL m_remoteUrl = null;

    protected RemoteValidatorCallable m_callable = null;

    protected AtomicInteger m_remoteAPIVersion = new AtomicInteger(-1);

    protected long m_lastLogTime = 0;
}
