/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016, OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.impl;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import javax.imageio.ImageIO;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.documentconverter.AsyncExecutor;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.IImageTransformer;
import com.openexchange.documentconverter.ILogProvider;
import com.openexchange.documentconverter.ImageServerTransformations;
import com.openexchange.documentconverter.LogData;
import com.openexchange.documentconverter.LogType;
import com.openexchange.documentconverter.ManagerBasics;
import com.openexchange.documentconverter.ServerType;
import com.openexchange.exception.OXException;
import com.openexchange.imagetransformation.ImageServerTransformationService;
import com.openexchange.imagetransformation.ImageTransformations;
import com.openexchange.imagetransformation.TransformedImage;
import com.openexchange.server.ServiceLookup;


/**
 * {@link ClientManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
public class ClientManager extends ManagerBasics implements IDocumentConverter, IImageTransformer, ImageServerTransformationService {


    /**
     * Initializes a new {@link ClientManager}.
     */
    protected ClientManager(ServiceLookup services, ClientConfig clientConfig, Logger sysLog, Logger userLog) {
        super();

        m_clientConfig = (null != clientConfig) ? clientConfig : new ClientConfig();
        m_clientLogProvider = this;

        m_services = services;
        m_sysLog = sysLog;
        m_userLog = userLog;

        m_asyncExecutor = new AsyncExecutor(this, this, ASYNC_CONVERTER_THREADCOUNT);

     // create remoteValidators for each configured remote server type
        if ((null != m_clientConfig.REMOTEURL_DOCUMENTCONVERTER) && (m_clientConfig.REMOTEURL_DOCUMENTCONVERTER.toString().length() > 0)) {
            m_remoteDocumentConverterValidator = new RemoteValidator(ServerType.DOCUMENTCONVERTER, m_clientConfig.REMOTEURL_DOCUMENTCONVERTER);
        } else if (isLogError()) {
            logError("Documentconverter remoteDocumentConverterUrl property is empty or not set at all. Client is not able to perform conversions until a valid Url is set!");
        }

        // create remoteValidators for each configured remote server type
        if ((null != m_clientConfig.REMOTEURL_IMAGESERVER) && (m_clientConfig.REMOTEURL_IMAGESERVER.toString().length() > 0)) {
            m_remoteImageServerValidator = new RemoteValidator(ServerType.IMAGESERVER, m_clientConfig.REMOTEURL_IMAGESERVER);
        } else if (isLogError()) {
            logError("Documentconverter remoteImageServerUrl property is empty or not set at all. Client is not able to perform image transformations until a valid Url is set!");
        }
    }

    /**
     * @param services
     * @param config
     * @return
     */
    static public synchronized ClientManager getInstance(ServiceLookup services, ClientConfig config, Logger sysLog, Logger userLog) {
        if (null == m_clientManager) {
            m_clientManager = new ClientManager(services, config, sysLog, userLog);
        }

        return m_clientManager;
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.IClientManager#convert(java.lang.String, java.util.HashMap, java.util.HashMap)
     */
    @Override
    public InputStream convert(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        final ServerType serverType = (!StringUtils.isEmpty(jobType) && jobType.equals("imagetransformation")) ? ServerType.IMAGESERVER : ServerType.DOCUMENTCONVERTER;

        return (new ManagedClientHashJob(this, serverType)).process(jobType, jobProperties, resultProperties);
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.IClientManager#beginPageConversion(java.lang.String, java.util.HashMap, java.util.HashMap)
     */
    @Override
    public String beginPageConversion(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        return (new ManagedClientHashJob(this, ServerType.DOCUMENTCONVERTER)).beginPageConversion(jobType, jobProperties, resultProperties);
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.IClientManager#getConversionPage(java.lang.String, int, java.util.HashMap, java.util.HashMap)
     */
    @Override
    public InputStream getConversionPage(String jobIdStr, int pageNumber, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        return (new ManagedClientHashJob(this, ServerType.DOCUMENTCONVERTER)).getConversionPage(jobIdStr, pageNumber, jobProperties, resultProperties);
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.IClientManager#endPageConversion(java.lang.String, java.util.HashMap, java.util.HashMap)
     */
    @Override
    public void endPageConversion(String jobIdStr, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        (new ManagedClientHashJob(this, ServerType.DOCUMENTCONVERTER)).endPageConversion(jobIdStr, jobProperties, resultProperties);
    }

    // - IImageTransformer -----------------------------------------------------

    @Override
    public TransformedImage transformImage(InputStream imageInputStm, String formatName, Object userData) {
        TransformedImage ret = null;

        if ((null != imageInputStm) && !StringUtils.isEmpty(formatName) && (null != userData) && (userData instanceof ImageServerTransformations)) {
            final ImageTransformations imageTransformations = m_services.getOptionalService(ImageTransformations.class);

            if (null != imageTransformations) {
                final ImageServerTransformations imageServerTransformations = (ImageServerTransformations) userData;

                // TODO (KA): impl.
            }
        }

        return ret;
    }

    // - ImageTransformationService --------------------------------------

    @Override
    public ImageTransformations transfom(BufferedImage sourceImage) throws IOException {
        return transfom(sourceImage, null);
    }

    @Override
    public ImageTransformations transfom(BufferedImage sourceImage, Object source) throws IOException {
        ImageTransformations ret = null;

        if ((null != sourceImage) && (sourceImage.getWidth() > 0) && (sourceImage.getHeight() > 0)) {
            final ByteArrayOutputStream byteArrayOutputStm = new ByteArrayOutputStream();

            ImageIO.write(sourceImage, "png", byteArrayOutputStm);

            final byte[] imageData = byteArrayOutputStm.toByteArray();

            if ((null != imageData) && (imageData.length > 0)) {
                try (final InputStream imageStm = new ByteArrayInputStream(imageData)) {
                    ret = transfom(imageStm, source);
                }
            }
        }

        return ret;
    }

    @Override
    public ImageTransformations transfom(InputStream imageStm) throws IOException {
        return transfom(imageStm, null);
    }

    @Override
    public ImageTransformations transfom(InputStream imageStm, Object source) throws IOException {
        return ((null != imageStm) ? new ImageServerTransformations(this, imageStm, source) : null);
    }

    @Override
    public ImageTransformations transfom(IFileHolder imageFile, Object source) throws IOException {
        ImageTransformations ret = null;

        if ((null != imageFile) && (imageFile.getLength() > 0)) {
            try (final InputStream inputStm = imageFile.getStream()) {

                if (null != inputStm) {
                    ret = transfom(inputStm, source);
                }
            } catch (OXException e) {
                logExcp(e);
            }
        }

        return ret;
    }

    @Override
    public ImageTransformations transfom(byte[] imageData) throws IOException {
        return transfom(imageData, null);
    }

    @Override
    public ImageTransformations transfom(byte[] imageData, Object source) throws IOException {

        ImageTransformations ret = null;

        if ((null != imageData) && (imageData.length > 0)) {
            try (final InputStream inputStm = new ByteArrayInputStream(imageData)) {
                ret = transfom(inputStm, source);
            }
        }

        return ret;
    }

    // - ILogProvider ----------------------------------------------------------

    /**
     * @return
     */
    @Override
    public Logger getSysLog() {
        return m_sysLog;
    }

    /**
     * @return
     */
    @Override
    public Logger getUserLog() {
        return m_userLog;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     */
    ServiceLookup getServices() {
        return m_services;
    }

    /**
     * @return
     */
    public boolean hasRemoteServer(ServerType serverType) {
        return (null != getRemoteValidator(serverType));
    }

    /**
     * @param serverType
     * @return
     */
    public URL getRemoteUrl(ServerType serverType) {
        final RemoteValidator remoteValidator = getRemoteValidator(serverType);
        return ((null != remoteValidator) ? remoteValidator.getRemoteUrl() : null);
    }

    /**
     * @return
     */
    public int getRemoteAPIVersion(ServerType serverType) {
        final RemoteValidator remoteValidator = getRemoteValidator(serverType);
        final int apiVersion = (null != remoteValidator) ? remoteValidator.getRemoteAPIVersion() : 0;

        if ((null != remoteValidator) && (apiVersion <= 1)) {
            remoteValidator.trigger();
        }

        return apiVersion;
    }

    public void triggerRemoteConnection(ServerType serverType) {
        final RemoteValidator remoteValidator = getRemoteValidator(serverType);

        if (null != remoteValidator) {
            remoteValidator.trigger();
        }
    }

    /**
     * Resetting the current remote status.
     * This method should be called, whenever
     * there is an error with the current connection,
     * so that we know, that there's no current
     * connection available.
     */
    @Override
    public void resetRemoteConnection(ServerType serverType) {
        final RemoteValidator remoteValidator = getRemoteValidator(serverType);

        if ((null != remoteValidator) && remoteValidator.isRemoteServerSupported()) {
            // reset the status flag at validator
            remoteValidator.resetConnection();
        }
    }

    /**
     * @param jobType
     * @param jobProperties
     * @param resultProperties
     */
    public void triggerAsyncConvert(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        m_asyncExecutor.triggerExecution(jobType, jobProperties, resultProperties);
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param serverType
     * @return
     */
    protected RemoteValidator getRemoteValidator(ServerType serverType) {
        return ((ServerType.DOCUMENTCONVERTER == serverType) ? m_remoteDocumentConverterValidator :
                ((ServerType.IMAGESERVER == serverType) ? m_remoteImageServerValidator : null));
    }

    // - Config ----------------------------------------------------------------

    /**
     * @return
     */
    public static ClientConfig getClientConfig() {
        return m_clientConfig;
    }

    // - Logging ---------------------------------------------------------------

    static public boolean isLogTrace() {
        return isLogTraceImpl(m_clientLogProvider);
    }

    /**
     * @return true, if the log level 'info' is enabled
     */
    static public boolean isLogDebug() {
        return isLogDebugImpl(m_clientLogProvider);
    }

    /**
     * @return true, if the log level 'info' is enabled
     */
    static public boolean isLogInfo() {
        return isLogInfoImpl(m_clientLogProvider);
    }

    /**
     * @return true, if the log level 'warn' is enabled
     */
    static public boolean isLogWarn() {
        return isLogWarnImpl(m_clientLogProvider);
    }

    /**
     * @return true, if the log level 'error' is enabled
     */
    static public boolean isLogError() {
        return isLogErrorImpl(m_clientLogProvider);
    }

    /**
     * @param info
     * @param extraData
     */
    static public void logInfo(final String info, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_INFO, info, null, extraData);
    }

    /**
     * @param info
     * @param message
     * @param jobProperties
     * @param extraData
     */
    static public void logInfo(final String info, final HashMap<String, Object> jobProperties, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_INFO, info, jobProperties, extraData);
    }

    /**
     * @param warning
     * @param extraData
     */
    static public void logWarn(final String warning, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_WARN, warning, null, extraData);
    }

    /**
     * @param warning
     * @param jobProperties
     * @param extraData
     */
    static public void logWarn(final String warning, final HashMap<String, Object> jobProperties, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_WARN, warning, jobProperties, extraData);
    }

    /**
     * @param error
     * @param extraData
     */
    static public void logError(final String error, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_ERROR, error, null, extraData);
    }

    /**
     * @param error
     * @param jobProperties
     * @param extraData
     */
    static public void logError(final String error, final HashMap<String, Object> jobProperties, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_ERROR, error, jobProperties, extraData);
    }

    /**
     * @param error
     * @param extraData
     */
    static public void logDebug(final String debugMsg, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_DEBUG, debugMsg, null, extraData);
    }

    /**
     * @param error
     * @param jobProperties
     * @param extraData
     */
    static public void logDebug(final String debugMsg, final HashMap<String, Object> jobProperties, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_DEBUG, debugMsg, jobProperties, extraData);
    }

    /**
     * @param error
     * @param extraData
     */
    static public void logTrace(final String traceMsg, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_TRACE, traceMsg, null, extraData);
    }

    /**
     * @param error
     * @param jobProperties
     * @param extraData
     */
    static public void logTrace(final String traceMsg, final HashMap<String, Object> jobProperties, LogData... extraData) {
        logImpl(m_clientLogProvider, LogType.LOGTYPE_TRACE, traceMsg, jobProperties, extraData);
    }

    /**
     * @param e
     */
    static public void logExcp(Exception e) {
        logExcpImpl(m_clientLogProvider, e);
    }

    /**
     * @param logType
     * @param message
     * @param jobProperties
     * @param extraData
     */
    static public void log(final int logType, final String message, HashMap<String, Object> jobProperties, LogData... extraData) {
        logImpl(m_clientLogProvider, logType, message, jobProperties, extraData);
    }

    // - Statics ---------------------------------------------------------------

    private final static int ASYNC_CONVERTER_THREADCOUNT = 3;

    private static ClientManager m_clientManager = null;

    private static ClientConfig m_clientConfig = null;

    private static ILogProvider m_clientLogProvider = null;

    // - ClientManager ---------------------------------------------------------

    private ServiceLookup m_services = null;

    private Logger m_sysLog = null;

    private Logger m_userLog = null;

    private RemoteValidator m_remoteDocumentConverterValidator = null;

    private RemoteValidator m_remoteImageServerValidator = null;

    private AsyncExecutor m_asyncExecutor = null;
}
