/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.FOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNCID;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * Handler for the USM-invalidateObjects command.
 * @author ldo
 *
 */

public class InvalidateObjectsHandler extends NormalCommandHandler {

	private static final String ERRORS = "errors";
    private static final String OK = "ok";
    private static final String CHILDONLY = "childrenonly";
    private static final String NOT_FOUND = "not found";
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID, SYNCID };
	private static final String[] OPTIONAL_PARAMETERS = { FOLDERID, OBJECTIDS };

    public InvalidateObjectsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		long syncID = getSyncID();
		String folderUUID = getStringParameter(FOLDERID, null);
		JSONArray objectUUIDs = _parameters.has(OBJECTIDS) ? getJSONArray(_parameters, OBJECTIDS) : new JSONArray();
		JSONArray errors = new JSONArray();

		try {
			DataObject[] objects = null;
			String folderID = null;
			boolean invalidatingFolders = false;
			if (folderUUID == null) {
				//client has sent folders in the objects list
				objects = _session.getCachedFolders(syncID);
				invalidatingFolders = true;
			} else {
			    //invalidating objects in a given folder
				try {
					Folder folder = getFolderByUUID(folderUUID);
					folderID = folder.getID();
					objects = _session.getCachedFolderElements(folderID, folder.getElementsContentType(), syncID);
				} catch (USMJSONAPIException e) {
					errors.put(folderUUID);
					JSONObject response = new JSONObject();
					response.put(SYNCID, syncID);
					response.put(ERRORS, errors);
					return new ResponseObject(ResponseStatusCode.BAD_REQUEST, response, ConnectorBundleErrorCodes.UUID_DOES_NOT_EXIST, "No object found for UUID " + folderUUID, null, null, e);
				}
			}

			DataObjectSet objectsSet = new DataObjectSet(objects);
			int length = objectUUIDs.length();
			for (int i = 0; i < length; i++) {
				String uuid = objectUUIDs.getString(i);
				boolean removeOnlyChildren = uuid.endsWith(CHILDONLY);

				DataObject removedObj = null;

				if (!removeOnlyChildren) {
				   removedObj = objectsSet.remove(UUID.fromString(uuid));
				} else {
				   uuid = uuid.substring(0, uuid.indexOf(":"));
				   removedObj = objectsSet.get(UUID.fromString(uuid));
				}
				if (removedObj == null)
					errors.put(uuid);

				if (invalidatingFolders && removedObj != null) { //remove the subfolders from the synced state
				    removeSubfoldersFromSyncedFoldersSet(removedObj, objectsSet, new DataObjectSet(objectsSet));
				}

				if (invalidatingFolders && removeOnlyChildren && removedObj != null) { //invalidate all children objects
				    long newestTimestamp = _session.getNewestTimestamp(removedObj.getID());
				    _session.storeSyncState(newestTimestamp, newestTimestamp, new DataObjectSet().toArray()); //save a empty object set for the newest known timestamp
				    _session.invalidateCachedData(removedObj.getID());
				}
			}
			_session.invalidateCachedData(folderID); //invalidates the cashed data and marks the folder as dirty; for folderID = null -folderHierarchy
			long newSyncID = (invalidatingFolders) ? _session.storeSyncState(syncID, syncID, objectsSet.toArray())
					: _session.storeSyncState(syncID, syncID, folderID, objectsSet.toArray());

			//create the response
			JSONObject response = new JSONObject();
			response.put(SYNCID, newSyncID);
			if (errors.length()>0)
				response.put(ERRORS, errors);
			logSyncData(" invalidateObjects", folderUUID, syncID, newSyncID, 0L);
			return new ResponseObject(ResponseStatusCode.SUCCESS, response);

		} catch (StorageAccessException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_DB_ERROR, e);
		} catch (USMStorageException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_SQL_ERROR, e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_JSON_ERROR, e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_INTERNAL_ERROR,
					e);
		}
	}

	private void removeSubfoldersFromSyncedFoldersSet (DataObject parentFolder, DataObjectSet objectSet, DataObjectSet copyObjectSet) {
	    for (DataObject dataObject : copyObjectSet) {
            if(dataObject.getParentFolderID().equals(parentFolder.getID())) {
                objectSet.remove(dataObject.getID());
                removeSubfoldersFromSyncedFoldersSet(dataObject, objectSet, copyObjectSet);
            }
	    }
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

}
