package com.openexchange.office.tools.htmldoc;

import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

public class Drawing
        extends NodeHolder
{

    private static final String  CONTENT_START       = "<div contenteditable='false' ";
    private static final String  START               = " class='drawing inline' style='width: 50px; height: 50px; border: none;'>";
    private static final String  DRAWING_CONTENT     = "<div class='content'>";
    private static final String  PLACEHOLDER_CONTENT = "<div class='content placeholder'>";
    private static final String  GROUP_CONTENT       = "<div class='content groupcontent'>";
    private static final String  SHAPE_CONTENT       = "<div class='content textframecontent'><div class='textframe' contenteditable='true'>";
    private static final String  CONTENT_IMAGE_START = "<img src='";
    private static final String  CONTENT_IMAGE_END   = "' style='left: 0px; top: 0px; width: 50px; height: 50px;'>";
    private static final String  DIV_END             = "</div>";

    // ///////////////////////////////////////////////////////////

    private static AtomicInteger idCounter           = new AtomicInteger();

    // ///////////////////////////////////////////////////////////

    private final String         id;
    private final String         type;
    private final String         fileVersion;
    private final String         folderId;
    private final String         fileId;

    private boolean              emptySpans          = true;
    private boolean              emptySpanStyle       = true;
    private boolean              havePlaceholder     = true;

    private String               src                 = null;
    private boolean              presentation;
    private String target;

    public Drawing(
            String type, String fileId, String folderId, String fileVersion)
                throws Exception
    {
        this.fileId = fileId;
        this.folderId = folderId;
        this.fileVersion = fileVersion;
        this.type = type;
        this.id = "frame" + idCounter.getAndIncrement();

    }

    public Drawing setPresentation(
        boolean presentation)
    {
        this.presentation = presentation;
        if (presentation) {
            havePlaceholder = false;
        }
        return this;
    }

    @Override
    public boolean setAttribute(
        JSONArray start,
        JSONArray end,
        JSONObject assignedAttrs)
            throws Exception
    {
        if (assignedAttrs != null && assignedAttrs.has("image")) {
            JSONObject drawing = assignedAttrs.getJSONObject("image");
            String imageUrl = drawing.optString("imageUrl", "");
            if (imageUrl.contains("://")) {
                imageUrl = imageUrl.replaceAll("_REPLACE_SESSION_ID_", "_REPLACESESSIONID_"); // avoiding session ID replacement on client side for remote URLs (54403)
                drawing.put("imageUrl", imageUrl);
            }
        }

        super.setAttribute(start, end, assignedAttrs);

        if (start.length() == 0)
        {
            // type must be checked now, because we support more than image
            if ((null != type) && (type.equalsIgnoreCase("image")))
            {
                JSONObject attrs = getAttribute();
                if (attrs != null)
                {
                    JSONObject drawing = attrs.getJSONObject("image");
                    String imageUrl = drawing.getString("imageUrl");
                    if (imageUrl.contains("://"))
                    {
                        src = imageUrl;
                    }
                    else
                    {
                        src = "/appsuite/api/oxodocumentfilter?session=_REPLACE_SESSION_ID_&id=" + fileId + "&folder_id=" + folderId + "&action=getfile&get_filename=" + imageUrl + "&version=" + fileVersion + "&filename=drawing";
                        // /appsuite/api/oxodocumentfilter?session=a8d5e5d2454447eebf1419a19fb72dd1&uid=1405331293709.2453&id=500&folder_id=103&filename=kleinertest.docx&version=16&action=getfile&get_filename=word/media/uid267d2855268d.jpg
                    }
                }
            }
            return false;
        }
        else
        {
            return true;
        }

    }

    @Override
    public boolean appendContent(
        StringBuilder document)
            throws Exception
    {

        if (emptySpans)
        {
            GenDocHelper.addEmptySpan(document, emptySpanStyle);
        }

        document.append(CONTENT_START);

        final JSONObject adapter = new JSONObject();

        adapter.put("type", this.type);
        adapter.put("uid", id);
        adapter.put("model", JSONObject.NULL);

        adapter.put("attributes", getAttribute());

        document.append(" jquerydata='");
        document.append(GenDocHelper.escapeSingleQuote(adapter.toString()));
        document.append("' data-type='");
        document.append(type);
        document.append("' ");

        // System.out.println("Drawing.appendContent() " + this.type);

        String content = START;

        if (presentation)
        {
            content = "";
            if (null != target) {
                content += " target='" + target + "'";                
            }
            
            content += " class='drawing absolute' style='" + getAbsoluteCSS() + "'>";
        }

        document.append(content);
        // System.out.println("Drawing.appendContent() " + content);
        if (type.equalsIgnoreCase("image"))
        {

            document.append(DRAWING_CONTENT);

            String image = CONTENT_IMAGE_START;

            String srcStr = StringEscapeUtils.escapeHtml(src);
            srcStr = srcStr.replaceAll("'", "&#39;"); // escaping all single quotes (49847)

            image += srcStr;
            image += CONTENT_IMAGE_END;

            document.append(image);
        }
        else if (type.equalsIgnoreCase("group"))
        {
            document.append(GROUP_CONTENT);
            super.appendContent(document);
        }
        else if (type.equalsIgnoreCase("shape"))
        {
            if (havePlaceholder)
            {
                document.append(PLACEHOLDER_CONTENT);
                super.appendContent(document);
            }
            else
            {
                document.append(SHAPE_CONTENT);
                super.appendContent(document);

                document.append(DIV_END);
            }
        }
        else
        {
            // default behavior for types we cant handle yet (chart)
            document.append(PLACEHOLDER_CONTENT);
            super.appendContent(document);
        }

        document.append(DIV_END);
        document.append(DIV_END);

        if (!presentation && emptySpans)
        {
            GenDocHelper.addEmptySpan(document, emptySpanStyle);
        }
        return false;
    }
    
    public String getPHType(){
        JSONObject attrs = getAttribute();
        if (null == attrs) {
            return null;
        }
        
        JSONObject pres = attrs.optJSONObject("presentation");
        if (null == pres) {
            return null;
        }
        String phType = pres.optString("phType");
        if (StringUtils.isEmpty(phType)) {
            return null;
        }
        return phType;
    }

    private String getAbsoluteCSS()
        throws Exception
    {

        if (getAttribute() != null) {
            JSONObject drawing = getAttribute().optJSONObject("drawing");
            if (drawing != null) {
                StringBuilder res = new StringBuilder();
                //res.append("border: 1px solid blue; ");
                res.append("border: none; ");
                append(drawing, res, "width");
                append(drawing, res, "height");
                append(drawing, res, "left");
                append(drawing, res, "top");
                return res.toString();
            }
        }
        
//        System.out.println("Drawing.getAbsoluteCSS()         " + getAttribute().getJSONObject("drawing"));
//        if (null != phAttrs) {
//            System.out.println("Drawing.getAbsoluteCSS() phAttrs " + phAttrs.getJSONObject("drawing"));
//        }

        return "width: 50px; height: 50px; border: none;";
    }

    private void append(
        JSONObject drawing,
        StringBuilder res,
        String key)
    {
        long value = drawing.optLong(key, -1);
        if (value > -1) {
            res.append(key + ": " + Math.round(GenDocHelper.convertHmmToPx(value)) + "px; ");
        } 
    }

    @Override
    public boolean needsEmptySpan()
    {
        return false;
    }

    @Override
    public int getTextLength()
    {
        return 1;
    }

    @Override
    public void insert(
        JSONArray start,
        INode newChild)
            throws Exception
    {
        // System.out.println("Drawing.insert() " + start + " " + newChild.getClass());
        if (start.length() == 1)
        {
            if (newChild instanceof Drawing)
            {
                ((Drawing) newChild).disableEmptySpans();
            }
            else if ((newChild instanceof Paragraph) || (newChild instanceof Table))
            {
                havePlaceholder = false;
            }
        }
        super.insert(start, newChild);
    }

    public void disableEmptySpans()
    {
        this.emptySpans = false;
    }    

    public void disableEmptySpanStyle()
    {
        this.emptySpanStyle = false;
    }

    public Drawing setTarget(
        String target)
    {
        this.target = target;
        return this;
    }
}
