/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.components;

import java.text.ParseException;

import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextField;
import org.docx4j.dml.CTTextLineBreak;
import org.docx4j.dml.CTTextListStyle;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.ITextCharacterProperties;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.wml.ContentAccessor;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.pml.CTPlaceholder;
import org.pptx4j.pml.STPlaceholderType;
import org.pptx4j.pml.Shape;
import org.pptx4j.pml.SldMaster;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.ComponentContext;
import com.openexchange.office.ooxml.components.IParagraph;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.pptx.tools.PMLShapeHelper;

public class ParagraphComponent extends PptxComponent implements IParagraph {

    public ParagraphComponent(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
    }

    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
        final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

        Component nextComponent = null;
        final IndexedNode<Object> paragraphNode = getNode();
        final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)paragraphNode.getData()).getContent();
        for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
            final IndexedNode<Object> childNode = nodeList.getNode(i);
            final Object o = getContentModel(childNode, paragraphNode.getData());
            if(o instanceof CTRegularTextRun) {
            	if(((CTRegularTextRun)o).getT()!=null&&!((CTRegularTextRun)o).getT().isEmpty()) {
            		nextComponent = new TextComponent(this, childNode, nextComponentNumber);
            	}
            }
            else if(o instanceof CTTextField) {
            	nextComponent = new TextFieldComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof CTTextLineBreak) {
            	nextComponent = new HardBreakComponent(this, childNode, nextComponentNumber);
            }
        }
        return nextComponent;
    }

	@Override
	public void insertText(OperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
			throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
        	final CTTextParagraph paragraph = (CTTextParagraph)getObject();
        	
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                	final CTRegularTextRun t = (CTRegularTextRun)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getT());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setT(s.toString());
                    cRet = childComponent;
                }
                else {
                    final CTRegularTextRun newRun = Context.getDmlObjectFactory().createCTRegularTextRun();
                    newRun.setParent(getObject());
                    newRun.setT(text);
                    ((IndexedNodeList<Object>)paragraph.getContent()).addNode(childComponent.getNode(), new IndexedNode<Object>(newRun), false);

                    CTTextCharacterProperties referenceRPr = null;
            		if(childComponent.getObject() instanceof ITextCharacterProperties) {
                    	referenceRPr = ((ITextCharacterProperties)childComponent.getObject()).getRPr(false);
                	}
            		if(referenceRPr!=null) {
            			final CTTextCharacterProperties newRPr = XmlUtils.deepCopy(referenceRPr);
            			if(newRPr!=null) {
            				newRun.setRPr(newRPr);
            			}
            		}
                    cRet = childComponent.getNextComponent();
                }
          }
          else {
        	  // the paragraph is empty, we have to create R and its text
        	  final CTRegularTextRun newRun = Context.getDmlObjectFactory().createCTRegularTextRun();
        	  newRun.setParent(getObject());
        	  paragraph.getContent().add(newRun);
        	  if(paragraph.getPPr(false)!=null&&paragraph.getPPr(false).getDefRPr(false)!=null) {
    			  newRun.setRPr(XmlUtils.deepCopy(paragraph.getPPr(false).getDefRPr(true)));
        	  }
        	  newRun.setT(text);
        	  cRet = getNextChildComponent(null, null);
          	}
            if(attrs!=null) {
	        	cRet.splitStart(textPosition, SplitMode.ATTRIBUTE);
	        	cRet.splitEnd(textPosition+text.length()-1, SplitMode.ATTRIBUTE);
	        	cRet.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
	}

	@Override
	public void splitParagraph(int textPosition) {
    	final CTTextParagraph paragraph = (CTTextParagraph)getObject();

        // creating and inserting our new paragraph
        final CTTextParagraph destParagraph = Context.getDmlObjectFactory().createCTTextParagraph();
        destParagraph.setParent(paragraph.getParent());
        final IndexedNode<Object> destParagraphNode = new IndexedNode<Object>(destParagraph);
        ((IndexedNodeList<Object>)((ContentAccessor)paragraph.getParent()).getContent()).addNode(getNode(), destParagraphNode, textPosition==0);

        CTTextCharacterProperties lastRPr = null;
        if(textPosition>0) {
        	// splitting the paragraph
	        Component childComponent = getChildComponent(textPosition-1);
	        if(childComponent!=null) {
	        	lastRPr = ((ITextCharacterProperties)childComponent.getObject()).getRPr(false);
	        	childComponent.splitEnd(textPosition-1, SplitMode.DELETE);

	        	// moving text runs into the new paragraph
	            int sourceIndex = childComponent.getNode().getIndex() + 1;
	            final int size = paragraph.getContent().size() - sourceIndex;
	            if(size>0) {
	    	        IndexedNodeList.moveNodes((IndexedNodeList<Object>)paragraph.getContent(), (IndexedNodeList<Object>)destParagraph.getContent(), sourceIndex,
	    	        	0, size, destParagraph);
	            }
	        }
        }

        final CTTextParagraphProperties sourceParagraphProperties = paragraph.getPPr(false);
        if(sourceParagraphProperties!=null){
            final CTTextParagraphProperties destParagraphProperties = XmlUtils.deepCopy(sourceParagraphProperties);
            destParagraph.setPPr(destParagraphProperties);
        }

        // taking care of paragraph attributes
        if(lastRPr!=null) {
            // if available, we have to get the character attributes from the last textrun
        	destParagraph.getPPr(true).setDefRPr(XmlUtils.deepCopy(lastRPr));
        }
	}

	@Override
	public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final CTTextParagraph paragraph = (CTTextParagraph)getObject();
        	final IndexedNodeList<Object> sourceContent = ((CTTextParagraph)nextParagraphComponent.getObject()).getContent();
            final IndexedNodeList<Object> destContent = paragraph.getContent();
            if(sourceContent.size()>0) {
            	IndexedNodeList.moveNodes(sourceContent, destContent, 0, destContent.size(), sourceContent.size(), paragraph);
            }
            IndexedNode<Object> parentContextNode = nextParagraphComponent.getParentContext().getNode();
            ((ContentAccessor)parentContextNode.getData()).getContent().remove(nextParagraphComponent.getObject());
    	}
	}

	@Override
	public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
		throws JSONException, InvalidFormatException, PartUnrecognisedException {

		if(attrs==null) {
			return;
		}
		final boolean hasParagraphAttrs = attrs.hasAndNotNull("paragraph");
		final boolean hasCharacterAttrs = attrs.hasAndNotNull("character");
		if(hasParagraphAttrs||hasCharacterAttrs) {
			final IShapeComponent iShapeComponent = getShapeComponent();
			if(iShapeComponent.isPresentationObject()) {
				final ShapeComponent shapeComponent = (ShapeComponent)iShapeComponent;
				final SlideComponent slideComponent = PMLShapeHelper.getSlideComponent(shapeComponent);
				final Part slidePart = (Part)slideComponent.getObject();
				if(slidePart instanceof SlideMasterPart) {
					final Shape shape = (Shape)shapeComponent.getObject();
					final CTPlaceholder placeholder = shape.getNvSpPr().getNvPr().getPh();
					if(placeholder.getType()==STPlaceholderType.TITLE) {
						final SldMaster sldMaster = ((SlideMasterPart)slidePart).getJaxbElement();
						applyStyleFormatting(operationDocument, sldMaster.getTxStyles().getTitleStyle(), attrs);
					}
					else if(placeholder.getType()==STPlaceholderType.BODY) {
						final SldMaster sldMaster = ((SlideMasterPart)slidePart).getJaxbElement();
						applyStyleFormatting(operationDocument, sldMaster.getTxStyles().getBodyStyle(), attrs);
					}
					else {
						// presentation object on master slide
						applyObjectStyleFormatting(operationDocument, shapeComponent, attrs);
					}
				}
				else if(slidePart instanceof SlideLayoutPart) {
					// presentation object on layout slide -> the object style is attributed
					applyObjectStyleFormatting(operationDocument, shapeComponent, attrs);
				}
				else {
					// presentation object on normal slide -> the object is hard formatted
					applyHardFormatting(operationDocument, attrs);
				}
			}
			else {
				// always hard formatting
				applyHardFormatting(operationDocument, attrs);
			}
		}
	}

	private void applyStyleFormatting(OperationDocument operationDocument, CTTextListStyle textListStyle, JSONObject attrs)
		throws InvalidFormatException, PartUnrecognisedException, JSONException {

		DMLHelper.applyTextParagraphPropertiesFromJson(textListStyle.getPPr(getLevel(), true), attrs, operationDocument, operationDocument.getContextPart());
	}

	private void applyObjectStyleFormatting(OperationDocument operationDocument, ShapeComponent shapeComponent, JSONObject attrs)
		throws InvalidFormatException, PartUnrecognisedException, JSONException {

		DMLHelper.applyTextParagraphPropertiesFromJson(shapeComponent.getTextBody(true), getLevel(), attrs, operationDocument, operationDocument.getContextPart());
	}

	private void applyHardFormatting(OperationDocument operationDocument, JSONObject attrs)
		throws InvalidFormatException, PartUnrecognisedException, JSONException {

		final CTTextParagraph textParagraph = (CTTextParagraph)getObject();
		final Part part = operationDocument.getContextPart();
		DMLHelper.applyTextParagraphPropertiesFromJson(textParagraph.getPPr(true), attrs, operationDocument, part);
    	if(attrs.hasAndNotNull("character")) {
    		DMLHelper.applyTextCharacterPropertiesFromJson(textParagraph.getEndParaRPr(true), attrs, part);
    	}
	}

	@Override
	public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException, ParseException, FilterException {

		final CTTextParagraph textParagraph = (CTTextParagraph)getObject();
		DMLHelper.createJsonFromTextParagraphProperties(attrs, textParagraph.getPPr(false), createOperationHelper.getOperationDocument().getContextPart());
		if(textParagraph.getEndParaRPr(false)!=null&&textParagraph.getContent().isEmpty()) {
			DMLHelper.createJsonFromTextCharacterProperties(attrs, textParagraph.getEndParaRPr(false), createOperationHelper.getOperationDocument().getContextPart());
		}
		return attrs;
	}

	public IShapeComponent getShapeComponent() {
		return (IShapeComponent)getParentComponent();
	}

	public int getLevel() {
		final CTTextParagraphProperties properties = ((CTTextParagraph)getObject()).getPPr(false);
		final Integer lvl = properties!=null ? properties.getLvl() : null;
		return lvl!=null ? lvl.intValue() : 0;
	}

    @Override
    public Component insertChildComponent(com.openexchange.office.ooxml.OperationDocument operationDocument, int textPosition, JSONObject attrs, Type childType)
        throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

        Object newChild = null;

        CTTextCharacterProperties textCharacterProperties = null;
        if(childType == Type.HARDBREAK_DEFAULT) {
        	newChild = Context.getDmlObjectFactory().createCTTextLineBreak();
        }
        else if(childType == Type.SIMPLE_FIELD) {
        	newChild = Context.getDmlObjectFactory().createCTTextField();
        }
        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1, SplitMode.ATTRIBUTE);
            }
            if(childComponent instanceof TextComponent) {
            	textCharacterProperties = ((CTRegularTextRun)childComponent.getObject()).getRPr(false);
            }
        	final ContentAccessor parentContextObject = (ContentAccessor)childComponent.getParentContext().getNode().getData();
        	((IndexedNodeList<Object>)parentContextObject.getContent()).addNode(childComponent.getNode(), new IndexedNode<Object>(newChild), false);
        }
        else {
        	final CTTextParagraph paragraph = (CTTextParagraph)getObject();
        	paragraph.getContent().addNode(new IndexedNode<Object>(newChild));
        	textCharacterProperties = paragraph.getEndParaRPr(false);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(textCharacterProperties!=null&&childComponent instanceof ITextCharacterProperties) {
        	((ITextCharacterProperties)childComponent).getRPr(true);
        }
        if(attrs!=null) {
            childComponent.applyAttrsFromJSON(operationDocument, attrs);
        }
        return childComponent;
    }
}
