/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import javax.imageio.ImageIO;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.Names;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.odftoolkit.odfdom.type.Base64Binary;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.DrawingType;
import com.openexchange.office.odf.IDrawing;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.ods.dom.MapHelper;

public class DrawImage implements IDrawing {

	private final AttributesImpl attributes;
	private final List<IElementWriter> childs = new ArrayList<IElementWriter>(1);

	public DrawImage() {
		this.attributes = new AttributesImpl();
	}

	public DrawImage(Attributes attributes) {
		this.attributes = new AttributesImpl(attributes);
	}

	public List<IElementWriter> getChilds() {
		return childs;
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		output.startElement(Namespaces.DRAW, "image", "draw:image");
		attributes.write(output);
		for(IElementWriter child:childs) {
			child.writeObject(output);
		}
		output.endElement(Namespaces.DRAW, "image", "draw:image");
	}

	@Override
	public DrawingType getType() {
		return DrawingType.IMAGE;
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		final JSONObject imageProps = attrs.optJSONObject("image");
		if(imageProps==null) {
			return;
		}
        if (imageProps.has("imageUrl")) {
            String href = imageProps.optString("imageUrl");
            if (href != null && !href.isEmpty()) {
                attributes.setValue(Namespaces.XLINK, "xlink:href", "href", href);
            }
        } else if (imageProps.has("imageData")) {
            String imageData = imageProps.optString("imageData");
            if (imageData != null && !imageData.isEmpty()) {
                //expected header is
                //		"data:image/png;base64,
                String[] header = imageData.split("base64,");
                String suffix = "png";
                String mediaTypeString = "image/png";
                mediaTypeString = header[0].substring(header[0].indexOf(":") + 1, header[0].indexOf(";"));
                suffix = header[0].substring(header[0].indexOf("/") + 1, header[0].indexOf(";"));
                String fileName = "img_" + new Random().nextInt() + "." + suffix;
                Base64Binary base64Binary = Base64Binary.valueOf(header[1]);
                operationDocument.getPackage().insert(base64Binary.getBytes(), "Pictures/" + fileName, mediaTypeString);
                attributes.setValue(Namespaces.XLINK, "xlink:href", "href", "Pictures/" + fileName);
            }
        }
        if (imageProps.has("imageXmlId")) {
            String xmlId = imageProps.optString("imageXmlId");
            if (xmlId != null && !xmlId.isEmpty()) {
            	attributes.setValue(Namespaces.XML, "xml:id", "id", xmlId);
            }
        }
	}

	@Override
	public JSONObject createJSONAttrs(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {
	
		final String hRef = attributes.getValue("xlink:href");
		if(hRef!=null) {
			JSONObject imageProps = attrs.optJSONObject("image");
			if(imageProps==null) {
				imageProps = new JSONObject();
				attrs.put("image", imageProps);
			}
            imageProps.put("imageUrl", hRef);

            if (imageProps.has("cropRight") && (imageProps.has("height") || imageProps.has("width"))) {
            	calculateCrops(operationDocument, hRef, imageProps);
            }
        }
		final String id = attributes.getValue("xml:id");
		if(id!=null) {
			JSONObject drawingProps = attrs.optJSONObject("drawing");
			if(drawingProps==null) {
				drawingProps = new JSONObject();
				attrs.put("drawing", drawingProps);
			}
            drawingProps.put("imageXmlId", id);
        }
		return attrs;
	}

    /**
     * LO/AO does not interpret clipping as in FO/CSS
     * http://www.w3.org/TR/CSS2/visufx.html#propdef-clip Instead the clip
     * values measure the distance from each border to the start of the viewed
     * area. The clip vales are taking measure from the original size, which is
     * not part of the OFD XML, therefore the image have to be loaded for
     * receiving the size.
     */
    private void calculateCrops(OdfOperationDocument operationDocument, String href, JSONObject imageProps)
    	throws JSONException {

    	try {
            // ToDo: Although the streams are cached we might cache the clipping for known href, help if images occure more than once
    		final OdfPackage pkg = operationDocument.getPackage();
            InputStream is = pkg.getInputStream(href);
            if (is != null) {
                BufferedImage bimg = ImageIO.read(is);
                if (bimg != null) {
                    double width = MapHelper.normalizeLength((bimg.getWidth() / Names.DOTS_PER_INCH) + "in");
                    double height = MapHelper.normalizeLength((bimg.getHeight() / Names.DOTS_PER_INCH) + "in");

                	// 2nd half of absolute fo:clip to relative crop (OX API) mapping
                    if (imageProps.has("cropRight")) {
                        Number cropRight = (Number) imageProps.get("cropRight");
                        if (cropRight != null) {
                            if (cropRight.doubleValue() != 0.0) {
                                imageProps.put("cropRight", cropRight.doubleValue() * 100.0 / width);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropRight");
                            }
                        }
                    }
                    if (imageProps.has("cropLeft")) {
                        Number cropLeft = (Number) imageProps.get("cropLeft");
                        if (cropLeft != null) {
                            if (cropLeft.doubleValue() != 0.0) {
                                imageProps.put("cropLeft", cropLeft.doubleValue() * 100.0 / width);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropLeft");
                            }
                        }
                    }
                    // 2nd half of absolute fo:clip to relative crop (OX API) mapping
                    if (imageProps.has("cropTop")) {
                        Number cropTop = (Number) imageProps.get("cropTop");
                        if (cropTop != null) {
                            if (cropTop.doubleValue() != 0.0) {
                                imageProps.put("cropTop", cropTop.doubleValue() * 100.0 / height);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropTop");
                            }
                        }
                    }
                    if (imageProps.has("cropBottom")) {
                        Number cropBottom = (Number) imageProps.get("cropBottom");
                        if (cropBottom != null) {
                            if (cropBottom.doubleValue() != 0.0) {
                                imageProps.put("cropBottom", cropBottom.doubleValue() * 100.0 / height);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropBottom");

                            }
                        }
                    }
                }
            }
        } catch (IOException ex) {
        }
    }
}
