/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom;

import java.util.HashMap;
import java.util.Map;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawFrameElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.AttributeImpl;
import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.DrawingType;
import com.openexchange.office.odf.IDrawing;
import com.openexchange.office.odf.INodeAccessor;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.Styles;
import com.openexchange.office.ods.dom.MapHelper;

public class DrawFrame implements IElementWriter, IDrawing, INodeAccessor {

	private final AttributesImpl attributes;
	private final DLList<Object> childs = new DLList<Object>();

	public DrawFrame(Attributes attributes) {
		this.attributes = new AttributesImpl(attributes);
	}

	public DrawFrame(IDrawing iDrawing) {
		this.attributes = new AttributesImpl();
		childs.add(iDrawing);
	}

	public DLList<Object> getChilds() {
		return childs;
	}

	public IDrawing getDrawing() {
		if(!childs.isEmpty()) {
			final Object o = childs.get(0);
			if(o instanceof IDrawing) {
				return (IDrawing)o;
			}
		}
		return null;
	}

	@Override
	public DLList<Object> getContent() {
		final IDrawing iDrawing = getDrawing();
		if(iDrawing instanceof INodeAccessor) {
			((INodeAccessor)iDrawing).getContent();
		}
		return new DLList<Object>();
	}

	public DrawingType getType() {
		final IDrawing drawing = getDrawing();
		if(drawing!=null) {
			return drawing.getType();
		}
		return DrawingType.UNDEFINED;
	}

	public AttributesImpl getAttributes() {
		return attributes;
	}

	public String getStyleName() {
		return attributes.getValue("draw:style-name");
	}

	public void setStyleName(String name) {
		attributes.setValue(Namespaces.DRAW, "draw:style-name", "style-name", name);
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		output.startElement(Namespaces.DRAW, "frame", "draw:frame");
		attributes.write(output);
		for(Object child:getChilds()) {
			((IElementWriter)child).writeObject(output);
		}
		output.endElement(Namespaces.DRAW, "frame", "draw:frame");
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		// TODO: big fake ... to be able to use the old "addStyle" functionality
		// we temporarily create a TextSpanElement to provide the actual style
		final OdfFileDom ownerDocument = operationDocument.getOwnerDocument();
		final OdfStylableElement e = new DrawFrameElement(ownerDocument);
		e.setAttributeNS(Namespaces.DRAW, "draw:style-name", getStyleName());
		Styles.addStyle(attrs, e, ownerDocument);
		setStyleName(e.getAttributeNS(Namespaces.DRAW, "style-name"));

		final JSONObject drawingProps = attrs.optJSONObject("drawing");
        if(drawingProps != null) {
        	if (drawingProps.has("width")) {
                int width = drawingProps.optInt("width");
                attributes.setValue(Namespaces.SVG, "svg:width", "width", (width / 100.0 + "mm"));
            }
            if (drawingProps.has("height")) {
                int height = drawingProps.optInt("height");
                attributes.setValue(Namespaces.SVG, "svg:height", "height", (height / 100.0 + "mm"));
            }
            if (drawingProps.has("name")) {
                String name = drawingProps.optString("name");
                if (name != null && !name.isEmpty()) {
                	attributes.setValue(Namespaces.DRAW, "draw:name", "name", name);
                }
            }
            if (drawingProps.has("anchorHorOffset")) {
                int x = drawingProps.optInt("anchorHorOffset");
                attributes.setValue(Namespaces.SVG, "svg:x", "x", (x / 100.0 + "mm"));
            }
            if (drawingProps.has("anchorVertOffset")) {
                int y = drawingProps.optInt("anchorVertOffset");
                attributes.setValue(Namespaces.SVG, "svg:y", "y", (y / 100.0 + "mm"));
            }
            if (drawingProps.hasAndNotNull("inline") && drawingProps.optBoolean("inline")) {
            	attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "as-char");
            } else if (drawingProps.has("anchorHorBase") && drawingProps.has("anchorVertBase")) {
                String anchorHorBase = drawingProps.optString("anchorHorBase");
                String anchorVertBase = drawingProps.optString("anchorVertBase");
                if (anchorHorBase != null && anchorVertBase != null) {
                    if (anchorHorBase.equals("page") && anchorVertBase.equals("page")) {
                        attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "paragraph");

                    } else if (anchorHorBase.equals("column") && anchorVertBase.equals("margin")) {
                    	attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "frame");

                    } else if (anchorHorBase.equals("column") && anchorVertBase.equals("paragraph")) {
                    	attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "paragraph");
                        //apply related default wrapping, if not part of the attributes:
                        if (!drawingProps.has("textWrapMode") && !drawingProps.has("textWrapSide")) {
                            drawingProps.put("textWrapMode", "topAndBottom");
                        }
                    } else if (anchorHorBase.equals("character") && anchorVertBase.equals("paragraph")) {
                        attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "char");
                    } else { // the default is "inline" a
                    	attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "as-char");
                    }
                }
            } else {
                if (drawingProps.has("anchorHorBase")) {
                    String anchorHorBase = drawingProps.optString("anchorHorBase");
                    if (anchorHorBase != null) {
                        if (anchorHorBase.equals("page")) {
                            attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "page");

                        } else if (anchorHorBase.equals("column")) {
                        	attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "paragraph");

                        } else if (anchorHorBase.equals("character")) {
                        	attributes.setValue(Namespaces.TEXT, "text:anchor-type", "anchor-type", "char");
                        }
                    }
                }
            }
            if (drawingProps.has("anchorLayerOrder")) {
                int anchorLayerOrder = drawingProps.optInt("anchorLayerOrder", 0);
                attributes.setIntValue(Namespaces.DRAW, "draw:z-index", "z-index", anchorLayerOrder);
            }
        }
        final JSONObject shapeProps = attrs.optJSONObject("shape");
        if(shapeProps!=null) {
            if (shapeProps.has("autoResizeHeight") && shapeProps.optBoolean("autoResizeHeight") == true) {
            	attributes.getMap().remove("svg:height");
            }
        }

		final IDrawing iDrawing = getDrawing();
		if(iDrawing!=null) {
			iDrawing.applyAttrsFromJSON(operationDocument, attrs);
		}
	}

	@Override
	public JSONObject createJSONAttrs(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		final HashMap<String, AttributeImpl> frameAttributes = getAttributes().getMap();

		// Hard formatted properties (automatic styles)
        Map<String, Object> hardFormatting;
        final String styleName = getStyleName();
        if(styleName!=null&&!styleName.isEmpty()) {
        	hardFormatting = Styles.getAutomaticStyleHierarchyProps(styleName, "graphic", ((Styles)operationDocument.getDocument().getStylesDom()).getStyles(), attrs.asMap());
        	hardFormatting.put("styleId", styleName);
        }
        else {
        	hardFormatting = attrs.asMap();
        }
        JSONObject drawingProps = (JSONObject)hardFormatting.get("drawing");
        if(drawingProps==null) {
            drawingProps = new JSONObject();
            hardFormatting.put("drawing", drawingProps);
        }

        int dx = 0;
        int dy = 0;

        AttributeImpl attr = frameAttributes.get("draw:transform");
        if(attr!=null) {
        	String transform = attr.getValue();
            int index = transform.indexOf("translate");
            if( index >= 0) {
                index = transform.indexOf('(', index);
                transform = transform.substring(index, transform.length());
                int separator = transform.indexOf(' ');
                String leftValue = transform.substring(1, separator);
                index = transform.indexOf(')', separator);
                String rightValue = transform.substring(separator + 1, index);
                dx = MapHelper.normalizeLength(leftValue);
                dy = MapHelper.normalizeLength(rightValue);
            }
        }

        attr = frameAttributes.get("svg:width");
        if(attr!=null) {
        	drawingProps.put("width", MapHelper.normalizeLength(attr.getValue()));
        }

        attr = frameAttributes.get("svg:height");
        if(attr!=null) {
        	drawingProps.put("height", MapHelper.normalizeLength(attr.getValue()));
        }

        attr = frameAttributes.get("svg:x");
        int x = dx;
        if(attr!=null) {
        	x += MapHelper.normalizeLength(attr.getValue());
        }
        if(x!=0) {
        	drawingProps.put("anchorHorOffset", x + dx);
        }

        attr = frameAttributes.get("svg:y");
        int y = dy;
        if(attr!=null) {
        	y += MapHelper.normalizeLength(attr.getValue());
        }
        if(y!=0) {
            drawingProps.put("anchorVertOffset", y + dy);
        }

        attr = frameAttributes.get("draw:z-index");
        if(attr!=null) {
            drawingProps.put("anchorLayerOrder", attr.getIntValue());
        }

        attr = frameAttributes.get("text:anchor-type");
        if(attr!=null) {
            String anchorVertBase = null;
            String anchorHorBase = null;
            String anchorType = attr.getValue();
            if (anchorType.equals("page")) {
                // OX API: true: image as character, false: floating mode
                drawingProps.put("inline", Boolean.FALSE);
                //page anchor requires page relation
                drawingProps.put("anchorHorBase", "page");
                drawingProps.put("anchorVertBase", "page");
            } else if (anchorType.equals("frame")) {
                // OX API: true: image as character, false: floating mode
                drawingProps.put("inline", Boolean.FALSE);
                anchorVertBase = "column";
                anchorVertBase= "margin";
            } else if (anchorType.equals("paragraph")) {
                // OX API: true: image as character, false: floating mode
                drawingProps.put("inline", Boolean.FALSE);
                anchorHorBase = "column";
                anchorVertBase = "paragraph";
            } else if (anchorType.equals("char")) {
                // OX API: true: image as character, true: floating mode
                drawingProps.put("inline", Boolean.FALSE);
                anchorHorBase = "character";
                anchorVertBase = "paragraph";
            } else if (anchorType.equals("as-char")) {
                // OX API: true: image as character, false: floating mode
                drawingProps.put("inline", Boolean.TRUE);
            }
            if(anchorVertBase != null && !drawingProps.has("anchorVertBase")) {
                drawingProps.put("anchorVertBase", anchorVertBase);
            }
            if(anchorHorBase != null && !drawingProps.has("anchorHorBase")) {
                drawingProps.put("anchorHorBase", anchorHorBase);
            }
        }
        final IDrawing iDrawing = getDrawing();
        if(iDrawing!=null) {
        	return iDrawing.createJSONAttrs(operationDocument, new JSONObject(hardFormatting));
        }
		return new JSONObject(hardFormatting);
	}
}
