/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.dom.attribute.draw.DrawStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.style.StyleProtectAttribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgHeightAttribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgWidthAttribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgX1Attribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgX2Attribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgXAttribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgY1Attribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgY2Attribute;
import org.odftoolkit.odfdom.dom.attribute.svg.SvgYAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableEndCellAddressAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableEndXAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableEndYAttribute;
import org.odftoolkit.odfdom.dom.element.draw.DrawFrameElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.type.Length;
import org.odftoolkit.odfdom.type.Length.Unit;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.SaxContextHandler;

public class Drawing {

	private DrawingAnchor drawingAnchor;
	private ElementNSImpl element;

	private String type;

	public Drawing(DrawingAnchor anchor) {
		drawingAnchor = anchor;
		element = null;
		type = "undefined";
	}

	public String getType() {
		return type;
	}

	public void setType(String type) {
		this.type = type;
	}

	public DrawingAnchor getAnchor() {
		return drawingAnchor;
	}

	public ElementNSImpl getElement() {
		return element;
	}

	public void setElement(ElementNSImpl element) {
		this.element = element;
	}

	// changes the DrawingAnchor, if the Drawing is already added to Drawings, then
	// this DrawingAnchor should be changed via Drawings.setDrawingAnchor() only
	public void setDrawingAnchor(DrawingAnchor anchor) {
		drawingAnchor = anchor;
	}

	public void writeDrawing(SerializationHandler output) {
		if(element!=null) {
			SaxContextHandler.serializeElement(output, element);
		}
		else {
			// write dummy drawing, this is necessary to ensure the same number
			// of drawings after save/reload
			SaxContextHandler.startElement(output, DrawFrameElement.ELEMENT_NAME);
			// TODO: ... create a valid dummy frame
			SaxContextHandler.endElement(output, DrawFrameElement.ELEMENT_NAME);
		}
	}

	public JSONObject createAttributes(Sheet sheet, JSONObject attrs)
		throws JSONException, SAXException {

		if(attrs==null) {
			attrs = new JSONObject();
		}
		final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
		final JSONObject drawingAttrs = initialDrawingAttrs!=null?initialDrawingAttrs:new JSONObject();

		if(element!=null) {

			boolean protectedSize = false;
			boolean protectedPosition = false;

			final OdfFileDom xDoc = (OdfFileDom)sheet.getOwnerDocument();
            final OdfDocument odfDoc = (OdfDocument)xDoc.getDocument();
            final OdfOfficeStyles officeStyles = odfDoc.getStylesDom().getOfficeStyles();
			
			final String graphicStyleName = element.getAttributeNS(DrawStyleNameAttribute.ATTRIBUTE_NAME.getUri(), "style-name");
			if(graphicStyleName!="") {
				OdfStyle ownStyle = officeStyles.getStyle(graphicStyleName, OdfStyleFamily.Graphic);
				if(ownStyle == null) {
					ownStyle = odfDoc.getContentDom().getAutomaticStyles().getStyle(graphicStyleName, OdfStyleFamily.Graphic);
				}
				if(ownStyle!=null) {
					final String styleProtect = ownStyle.getAttributeNS(StyleProtectAttribute.ATTRIBUTE_NAME.getUri(), "protect");
					if(styleProtect!="") {
						if(styleProtect.contains("size")) {
							protectedSize = true;
						}
						if(styleProtect.contains("position")) {
							protectedPosition = true;
						}
					}
				}
			}

			Double posX = null;
			Double posY = null;
			Double width = null;
			Double height = null;
			if(element.getLocalName().equals("line")) {
				// nice ... special code for lines...
				final String x1 = element.getAttributeNS(SvgX1Attribute.ATTRIBUTE_NAME.getUri(), "x1");
				final String y1 = element.getAttributeNS(SvgY1Attribute.ATTRIBUTE_NAME.getUri(), "y1");
				final String x2 = element.getAttributeNS(SvgX2Attribute.ATTRIBUTE_NAME.getUri(), "x2");
				final String y2 = element.getAttributeNS(SvgY2Attribute.ATTRIBUTE_NAME.getUri(), "y2");
				if(!x1.isEmpty()&&!y1.isEmpty()&&!x2.isEmpty()&&!y2.isEmpty()) {
					final Double posX1 = Length.getLength(x1, Unit.MILLIMETER);
					final Double posY1 = Length.getLength(y1, Unit.MILLIMETER);
					final Double posX2 = Length.getLength(x2, Unit.MILLIMETER);
					final Double posY2 = Length.getLength(y2, Unit.MILLIMETER);
					if(posX1!=null&&posY1!=null&&posX2!=null&&posY2!=null) {
						posX = Math.min(posX1, posX2);
						posY = Math.min(posY1, posY2);
						width = Math.abs(posX2-posX1);
						height = Math.abs(posY2-posY1);
					}
				}
			}
			else {
				final String x = element.getAttributeNS(SvgXAttribute.ATTRIBUTE_NAME.getUri(), "x");
				final String y = element.getAttributeNS(SvgYAttribute.ATTRIBUTE_NAME.getUri(), "y");
				if(!x.isEmpty()) {
					posX = Length.getLength(x, Unit.MILLIMETER);
				}
				if(!y.isEmpty()) {
					posY = Length.getLength(y, Unit.MILLIMETER);
				}
				final String w = element.getAttributeNS(SvgWidthAttribute.ATTRIBUTE_NAME.getUri(), "width");
				final String h = element.getAttributeNS(SvgHeightAttribute.ATTRIBUTE_NAME.getUri(), "height");
				if(!w.isEmpty()) {
	    			width = Length.getLength(w, Unit.MILLIMETER);
				}
				if(!h.isEmpty()) {
					height = Length.getLength(h, Unit.MILLIMETER);
				}
			}

			if(drawingAnchor==null) {
				drawingAttrs.put("anchorType", "absolute");
    			if(posX!=null) {
    				drawingAttrs.put("left", posX*100);
    			}
    			if(posY!=null) {
    				drawingAttrs.put("top", posY*100);
    			}
			}
			else {
				drawingAttrs.put("anchorType", "oneCell"); 			// protectedSize ? "oneCell" : "twoCell");
				drawingAttrs.put("startCol", drawingAnchor.getColumn());
				drawingAttrs.put("startRow", drawingAnchor.getRow());
				if(posX!=null) {
					drawingAttrs.put("startColOffset", posX*100);
				}
				if(posY!=null) {
					drawingAttrs.put("startRowOffset", posY*100);
				}
			}
			if(width!=null) {
				drawingAttrs.put("width", width*100);
			}
			if(height!=null) {
				drawingAttrs.put("height", height*100);
			}
		}
		if(!drawingAttrs.isEmpty()&&initialDrawingAttrs!=drawingAttrs) {
			attrs.put("drawing", drawingAttrs);
		}
		return attrs;
	}

	// anchorType and startCol/startRow is already set
	public void applyAttributes(OdfOperationDocument opsDoc, JSONObject attrs)
		throws JSONException {

		if(element==null) {
			return;
		}

		final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
		if(drawingAttrs!=null) {
			Double width = null;
			Double height = null;

			if(drawingAttrs.has("width")) {
				width = drawingAttrs.getDouble("width");
			}
			if(drawingAttrs.has("height")) {
				height = drawingAttrs.getDouble("height");
			}

			if(element.getLocalName().equals("line")) {
				
			}
			else {

				if(drawingAnchor==null) {
					if(drawingAttrs.has("left")) {
						element.setAttributeNS(SvgXAttribute.ATTRIBUTE_NAME.getUri(), SvgXAttribute.ATTRIBUTE_NAME.getQName(), Double.toString(drawingAttrs.getDouble("left")/100) + "mm");
					}
					if(drawingAttrs.has("top")) {
						element.setAttributeNS(SvgYAttribute.ATTRIBUTE_NAME.getUri(), SvgYAttribute.ATTRIBUTE_NAME.getQName(), Double.toString(drawingAttrs.getDouble("top")/100) + "mm");
					}
				}
				else {
					if(drawingAttrs.has("startColOffset")) {
						element.setAttributeNS(SvgXAttribute.ATTRIBUTE_NAME.getUri(), SvgXAttribute.ATTRIBUTE_NAME.getQName(), Double.toString(drawingAttrs.getDouble("startColOffset")/100) + "mm");
					}
					if(drawingAttrs.has("startRowOffset")) {
						element.setAttributeNS(SvgYAttribute.ATTRIBUTE_NAME.getUri(), SvgYAttribute.ATTRIBUTE_NAME.getQName(), Double.toString(drawingAttrs.getDouble("startRowOffset")/100) + "mm");
					}
/*
					if(!drawingAttrs.has("anchorType")||drawingAttrs.getString("anchorType").startsWith("twoCell")) {
						if(drawingAttrs.has("endColOffset")) {
							
						}
						if(drawingAttrs.has("endRowOffset")) {
							
						}
					}
*/					
				}
				if(width!=null) {
					element.setAttributeNS(SvgWidthAttribute.ATTRIBUTE_NAME.getUri(), SvgWidthAttribute.ATTRIBUTE_NAME.getQName(), Double.toString(width/100) + "mm");
				}
				if(height!=null) {
					element.setAttributeNS(SvgHeightAttribute.ATTRIBUTE_NAME.getUri(), SvgHeightAttribute.ATTRIBUTE_NAME.getQName(), Double.toString(height/100) + "mm");
				}
				if(width!=null||height!=null) {
					element.removeAttributeNS(TableEndCellAddressAttribute.ATTRIBUTE_NAME.getUri(), TableEndCellAddressAttribute.ATTRIBUTE_NAME.getLocalName());
					element.removeAttributeNS(TableEndXAttribute.ATTRIBUTE_NAME.getUri(), TableEndXAttribute.ATTRIBUTE_NAME.getLocalName());
					element.removeAttributeNS(TableEndYAttribute.ATTRIBUTE_NAME.getUri(), TableEndYAttribute.ATTRIBUTE_NAME.getLocalName());
				}
			}
		}
	}
}
