/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.List;

import org.apache.xerces.dom.ElementNSImpl;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.ElementNS;
import com.openexchange.office.odf.OdfOperationDoc;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.odf.UnknownContentHandler;
import com.openexchange.office.ods.dom.SmlUtils.CellRef;

public class CellHandler extends SaxContextHandler {

	final private Sheet sheet;
	final private Row row;
	final private Cell cell;

	private String value;
	private String officeValueType;
	private String calcExtValueType;
	private String formula;
	private String currency;
	private String boolValue;
	private String dateValue;
	private String timeValue;
	private String stringValue;
	private boolean hasStringValueAttribute;

	public CellHandler(RowHandler parentContextHandler, Attributes attributes, Sheet sheet, Row row, Cell cell) {
		super(parentContextHandler);
		this.sheet = sheet;
		this.row = row;
		this.cell = cell;

		Integer columnsSpanned = 1;
		Integer rowsSpanned = 1;

		value = null;
		officeValueType = null;
		calcExtValueType = null;
		formula = null;
		currency = null;
		boolValue = null;
		dateValue = null;
		timeValue = null;
		hasStringValueAttribute = false;
		stringValue = null;

		for(int i=0; i<attributes.getLength(); i++) {
			final String localName = attributes.getLocalName(i);
			if(localName.equals("style-name")) {
				cell.setCellStyle(attributes.getValue(i));
			}
			else if(localName.equals("number-columns-repeated")) {
				final int repeated = Integer.valueOf(attributes.getValue(i));
				if((repeated>=1)&&(repeated<=16384)&&(repeated+cell.getColumn()<=16384)) {
					cell.setRepeated(repeated);
				}
			}
			else if(localName.equals("number-columns-spanned")) {
				columnsSpanned = Integer.valueOf(attributes.getValue(i));
			}
			else if(localName.equals("number-rows-spanned")) {
				rowsSpanned = Integer.valueOf(attributes.getValue(i));
			}
			else if(localName.equals("value-type")) {
				if(attributes.getURI(i).equals("urn:oasis:names:tc:opendocument:xmlns:office:1.0")) {
					officeValueType = attributes.getValue(i);
				}
				else if(attributes.getURI(i).equals("urn:org:documentfoundation:names:experimental:calc:xmlns:calcext:1.0")) {
					calcExtValueType = attributes.getValue(i);
				}
			}
			else if(localName.equals("value")) {
				value = attributes.getValue(i);
			}
			else if(localName.equals("formula")) {
				formula = getFormulaValue(attributes.getValue(i));
			}
			else if(localName.equals("currency")) {
				currency = attributes.getValue(i);
			}
			else if(localName.equals("boolean-value")) {
				boolValue = attributes.getValue(i);
			}
			else if(localName.equals("date-value")) {
				dateValue = attributes.getValue(i);
			}
			else if(localName.equals("time-value")) {
				timeValue = attributes.getValue(i);
			}
			else if(localName.equals("string-value")) {
				hasStringValueAttribute = true;
				stringValue = attributes.getValue(i);
				if(stringValue.isEmpty()) {
					hasStringValueAttribute = false;
					stringValue = null;
				}
			}
			else if(localName.equals("content-validation-name")) {
				cell.getCellAttributesEnhanced(true).setContentValidationName(attributes.getValue(i));
			}
			else if(localName.equals("number-matrix-rows-spanned")) {
				cell.getCellAttributesEnhanced(true).setNumberMatrixRowsSpanned(attributes.getValue(i));
			}
			else if(localName.equals("number-matrix-columns-spanned")) {
				cell.getCellAttributesEnhanced(true).setNumberMatrixColumnsSpanned(attributes.getValue(i));
			}
			else if(localName.equals("protect")) {
				cell.getCellAttributesEnhanced(true).setProtect(attributes.getValue(i));
			}
			else if(localName.equals("protected")) {
				cell.getCellAttributesEnhanced(true).setProtected(attributes.getValue(i));
			}
/* TODO store unspecified attributes...
			else {

			}
*/
			// inserting a new mergeCells area
			if(columnsSpanned>1||rowsSpanned>1) {
				final List<MergeCell> mergeCells = parentContextHandler.getSheet().getMergeCells();
				mergeCells.add(new MergeCell(new CellRef(cell.getColumn(), parentContextHandler.getRow().getRow()), columnsSpanned, rowsSpanned));
			}
		}
		// set the maximum allowed number of columns to 16384 if a column greater than 1024 is used
		if(cell.getColumn()+cell.getRepeated()>1024) {
			((Content)parentContextHandler.getFileDom()).setMaxColumnCount(16384);
		}
	}

	public String getStringValue() {
		return stringValue;
	}

	public void appendStringValue(String value) {
		if(!hasStringValueAttribute&&value!=null) {
			if(stringValue!=null) {
				stringValue = stringValue + value;
			}
			else {
				stringValue = value;
			}
		}
	}

	@Override
    public SaxContextHandler startElement(Attributes attributes, String uri, String localName, String qName)
    	throws SAXException {

		if(OdfOperationDoc.isLowMemoryAbort()) {
			throw new RuntimeException();
		}
		if(qName.equals("text:p")) {
			if(!hasStringValueAttribute && stringValue!=null) {
				stringValue = stringValue + "\r\n";
			}
			return new CellPHandler(this, sheet, row, cell);
		}
		String type = null;
		switch(qName) {
			case "draw:a": {
				return this;	// TODO: hyperlinks including a shape... we will skip this element...
			}
			case "draw:caption": {
				type = "undefined";
				break;
			}
			case "draw:circle": {
				type = "undefined";
				break;
			}
			case "draw:connector": {
				type = "undefined";
				break;
			}
			case "draw:control": {
				type = "undefined";
				break;
			}
			case "draw:custom-shape": {
				type = "undefined";
				break;
			}
			case "draw:ellipse": {
				type = "undefined";
				break;
			}
			case "draw:frame": {
				final ElementNSImpl element = new ElementNS(getFileDom(), attributes, uri, qName);
				final Frame frame = new Frame(new DrawingAnchor(cell.getColumn(), row.getRow()), element);
				sheet.getDrawings().addDrawing(frame);
				return new FrameContentHandler(this, frame);
			}
			case "draw:g": {
				type = "undefined";
				break;
			}
			case "draw:line": {
				type = "undefined";
				break;
			}
			case "draw:measure": {
				type = "undefined";
				break;
			}
			case "draw:page-thumbnail": {
				type = "undefined";
				break;
			}
			case "draw:path": {
				type = "undefined";
				break;
			}
			case "draw:polygon": {
				type = "undefined";
				break;
			}
			case "draw:polyline": {
				type = "undefined";
				break;
			}
			case "draw:rect": {
				type = "undefined";
				break;
			}
			case "draw:regular-polygon": {
				type = "undefined";
				break;
			}
		}
		if(type!=null) {
			// we found a drawing, this is not inserted as cellChild instead it
			// is inserted into the Drawings list at the sheet, each shape that
			// is child of a cell is having a DrawingAnchor
			final ElementNSImpl element = new ElementNS(getFileDom(), attributes, uri, qName);
			final Drawing drawing = new Drawing(new DrawingAnchor(cell.getColumn(), row.getRow()));
			drawing.setElement(element);
			sheet.getDrawings().addDrawing(drawing);
			return new UnknownContentHandler(this, element);
		}
		// using a new Handler that adds each element (beside p elements) as child to the cellElement, this is done only for complex
		// cells as most cells are only having one p element.
		return new CellChildHandler(this, cell.createNode(getFileDom()), new ElementNS(getFileDom(), attributes, uri, qName));
	}

    @Override
    public void endContext(String qName, String characters) {
    	if(qName.equals("table:table-cell")||qName.equals("table:covered-table-cell")) {
			Object content = null;
			if(calcExtValueType!=null&&calcExtValueType.equals("error")) {
				content = new Cell.ErrorCode(stringValue);
			}
			else if(officeValueType!=null&&!officeValueType.isEmpty()) {
	        	if(value!=null&&!value.isEmpty()) {
	        		try {
	        			content = Double.parseDouble(value);
	        		}
	        		catch(NumberFormatException e) {
	        			
	        		}
	    		}
	            if (officeValueType.equals("boolean")) {
	                //office:boolean-value
	                if(boolValue!=null&&!boolValue.isEmpty()) {
	                	switch(boolValue.toLowerCase()) {
	                		case "1" :
	                		case "true" :
	                		case "yes" :
	                			content = Boolean.valueOf(true);
	                		break;
	                		case "0" :
	                		case "false" :
	                		case "no" :
	                			content = Boolean.valueOf(false);
	                		break;
	                	}
	                }
	            }
	            else if (officeValueType.equals("date")) {
	                if(dateValue!=null&&!dateValue.isEmpty() ) {
	                    content = MapHelper.dateToDouble(dateValue);
	                }
	                else if(content==null) {
	                	content = tryGetOtherValues();
	                }
	            }
	            else if (officeValueType.equals("float")&&content==null) {
	            	content = tryGetOtherValues();
	            }
	            else if (officeValueType.equals("currency")&&content==null) {
	            	content = tryGetOtherValues();
	            }
	            else if (officeValueType.equals("percentage")&&content==null) {
	            	content = tryGetOtherValues();
	            }
	            else if (officeValueType.equals("string")) {
	            	content = stringValue;
	            }
	            else if (officeValueType.equals("time")) {
	                //office:time-value
	                if(timeValue!=null&&!timeValue.isEmpty()) {
	                    content = MapHelper.timeToDouble(timeValue);
	                }
	                else if(content==null) {
	                	content = tryGetOtherValues();
	                }
	            }
	            // do not allow empty strings.
	            if (content instanceof String && ((String)content).isEmpty()) {
	                content = null;
	            }
	        }
	        else if (value!=null&&!value.isEmpty()) {
        		try {
        			content = Double.parseDouble(value);
        		}
        		catch(NumberFormatException e) {
        			
        		}
	        }
			if (formula!=null&&!formula.isEmpty()) {
				cell.setCellFormula(getFormulaValue(formula));
			}
	        if(content==null&&!characters.isEmpty()) {
	        	content = characters;
	        }
	        cell.setCellContent(content);
    	}
	}

	// sometimes a officeValueType "float", "currency" or "percentage" is used without corresponding office-value 
	// then we have to check if we can other values such as date-value 
    private Object tryGetOtherValues() {
    	if(dateValue!=null&&!dateValue.isEmpty()) {
            return MapHelper.dateToDouble(dateValue);
    	}
    	else if(timeValue!=null&&!timeValue.isEmpty()) {
            return MapHelper.timeToDouble(timeValue);
    	}
    	else if(boolValue!=null&&!boolValue.isEmpty()) {
            try {
            	return Double.parseDouble(boolValue);
            } catch(NumberFormatException e) {

            }
        }
    	return null;
    }

    public String getFormulaValue(String formula) {

		if (formula.startsWith("of:=")) {
            return formula.substring(4);
        }
        else if (formula.startsWith("oooc:=")) {
            return formula.substring(6);
        }
/*        else if (formula.startsWith("msoxl:")) {
        	return formula.substring(6);
        }
*/
        return formula;
    }
}
